/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.wizards;

import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowActionDelegate;
import org.eclipse.ui.IWorkbenchWizard;
import org.eclipse.ui.actions.NewProjectAction;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.ExceptionHandler;

public abstract class AbstractOpenWizardAction extends Action implements IWorkbenchWindowActionDelegate
{
    private final boolean fAcceptEmptySelection;
    private boolean fNoChecking;

    /**
     * Creates a AbstractOpenWizardAction with no restrictions on types, and
     * does allow an empty selection.
     */
    protected AbstractOpenWizardAction() {
        fAcceptEmptySelection = true;
        fNoChecking = true;
    }

    protected IWorkbench getWorkbench()
    {
        return CurlPlugin.getDefault().getWorkbench();
    }

    private boolean isEnabled(
            IStructuredSelection selection)
    {
        for (Object obj : selection.toList()) {
            if (!shouldAcceptElement(obj)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Can be overridden to add more checks. obj is guaranteed to be instance of
     * one of the accepted types
     */
    protected boolean shouldAcceptElement(
            Object obj)
    {
        return true;
    }

    /**
     * Creates the specific wizard. (to be implemented by a subclass)
     */
    abstract protected Wizard createWizard() throws CoreException;

    protected IStructuredSelection getCurrentSelection()
    {
        IWorkbenchWindow window = CurlPlugin.getActiveWorkbenchWindow();
        if (window != null) {
            ISelection selection = window.getSelectionService().getSelection();
            if (selection instanceof IStructuredSelection) {
                return (IStructuredSelection)selection;
            }
        }
        return null;
    }

    /**
     * The user has invoked this action.
     */
    @Override
    public void run()
    {
        if (!fNoChecking && !canActionBeAdded()) {
            return;
        }
        if (!checkWorkspaceNotEmpty()) {
            return;
        }
        Shell shell = CoreUtil.getActiveWorkbenchShell();
        try {
            Wizard wizard = createWizard();
            if (wizard instanceof IWorkbenchWizard) {
                ((IWorkbenchWizard)wizard).init(getWorkbench(), getCurrentSelection());
            }
            WizardDialog dialog = new WizardDialog(shell, wizard);
            // PixelConverter converter= new
            // PixelConverter(CurlPlugin.getActiveWorkbenchShell());
            // dialog.setMinimumPageSize(converter.convertWidthInCharsToPixels(70),
            // converter.convertHeightInCharsToPixels(20));
            dialog.create();
            int res = dialog.open();
            notifyResult(res == Window.OK);
        } catch (CoreException e) {
            String title = WizardMessages.AbstractOpenWizardAction_createerror_title;
            String message = WizardMessages.AbstractOpenWizardAction_createerror_message;
            ExceptionHandler.handle(e, shell, title, message);
        }
    }

    /**
     * Tests if the action can be run on the current selection.
     */
    public boolean canActionBeAdded()
    {
        IStructuredSelection selection = getCurrentSelection();
        if (selection == null || selection.isEmpty()) {
            return fAcceptEmptySelection;
        }
        return isEnabled(selection);
    }

    /*
     * @see IActionDelegate#run(IAction)
     */
    public void run(
            IAction action)
    {
        run();
    }

    /*
     * s
     *
     * @see IWorkbenchWindowActionDelegate#dispose()
     */
    public void dispose()
    {
    }

    /*
     * @see IWorkbenchWindowActionDelegate#init(IWorkbenchWindow)
     */
    public void init(
            IWorkbenchWindow window)
    {
    }

    /*
     * @see IActionDelegate#selectionChanged(IAction, ISelection)
     */
    public void selectionChanged(
            IAction action,
            ISelection selection)
    {
        // selection taken from selectionprovider
    }

    protected boolean checkWorkspaceNotEmpty()
    {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        if (workspace.getRoot().getProjects().length == 0) {
            Shell shell = CoreUtil.getActiveWorkbenchShell();
            String title = WizardMessages.AbstractOpenWizardAction_noproject_title;
            String message = WizardMessages.AbstractOpenWizardAction_noproject_message;
            if (MessageDialog.openQuestion(shell, title, message)) {
                IWorkbenchWindow window = CurlPlugin.getActiveWorkbenchWindow();
                (new NewProjectAction(window)).run();
                return workspace.getRoot().getProjects().length != 0;
            }
            return false;
        }
        return true;
    }

    protected static boolean isOnBuildPath(
            Object obj)
    {
        /*
         * if (obj instanceof IJavaElement) { IJavaElement elem=
         * (IJavaElement)obj; return elem.getJavaProject().isOnClasspath(elem); }
         */
        return true;
    }
    /*
     * protected static boolean isInArchive(Object obj) { if (obj instanceof
     * IJavaElement) { IPackageFragmentRoot root=
     * JavaModelUtil.getPackageFragmentRoot((IJavaElement)obj); return (root !=
     * null) && root.isArchive(); } return false; }
     */
}
