/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.util;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.debug.CurlExceptionBreakpointDescriptor;
import com.curl.eclipse.debug.CurlLineBreakpointDescriptor;
import com.curl.eclipse.remote.WorkbenchProxy;
import com.curl.eclipse.remote.WorkbenchWindowProxy;
import com.curl.eclipse.remote.WorkbenchProxy.ToolEntry;
import com.curl.eclipse.remote.WorkbenchWindowProxy.ProjectTargets;

/**
 * @author pshapiro
 *
 * This is used to track Workbench state that Curl is interested in, and contains
 * commands that are handled by the curl "workbench" objects.
 */
public class WorkbenchOperations
{
    private final WorkbenchProxy fworkbenchProxy;
    private final List<MenuDescriptor> fToolMenuDescriptors = new ArrayList<MenuDescriptor>();
    private IResourceChangeListener fWorkspaceResourceListener;
    private final WindowListener fWindowListener;

    /**
     * tracks CurlEditor activation/deactivation to keep CurlOutlineView in sync
     */
    private static final class WindowListener implements IWindowListener
    {
        private final PartListenerForCurlEditor fPartListener = new PartListenerForCurlEditor();
        private final Map<IWorkbenchWindow, WorkbenchWindowProxy> fMapWorkbenchWindowToWorkbenchWindowProxy;
        private final WorkbenchProxy fWorkbenchProxy;

        WindowListener(WorkbenchProxy workbenchProxy)
        {
            fWorkbenchProxy = workbenchProxy;
            fMapWorkbenchWindowToWorkbenchWindowProxy = new HashMap<IWorkbenchWindow, WorkbenchWindowProxy>();
            /*
             * register listener to all pre-existing windows now because during
             * startup sequence the window-event are fired "too late" for us.
             * I mean the views in the windows are created before the window open events fired
             * thus we didn't get a chance to register them.
             */
            IWorkbenchWindow[] workbenchWindows = PlatformUI.getWorkbench().getWorkbenchWindows();
            for (IWorkbenchWindow workbenchWindow : workbenchWindows) {
                IWorkbenchPage[] pages = workbenchWindow.getPages();
                for (IWorkbenchPage workbenchPage : pages) {
                    workbenchPage.addPartListener(fPartListener);
                }
            }
        }
        
        private WorkbenchWindowProxy getAndCreateIfNeeded(
                IWorkbenchWindow window)
        {
            if (! fMapWorkbenchWindowToWorkbenchWindowProxy.containsKey(window)) {
                createProxyFor(window);
            }
            WorkbenchWindowProxy workbenchWindowProxy = fMapWorkbenchWindowToWorkbenchWindowProxy.get(window);
            return workbenchWindowProxy;
        }


        public void windowActivated(
                IWorkbenchWindow window)
        {
            getAndCreateIfNeeded(window).windowActivated();
        }

        public void windowClosed(
                IWorkbenchWindow window)
        {
            // With eclipse > 3.0 only 1 page per window
            IWorkbenchPage activePage = window.getActivePage();
            if (activePage != null) {
                activePage.removePartListener(fPartListener);
            }
            WorkbenchWindowProxy removed = fMapWorkbenchWindowToWorkbenchWindowProxy.remove(window);
            if (removed != null) {
                removed.destroy();
            }
            // else, this window never got an opportunity to contain a curl view.
        }

        public void windowDeactivated(
                IWorkbenchWindow window)
        {
            getAndCreateIfNeeded(window).windowDeactivated();
        }

        public void windowOpened(
                IWorkbenchWindow window)
        {
            // With eclipse > 3.0 only 1 page per window
            window.getActivePage().addPartListener(fPartListener);
        }

        private void createProxyFor(
                IWorkbenchWindow window)
        {
            // since we register windows at startup without waiting for the window-open event
            // we must not initialize twice
            if (! fMapWorkbenchWindowToWorkbenchWindowProxy.containsKey(window)) {
                fMapWorkbenchWindowToWorkbenchWindowProxy.put(window, new WorkbenchWindowProxy(fWorkbenchProxy, window));
            }
            // else the window was restored during the plug-in startup sequence, and
            // was explicitly registered in the constructor of this class.
        }

        public IEditorInput retrieveHistoryEditorInput(
                String url)
        {
            return fPartListener.retrieveHistoryEditorInput(url);
        }

        public String storeHistoryEditorInput(
                IEditorInput editorInput)
        {
            return fPartListener.storeHistoryEditorInput(editorInput);
        }

        public void setCurrentTarget(
                IWorkbenchWindow window,
                String currentTargetName)
        {
            WorkbenchWindowProxy workbenchWindowProxy = getAndCreateIfNeeded(window);
            workbenchWindowProxy.setCurrentTarget(currentTargetName);
        }
    }
    
    public WorkbenchProxy getWorkbenchProxy()
    {
        return fworkbenchProxy;
    }

    /**
     * A workspace resource change listener for notifying the mediator of project related events.
     */
    private class WorkspaceResourceChangeListener implements IResourceChangeListener
    {
        /**
         * An Eclipse resource has changed.  If it is a Curl
         * project then notify the mediator to take appropriate action.
         */
        public void resourceChanged(
                IResourceChangeEvent event)
        {
            int type = event.getType();
            IResource resource = event.getResource();
            
            if (type == IResourceChangeEvent.PRE_CLOSE || type == IResourceChangeEvent.PRE_DELETE) {
                if (!(resource instanceof IProject))
                    return;                
                IProject project = (IProject)resource;
                URL curlProjectURL = CoreUtil.getCurlProjectURL(project, false);
                if (curlProjectURL != null && project.isOpen()) {
                    if (type == IResourceChangeEvent.PRE_CLOSE)
                        fworkbenchProxy.closeProject(curlProjectURL.toString());
                    else
                        fworkbenchProxy.closeToDelete(curlProjectURL.toString());
                }
            } else {
                IResourceDeltaVisitor visitor = new IResourceDeltaVisitor() {
                    public boolean visit(
                            IResourceDelta delta)
                    {
                        // Make sure the mediator knows about projects that have been opened.  Also,
                        // if a project has been removed, then close it (why doesn't Eclipse do this for us?!)
                        IResource resource = delta.getResource();
                        if (resource.getType() == IResource.PROJECT) {
                            IProject project = (IProject)resource;
                            int kind = delta.getKind();
                            if (project.isOpen() && CoreUtil.getCurlProjectURL(project, false) != null) {
                                int flags = delta.getFlags();
                                if (((kind == IResourceDelta.CHANGED && (flags & IResourceDelta.OPEN) != 0) ||
                                     kind == IResourceDelta.ADDED)) {
                                    // This project was closed and now opened, or it has been added.
                                    openProjectAddPersistedBreakpoints(project);
                                }
                            } else if (kind == IResourceDelta.REMOVED) {
                                CoreUtil.logInfo("Removed");  //$NON-NLS-1$
                            }
                        }
                        return true;
                    }
                };
                try {
                    event.getDelta().accept(visitor);
                } catch (CoreException e) {
                    CoreUtil.logError("Error visiting changed projects.", e);  //$NON-NLS-1$
                }
            }
        }
    }

    public void openProjectAddPersistedBreakpoints(
            IProject project)
    {
        URL projectDirectoryURL = null;
        try {
            projectDirectoryURL = project.getLocationURI().toURL();
        } catch (Exception e) {
            CoreUtil.logError("Could not determine the location of project to open", e);  //$NON-NLS-1$
            return;
        }
        fworkbenchProxy.openProject(projectDirectoryURL, project.getName());
        sendPersistedLineBreakpoints(project);
    }

    public WorkbenchOperations()
    {
        fworkbenchProxy = new WorkbenchProxy(this);

        initOpenProjects();
        fWorkspaceResourceListener = new WorkspaceResourceChangeListener();
        ResourcesPlugin.getWorkspace().addResourceChangeListener(
                fWorkspaceResourceListener,
                IResourceChangeEvent.POST_CHANGE | IResourceChangeEvent.PRE_CLOSE | IResourceChangeEvent.PRE_DELETE);
        sendPersistedExceptionBreakpoints();

        /*
         * track part activation/deactivation across all possible windows
         */
        fWindowListener = new WindowListener(fworkbenchProxy);
        PlatformUI.getWorkbench().addWindowListener(fWindowListener);

    }

    public void shutdown()
    {
        PlatformUI.getWorkbench().removeWindowListener(fWindowListener);
        
        if (fWorkspaceResourceListener != null) {
            ResourcesPlugin.getWorkspace().removeResourceChangeListener(fWorkspaceResourceListener);
            fWorkspaceResourceListener = null;
        }
        fworkbenchProxy.shutdown();
    }

    private void initOpenProjects()
    {
        IProject[] projects = ResourcesPlugin.getWorkspace().getRoot().getProjects();
        for (IProject project : projects) {
            try {
                if (project.isOpen() && project.hasNature(CurlUIIDs.ID_CURL_NATURE))
                    openProjectAddPersistedBreakpoints(project);
            } catch (Exception e) {
                CoreUtil.logError("Project cannot be found", e); //$NON-NLS-1$
            }
        }
    }

    /**
     * Send any registered line breakpoints for resources in this project to the mediator.
     */
    synchronized private void sendPersistedLineBreakpoints(
            IProject project)
    {
        IMarker[] curlDebugMarkers = {};
        try {
            curlDebugMarkers = project.findMarkers(
                    CurlUIIDs.ID_CURL_LINE_DEBUG_MARKER,
                    true,
                    IResource.DEPTH_INFINITE);
        } catch (CoreException e) {
            CoreUtil.logError("Error finding line debug markers", e); //$NON-NLS-1$
        }
        for (IMarker marker : curlDebugMarkers) {
            fworkbenchProxy.addBreakpoint(new CurlLineBreakpointDescriptor(marker));
        }
    }

    /**
     * Send any Curl exception breakpoints registered with the workspace to the mediator.
     */
    synchronized private void sendPersistedExceptionBreakpoints()
    {
        IMarker[] curlDebugMarkers = {};
        try {
            curlDebugMarkers = ResourcesPlugin.getWorkspace().getRoot().findMarkers(
                    CurlUIIDs.ID_CURL_EXCEPTION_DEBUG_MARKER,
                    true,
                    IResource.DEPTH_ZERO);
        } catch (CoreException e) {
            CoreUtil.logError("Error finding exceptioin debug markers", e); //$NON-NLS-1$
        }
        for (IMarker marker : curlDebugMarkers) {
            fworkbenchProxy.addBreakpoint(new CurlExceptionBreakpointDescriptor(marker));
        }
    }

    public void setCurrentTarget(
            IWorkbenchWindow fworkbenchWindow, String currentTargetName)
    {
        fWindowListener.setCurrentTarget(fworkbenchWindow, currentTargetName);
    }

    public static class MenuDescriptor
    {
        public final String fId;
        public final String fName;

        public MenuDescriptor(
                String id,
                String name)
        {
            fId = id;
            fName = name;
        }
    }

    /**
     * Process list of tool descriptions from mediator,
     * and create menu items in "Tools" menu (id "com.curl.eclipse.CurlToolsMenu")
     */
    public void setTools(List<ToolEntry> toolsSet)
    {
        // cleanup existing items under Tools
        fToolMenuDescriptors.clear();
        for (ToolEntry toolEntry : toolsSet) {
            fToolMenuDescriptors.add(new MenuDescriptor(toolEntry.fCurlEditorID, toolEntry.fName));
        }
    }
    public List<MenuDescriptor> getToolMenuDescriptors()
    {
        return fToolMenuDescriptors;
    }

    public void launchTool(
            String id)
    {
        launchTool(id, null);
    }

    public void launchTool(
            String id,
            URL fileURL)
    {
        fworkbenchProxy.launchToolForId(id, fileURL);
    }

    /**
     * @see #retrieveHistoryEditorInput(String)
     * @param docURL
     * @param editorInput
     */
    public String storeHistoryEditorInput(
            IEditorInput editorInput)
    {
        return fWindowListener.storeHistoryEditorInput(editorInput);
    }
    

    /**
     * @see #storeHistoryEditorInput(String, IEditorInput)
     * @param url
     * @return null if url isn't associates with a IEditorInput
     */
    public IEditorInput retrieveHistoryEditorInput(
            String url)
    {
        return fWindowListener.retrieveHistoryEditorInput(url);
    }

    public WorkbenchWindowProxy getWorkbenchWindowProxyFor(
            IWorkbenchWindow workbenchWindow)
    {
        return fWindowListener.getAndCreateIfNeeded(workbenchWindow);
    }

    public void savedURLResponse(
            String savedURLString)
    {
        try {
            IContainer[] containers = ResourcesPlugin.getWorkspace().getRoot().findContainersForLocationURI(new URI(savedURLString));
            for (IContainer container : containers) {
                container.refreshLocal(IResource.DEPTH_INFINITE, null);
            }
        } catch (URISyntaxException e) {
            CoreUtil.logError("Invalid url=" + savedURLString, e); //$NON-NLS-1$
        } catch (CoreException e) {
            CoreUtil.logError("Could not resynchronize eclipse cache with url=" + savedURLString, e); //$NON-NLS-1$
        }
    }

    public ProjectTargets getProjectTargets(
            IWorkbenchWindow window)
    {
        return getWorkbenchWindowProxyFor(window).getProjectTargets();
    }

    public void raiseActiveWindow()
    {
        CurlPlugin.getDefault().getWorkbench().getDisplay().asyncExec(new Runnable() {
            public void run()
            {
                Shell shell = CoreUtil.getActiveWorkbenchShell();
                if (shell != null) {
                    shell.forceActive();
                }
            }
        });
    }
}
