/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.util;

import org.eclipse.core.runtime.IStatus;

/**
 * Represents outcome of an operation.  Used to report problems in dialogs with problem message area.
 */
public class StatusInfo implements IStatus
{
    public static final IStatus OK_STATUS = new StatusInfo();
    private String fStatusMessage;
    private int fSeverity;

    public StatusInfo()
    {
        this(OK, null);
    }

    public StatusInfo(
            int severity,
            String message)
    {
        fStatusMessage = message;
        fSeverity = severity;
    }

    public boolean isOK()
    {
        return fSeverity == IStatus.OK;
    }

    public boolean isWarning()
    {
        return fSeverity == IStatus.WARNING;
    }

    public boolean isInfo()
    {
        return fSeverity == IStatus.INFO;
    }

    public boolean isError()
    {
        return fSeverity == IStatus.ERROR;
    }

    public String getMessage()
    {
        return fStatusMessage;
    }

    public void setError(
            String errorMessage)
    {
        fStatusMessage = errorMessage;
        fSeverity = IStatus.ERROR;
    }

    public void setWarning(
            String warningMessage)
    {
        fStatusMessage = warningMessage;
        fSeverity = IStatus.WARNING;
    }

    public void setInfo(
            String infoMessage)
    {
        fStatusMessage = infoMessage;
        fSeverity = IStatus.INFO;
    }

    public void setOK()
    {
        fStatusMessage = null;
        fSeverity = IStatus.OK;
    }

    public boolean matches(
            int severityMask)
    {
        return (fSeverity & severityMask) != 0;
    }

    public boolean isMultiStatus()
    {
        return false;
    }

    public int getSeverity()
    {
        return fSeverity;
    }

    public String getPlugin()
    {
        return CurlUIIDs.ID_PLUGIN;
    }

    public Throwable getException()
    {
        return null;
    }

    public int getCode()
    {
        return fSeverity;
    }

    public IStatus[] getChildren()
    {
        return new IStatus[0];
    }

    public String toString()
    {
        StringBuffer buf = new StringBuffer();
        buf.append("StatusInfo "); //$NON-NLS-1$
        if (fSeverity == OK) {
            buf.append("OK"); //$NON-NLS-1$
        } else if (fSeverity == ERROR) {
            buf.append("ERROR"); //$NON-NLS-1$
        } else if (fSeverity == WARNING) {
            buf.append("WARNING"); //$NON-NLS-1$
        } else if (fSeverity == INFO) {
            buf.append("INFO"); //$NON-NLS-1$
        } else {
            buf.append("severity="); //$NON-NLS-1$
            buf.append(fSeverity);
        }
        buf.append(": "); //$NON-NLS-1$
        buf.append(fStatusMessage);
        return buf.toString();
    }
}
