/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.util;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.team.core.history.IFileRevision;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IWorkbenchPartReference;

import com.curl.eclipse.editors.CurlEditor;

/**
 * 
 * Listen for editor part events of interest
 * 
 */
class PartListenerForCurlEditor implements IPartListener2
{

    private final Map<String, List<IEditorInput>> fMapOfEditorInput = new Hashtable<String, List<IEditorInput>>();

    

    public void partHidden(
            IWorkbenchPartReference partRef)
    {
        if (partRef.getId().equals(CurlUIIDs.ID_CURL_EDITOR)) {
            CurlEditor curlEditor = (CurlEditor)partRef.getPart(false);
            curlEditor.editorHidden();
        }
    }

    public void partBroughtToTop(
            IWorkbenchPartReference partRef)
    {
        if (partRef.getId().equals(CurlUIIDs.ID_CURL_EDITOR)) {
            CurlEditor curlEditor = (CurlEditor)partRef.getPart(false);
            curlEditor.editorBroughtToTop();
        }
    }

    public void partClosed(
            IWorkbenchPartReference partRef)
    {
        if (partRef.getId().equals(CurlUIIDs.ID_CURL_EDITOR)) {
            CurlEditor curlEditor = (CurlEditor)partRef.getPart(false);
            IEditorInput editorInput = curlEditor.getEditorInput();
            String url = createHistoryURLFrom(editorInput);
            if (url == null) {
                // edited element wasn't a history revision
                return;
            }
            List<IEditorInput> list = fMapOfEditorInput.get(url);
            list.remove(editorInput);
            if (list.isEmpty()) {
                fMapOfEditorInput.remove(url);
            }
        }
    }
    
    static public String createHistoryURLFrom(
            IEditorInput editorInput)
    {
        String docURL;
        {
            IFileRevision  fileRevision = (IFileRevision)editorInput.getAdapter(IFileRevision.class);
            if (fileRevision == null) {
                return null;
            }
            // "history://" is a dummy protocol with no server, no client should try to resolve it
            // a nice side effect of this dummy protocol is that the curl editor will mark this buffer
            // as read-only.
            // all we really need here is a unique string for this editorInput
            docURL = "eclipsehistory:/" + fileRevision.getURI().getPath(); //$NON-NLS-1$
            String query;
            if (fileRevision.getContentIdentifier() != null
                    && fileRevision.getContentIdentifier().length() > 0){
                query = "contentid=" + fileRevision.getContentIdentifier(); //$NON-NLS-1$
            }
            else if (fileRevision.getTimestamp() != -1) {
                query = "timestamp=" + Long.toString(fileRevision.getTimestamp()); //$NON-NLS-1$
            }
            else {
                // here we're desparate... but at least the url is unique
                query = "dummy=" + Long.toString(System.currentTimeMillis()); //$NON-NLS-1$
            }
            docURL = docURL + "?" + query; //$NON-NLS-1$
        }
        return docURL;
    }

    public void partDeactivated(
            IWorkbenchPartReference partRef)
    {
    }

    public void partActivated(
            IWorkbenchPartReference partRef)
    {
        // testing shows that we don't need to handle this call
//        if (partRef.getId().equals(CurlUIIDs.ID_CURL_EDITOR)) {
//            CurlEditor curlEditor = (CurlEditor)partRef.getPart(false);
//            curlEditor.editorBroughtToTop();
//            System.out.println("partActivated:" + curlEditor);
//        }
    }

    public void partOpened(
            IWorkbenchPartReference partRef)
    {
    }

    public void partVisible(
            IWorkbenchPartReference partRef)
    {
    }

    public void partInputChanged(
            IWorkbenchPartReference partRef)
    {
    }

    public IEditorInput retrieveHistoryEditorInput(
            String url)
    {
        List<IEditorInput> list = fMapOfEditorInput.get(url);
        return list != null ? list.get(0) : null;
    }

    public String storeHistoryEditorInput(
            IEditorInput editorInput)
    {
        String url = createHistoryURLFrom(editorInput);
        List<IEditorInput> list = fMapOfEditorInput.get(url);
        if (list == null) {
            list = new ArrayList<IEditorInput>();
            fMapOfEditorInput.put(url, list);
        }
        list.add(editorInput);
        return url;
    }
}