/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.util;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.MenuListener;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorRegistry;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.IURIEditorInput;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.editors.text.EditorsUI;
import org.eclipse.ui.ide.FileStoreEditorInput;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.texteditor.IEditorStatusLine;
import org.osgi.framework.Bundle;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.core.CurlElement;
import com.curl.eclipse.debug.CurlLineBreakpoint;
import com.curl.eclipse.debug.CurlLineBreakpointDescriptor;
import com.curl.eclipse.debug.RemoteSourceFileStore;
import com.curl.eclipse.editors.CurlEditor;

public class CoreUtil
{
//    public static IFile openEditorOnUrl(
//            String urlString,
//            int offsetOrLine,
//            int length)
//    {
//        return openEditorOnUrl(urlString, true, offsetOrLine, length);
//    }

    private static final Map<String, String> CURL_TOOL_ID_TO_ECLIPSE_EDITOR_ID =
        new HashMap<String, String>();
    static {
        CURL_TOOL_ID_TO_ECLIPSE_EDITOR_ID.put(CurlUIIDs.TOOL_ID_INTEGRATED_TEXT_EDITOR, CurlUIIDs.ID_CURL_EDITOR);
        CURL_TOOL_ID_TO_ECLIPSE_EDITOR_ID.put(CurlUIIDs.TOOL_ID_VISUAL_LAYOUT_EDITOR, CurlUIIDs.ID_CURL_VLE_EDITOR);
    }

    public static void openEditorOn(
            CurlElement element, 
            boolean activate)
    {
        IFile file = element.getFile();
        CoreUtil.openEditorOnUrl(
                file.getLocationURI().toString(),
                element.getRow(),
                element.getCol(),
                element.getSymbolName().length(),
                activate,
                CurlUIIDs.TOOL_ID_INTEGRATED_TEXT_EDITOR);
    }
    
    public static IFile openEditorOnUrl(
            String urlString)
    {
        return openEditorOnUrl(urlString, 0, 0, 0, true, null);
    }

    public static IFile openEditorOnUrl(
            String urlString,
            int row,
            int col,
            int length,
            boolean activate,
            String toolID)
    {
        IFile fileToRet = null;
        try {
            IFile[] mappedFiles = ResourcesPlugin.getWorkspace().getRoot().findFilesForLocationURI(new URI(urlString));
            if (mappedFiles != null && mappedFiles.length != 0 && mappedFiles[0] != null) {
                fileToRet = mappedFiles[0];
                IProject project = fileToRet.getProject();
                if (project != null && project.isAccessible()) {
                    fileToRet.refreshLocal(IResource.DEPTH_ZERO, null);
                    IEditorPart part;
                    // TODO: following could probably be refactored with the "eclipsehistory://" case below
                    if (toolID != null && toolID.length() > 0) {
                        String eclipseEditorID = convertCurlToolIDToEclipseEditorID(toolID);
                        part = IDE.openEditor(CurlPlugin.getActivePage(), fileToRet, eclipseEditorID, activate);
                    } else {
                        part = IDE.openEditor(CurlPlugin.getActivePage(), fileToRet, activate);
                    }
                    selectAndReveal(row, col, length, part);
                    return fileToRet;
                }
            }
            else {
                // maybe the url refers to the history of a resource
                if (urlString.startsWith("eclipsehistory://")) { //$NON-NLS-1$
                    IEditorInput editorInput = CurlPlugin.getDefault().getWorkbenchOperations().retrieveHistoryEditorInput(urlString);
                    if (editorInput != null) {
                        IEditorPart part;
                        if (toolID != null && toolID.length() > 0) {
                            String eclipseEditorID = convertCurlToolIDToEclipseEditorID(toolID);
                            part = IDE.openEditor(CurlPlugin.getActivePage(), editorInput, eclipseEditorID, activate);
                        } else {
                            part = IDE.openEditor(CurlPlugin.getActivePage(), editorInput, CurlUIIDs.ID_CURL_EDITOR, activate);
                        }
                        selectAndReveal(row, col, length, part);
                    }
                    return null;
                }
            }
            // Must be a file not mapped to any open projects.
//            String eclipseEditorID;
//            if (toolID != null && toolID.length() > 0) {
//                eclipseEditorID = convertCurlToolIDToEclipseEditorID(toolID);
//            }
//            else {
//                eclipseEditorID = CurlUIIDs.ID_CURL_EDITOR;
//            }
//            URI uri = new URI(urlString);
//            IEditorPart part = IDE.openEditor(CurlPlugin.getActivePage(), uri, eclipseEditorID, true);
            IEditorPart part = openEditorOnFileSystemFile(urlString, activate);
            selectAndReveal(row, col, length, part);

            // TODO: What if the url itself is bogus. User typed a url in the
            // manifest that is bad.
            // In that case, we should pop up a message and tell the user.
        } catch (Exception e) {
            CoreUtil.logError("Error locating the url", e); //$NON-NLS-1$
        }
        return fileToRet;
    }

    private static String convertCurlToolIDToEclipseEditorID(
            String toolID)
    {
        String editorID = CURL_TOOL_ID_TO_ECLIPSE_EDITOR_ID.get(toolID);
        if (editorID == null) {
            CoreUtil.logError("Couldn't map curl id:" + toolID); //$NON-NLS-1$
        }
        return editorID;
    }

    private static void selectAndReveal(
            int row,
            int col,
            int length,
            IEditorPart part)
    {
        if (row < 0) {
            // legitimate, happens when the Manifest explorer opens the file.
            return;
        }
        if (part instanceof CurlEditor) {
            CurlEditor curlEditor = (CurlEditor)part;
            curlEditor.selectAndRevealByRowCol(row, col, length);
        }
    }

    public static IEditorPart openEditorOnFileSystemFile(
            String urlString,
            boolean activate)
    {
        try {
            IFileStore fileStore = EFS.getStore(new URI(urlString));
            if (!fileStore.fetchInfo().isDirectory() && fileStore.fetchInfo().exists()) {
                IEditorInput input = getEditorInput(fileStore);
                String editorId = getEditorId(fileStore);
                IWorkbenchPage page = CurlPlugin.getActiveWorkbenchWindow().getActivePage();
                return page.openEditor(input, editorId, activate);
            }
        } catch (Exception e) {
            CoreUtil.logError("Error locating file system url", e); //$NON-NLS-1$
        }
        return null;
    }

    /**
     * Create the Editor Input appropriate for the given <code>IFileStore</code>.
     * The result is a normal file editor input if the file exists in the
     * workspace and, if not, we create a wrapper capable of managing an
     * 'external' file using its <code>IFileStore</code>.
     * 
     * @param fileStore
     *            The file store to provide the editor input for
     * @return The editor input associated with the given file store
     */
    public static IEditorInput getEditorInput(IFileStore fileStore) {
        IFile workspaceFile = getWorkspaceFile(fileStore);
        if (workspaceFile != null)
            return new FileEditorInput(workspaceFile);
        return new FileStoreEditorInput(fileStore);
    }
    
    /**
     * Determine whether or not the <code>IFileStore</code> represents a file
     * currently in the workspace.
     * 
     * @param fileStore
     *            The <code>IFileStore</code> to test
     * @return The workspace's <code>IFile</code> if it exists or
     *         <code>null</code> if not
     */
    private static IFile getWorkspaceFile(IFileStore fileStore) {
        IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
        IFile[] files = root.findFilesForLocationURI(fileStore.toURI());
        files = filterNonExistentFiles(files);
        if (files == null || files.length == 0)
            return null;

        // for now only return the first file
        return files[0];
    }
    
    /**
     * Filter the incoming array of <code>IFile</code> elements by removing
     * any that do not currently exist in the workspace.
     * 
     * @param files
     *            The array of <code>IFile</code> elements
     * @return The filtered array
     */
    private static IFile[] filterNonExistentFiles(IFile[] files) {
        if (files == null)
            return null;

        int length = files.length;
        ArrayList<IFile> existentFiles = new ArrayList<IFile>(length);
        for (int i = 0; i < length; i++) {
            if (files[i].exists())
                existentFiles.add(files[i]);
        }
        return existentFiles.toArray(new IFile[existentFiles.size()]);
    }
    
    public static String getEditorId(
            IFileStore file)
    {
        // Always use Curl editor for RemoteSourceFileStore. Remote 
        // resources could be from http: or from https: resource and
        // may not end with .curl, .scurl, .xcurl extension. 
        if (file instanceof RemoteSourceFileStore) {
            return CurlUIIDs.ID_CURL_EDITOR;
        }
        
        IWorkbench workbench =  CurlPlugin.getActiveWorkbenchWindow().getWorkbench();
        IEditorRegistry editorRegistry = workbench.getEditorRegistry();
        IEditorDescriptor descriptor = editorRegistry.getDefaultEditor(file.getName(), getContentType(file));
        // check the OS for in-place editor (OLE on Win32)
        if (descriptor == null && editorRegistry.isSystemInPlaceEditorAvailable(file.getName()))
            descriptor = editorRegistry.findEditor(IEditorRegistry.SYSTEM_INPLACE_EDITOR_ID);
        // check the OS for external editor
        if (descriptor == null && editorRegistry.isSystemExternalEditorAvailable(file.getName()))
            descriptor = editorRegistry.findEditor(IEditorRegistry.SYSTEM_EXTERNAL_EDITOR_ID);
        if (descriptor != null)
            return descriptor.getId();
        return EditorsUI.DEFAULT_TEXT_EDITOR_ID;
    }

    private static IContentType getContentType(
            IFileStore fileStore)
    {
        if (fileStore == null)
            return null;
        InputStream stream = null;
        try {
            stream = fileStore.openInputStream(EFS.NONE, null);
            return Platform.getContentTypeManager().findContentTypeFor(stream, fileStore.getName());
        } catch (IOException e) {
            CoreUtil.logError("Could not get content type for file system file", e); //$NON-NLS-1$
            return null;
        } catch (CoreException e) {
            // Do not log FileNotFoundException (no access)
            if (!(e.getStatus().getException() instanceof FileNotFoundException))
                CoreUtil.logError("Could not get content type for file system file (core exception)", e); //$NON-NLS-1$
            return null;
        } finally {
            try {
                if (stream != null)
                    stream.close();
            } catch (IOException e) {
                CoreUtil.logError("Could not get content type for file system file (while closing)", e); //$NON-NLS-1$
            }
        }
    }

    public static String getSelectedResourceDirectory(
            IStructuredSelection selection)
    {
        IResource resource = null;
        if (selection != null && !selection.isEmpty()) {
            Object selectedElement = selection.getFirstElement();
            if (selectedElement instanceof IAdaptable) {
                IAdaptable adaptable = (IAdaptable)selectedElement;
                resource = (IResource)adaptable.getAdapter(IResource.class);
                if (resource != null && resource.getType() != IResource.ROOT) {
                    while (resource != null && resource.getType() != IResource.PROJECT
                            && resource.getType() != IResource.FOLDER) {
                        resource = resource.getParent();
                    }
                }
            }
        }
        if (resource != null)
            return resource.getLocation().toOSString();
        else
            return ""; //$NON-NLS-1$
    }

    public static URI getSelectedResourceURI(
            IWorkbenchPart part,
            ISelection selection)
    {
      URI uri = null;
      if (part instanceof IEditorPart) {
          IEditorPart editor = (IEditorPart) part;
          if (editor.getEditorInput() instanceof IPathEditorInput) {
              IPathEditorInput editorInput = (IPathEditorInput) editor.getEditorInput();
              uri = editorInput.getPath().makeAbsolute().toFile().toURI();
          } else if (editor.getEditorInput() instanceof IURIEditorInput) {
              IURIEditorInput editorInput = (IURIEditorInput) editor.getEditorInput();
              uri = editorInput.getURI();
          }
      } else {
          if (selection instanceof IStructuredSelection) {
              IStructuredSelection structuredSelection = (IStructuredSelection)selection;
              Object firstElement = structuredSelection.getFirstElement();
              if (firstElement instanceof IResource) {
                  IResource resource = (IResource)firstElement;
                  uri = resource.getLocationURI();
              } else if (firstElement instanceof IFileStore) {
                  IFileStore fileStore = (IFileStore)firstElement;
                  uri = fileStore.toURI();
                  // FIXME: this external file might in fact belong to
                  // an opened curl project, we should then set
                  // "currentProject" accordingly
              } else {
                  // nothing interresting for mediator
              }
          }
      }
      return uri;
    }

    public static CurlProject[] getCurlProjects()
    {
        IProject allProjects[] = ResourcesPlugin.getWorkspace().getRoot().getProjects();
        List<CurlProject> curlProjects = new ArrayList<CurlProject>();
        for (IProject project : allProjects) {
            if (project.isOpen() && isCurlProject(project)) {
                curlProjects.add(new CurlProject(project, null));
            }
        }
        return curlProjects.toArray(new CurlProject[curlProjects.size()]);
    }


    public static IProject getProject(
            IWorkbenchPart part)
    {
        if (part == null) {
            return null;
        }
        if (part instanceof IEditorPart) {
            IEditorPart editor = (IEditorPart)part;
            return getProject(editor.getEditorInput());
        } else if (part instanceof ISelectionProvider) {
            ISelectionProvider selectionProvider = (ISelectionProvider)part;
            return CoreUtil.getCurlProjectFromSelection(selectionProvider.getSelection());
        }
        return getCurlProjectFromSelection(part.getSite().getSelectionProvider().getSelection());
    }

    public static IProject getProject(
            IEditorInput input)
    {
        if (!(input instanceof IFileEditorInput))
            return null;
        IProject eclipseProject = ((IFileEditorInput)input).getFile().getProject();
        return eclipseProject;
    }

    public static CurlProject getCurlProject(
            ISelection selection)
    {
        IProject project = getCurlProjectFromSelection(selection);
        if (project == null) {
            return null;
        }
        return new CurlProject(project, null);
    }

    public static URL getCurlProjectURL(
            IProject project,
            boolean checkForExistance)
    {
        try {
            if (project == null || 
               (checkForExistance && !project.exists()) || 
               (!project.hasNature(CurlUIIDs.ID_CURL_NATURE))) {
                return null;
            }
        } catch (CoreException e) {
            return null;
        }

        IFile curlProjectFile = project.getFile("project.cprj"); //$NON-NLS-1$
        if (checkForExistance && !curlProjectFile.exists())
            return null;
        try {
            return curlProjectFile.getLocationURI().toURL();
        } catch (MalformedURLException e) {
            CoreUtil.logError("Getting Curl project URL failed", e); //$NON-NLS-1$
        }
        return null;
    }

    public static URL getCurlProjectURL(
            String projectName)
    {
        IProject project = CoreUtil.getWorkspaceCurlProject(projectName);
        if (project == null)
            return null;
        else
            try {
                return project.getLocationURI().toURL();
            } catch (Exception e) {
                return null;
            }
    }

    public static IProject getWorkspaceCurlProject(
            String projectName)
    {
        String name = projectName.trim();
        if (name.length() == 0)
            return null;
        IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(name);
        try {
            if (!project.hasNature(CurlUIIDs.ID_CURL_NATURE))
                return null;
            return project;
        } catch (Exception e) {
            return null;
        }
    }

    public static boolean isCurlProject(
            IProject project)
    {
        try {
            if (!project.hasNature(CurlUIIDs.ID_CURL_NATURE))
                return false;
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    /**
     * @param projectName
     * @return null if the project is not CURL enabled or the manifest file doesn't exist
     */
    public static URL getCurlProjectManifestURL(String projectName)
    {
        return getCurlProjectFileURL(projectName, "manifest.mcurl"); //$NON-NLS-1$
    }

    /**
     * @param projectName
     * @param fileName
     * @return the URL corresponding to the resource fileName for the poject, or null
     *  if the project is not a curl project,
     *  or the fileName is an absolute URL that is invalid for some reason
     *  or the filename is an URI that doesn't belong to the project
     */
    public static URL getCurlProjectFileURL(
            String projectName,
            String fileName)
    {
        IFile file = CoreUtil.getCurlProjectFile(projectName, fileName);
        URL url = null;
        try {
            if (file != null) {
                url = file.getLocationURI().toURL();
            } else {
                URI uri = URI.create(fileName);
                if (uri.isAbsolute())
                    url = uri.toURL();
            }
        } catch (MalformedURLException e) {
            // continue to return null
        }
        return url;
    }

    public static IFile getCurlProjectFile(
            String projectName,
            String fileName)
    {
        if (getCurlProjectURL(projectName) == null)
            return null;
        IFile file = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName).getFile(fileName);
        if (file.exists())
            return file;
        return null;
    }

    public static IFile getCurlProjectFile(
            IProject project,
            String fileName)
    {
        IFile file = project.getFile(fileName);
        if (file.exists())
            return file;
        return null;
    }


    public static IFile getFile(
            String fileURI)
    {
        return getFile(URI.create(fileURI));
    }

    public static IFile getFile(
            URI uri)
    {
        try {
            //TODO:  What about external files????
            IFile[] file = ResourcesPlugin.getWorkspace().getRoot().findFilesForLocationURI(uri);
            if (file.length != 0 && file[0].exists())
                return file[0];
        } catch (IllegalArgumentException e) {
            // Ignore and return null.
        }
        return null;
    }

    /**
     * Creates a folder and all parent folders if not existing. Project must
     * exist. <code> org.eclipse.ui.dialogs.ContainerGenerator</code> is too
     * heavy (creates a runnable)
     */
    public static void createFolder(
            IFolder folder,
            boolean force,
            boolean local,
            IProgressMonitor monitor) throws CoreException
    {
        if (!folder.exists()) {
            IContainer parent = folder.getParent();
            if (parent instanceof IFolder) {
                createFolder((IFolder)parent, force, local, null);
            }
            folder.create(force, local, monitor);
        }
    }

    public static IProgressMonitor getNonNullMonitor(
            IProgressMonitor monitor)
    {
        return (monitor == null) ? new NullProgressMonitor() : monitor;
    }

    public static void logError(
            String message)
    {
        CurlPlugin.getDefault().getLog().log(
                new Status(IStatus.ERROR, CurlUIIDs.ID_PLUGIN, ICurlStatusConstants.INTERNAL_ERROR, message, new Throwable()));
    }

    public static void logStatus(
            IStatus status)
    {
        CurlPlugin.getDefault().getLog().log(status);
    }

    public static void logError(
            String message,
            Throwable e)
    {
        CurlPlugin.getDefault().getLog().log(
                new Status(IStatus.ERROR, CurlUIIDs.ID_PLUGIN, ICurlStatusConstants.INTERNAL_ERROR, message, e));
    }

    public static void logWarning(
            String message)
    {
        CurlPlugin.getDefault().getLog().log(
                new Status(IStatus.WARNING, CurlUIIDs.ID_PLUGIN, IStatus.OK, message, new Throwable()));
    }

    public static void logInfo(
            String message)
    {
        CurlPlugin.getDefault().getLog().log(new Status(IStatus.INFO, CurlUIIDs.ID_PLUGIN, IStatus.OK, message, new Throwable()));
    }

    public static void logInfo(
            String message,
            Throwable t)
    {
        CurlPlugin.getDefault().getLog().log(new Status(IStatus.INFO, CurlUIIDs.ID_PLUGIN, IStatus.OK, message, t));
    }

    public static Shell getActiveWorkbenchShell()
    {
        IWorkbenchWindow window = CurlPlugin.getActiveWorkbenchWindow();
        if (window != null) {
            return window.getShell();
        }
        return null;
    }

    public static boolean amITheUIThread()
    {
        Shell shell = getActiveWorkbenchShell();
        if (shell == null)
            return false;
        Display display = shell.getDisplay();
        if (display == null || display.isDisposed())
            return false;
        if (Thread.currentThread() == display.getThread())
            return true;
        return false;
    }

    public static  void abort(
            String message,
            Throwable exception) throws CoreException
    {
        throw new CoreException(new Status(
                IStatus.ERROR,
                CurlUIIDs.ID_PLUGIN,
                Status.OK,
                message,
                exception));
    }

    /*
     * Creates an image descriptor for the given path in a bundle. The path can contain variables
     * like $NL$.
     * If no image could be found, useMissingImageDescriptor decides if either
     * the 'missing image descriptor' is returned or null.
     */
    public static ImageDescriptor createImageDescriptor(
            Bundle bundle,
            IPath path,
            boolean useMissingImageDescriptor)
    {
        URL url = FileLocator.find(bundle, path, null);
        if (url != null) {
            return ImageDescriptor.createFromURL(url);
        }
        if (useMissingImageDescriptor) {
            return ImageDescriptor.getMissingImageDescriptor();
        }
        return null;
    }

    public static void reportInStatusLine(
            final String message,
            final IWorkbenchPart part)
    {
        if (Display.getCurrent() == null) {
            Display.getDefault().asyncExec(new Runnable() {
                public void run()
                {
                    CoreUtil.reportInStatusLineWorker(message, part);
                }
            });
        } else {
            CoreUtil.reportInStatusLineWorker(message, part);
        }
    }

    public static CurlLineBreakpoint getLineBreakpoint(
            final CurlLineBreakpointDescriptor breakpointDesc)
    {
        IBreakpoint[] breakpoints =
            DebugPlugin.getDefault().getBreakpointManager().getBreakpoints(
                CurlUIIDs.ID_CURL_DEBUG_MODEL);
        for (int i = 0; i < breakpoints.length; i++) {
            if (!(breakpoints[i] instanceof CurlLineBreakpoint)) {
                continue;
            }
            CurlLineBreakpoint bp = (CurlLineBreakpoint)breakpoints[i];
            IMarker marker = bp.getMarker();
            try {
                if (marker != null &&
                    marker.exists() &&
                    marker.getType().equals(CurlUIIDs.ID_CURL_LINE_DEBUG_MARKER)) {
                    //TODO:  Don't forget the breakpoint condition
                    if (bp.fileNameEqual(breakpointDesc.getLocationURI().toString()) &&
                        bp.getLineNumber() == breakpointDesc.fLineNumber) {
                        return bp;
                    }
                }
            } catch (Exception e) {
                CoreUtil.logWarning("Error finding a breakpoint"); //$NON-NLS-1$
            }
        }
        return null;
    }

    public static boolean isPartEditor(
            IWorkbenchPart part)
    {
        CurlEditor editor = CoreUtil.getCurlEditor(part);
        if (editor != null && editor.getEditorInput() != null) {
            return true;
        }
        return false;
    }

    public static CurlEditor getCurlEditor(
            IWorkbenchPart part)
    {
        if (part instanceof CurlEditor) {
            return (CurlEditor)part;
        }
        return null;
    }

    private static  void reportInStatusLineWorker(
            final String message,
            final IWorkbenchPart part)
    {
        IEditorStatusLine statusLine = (IEditorStatusLine)part.getAdapter(IEditorStatusLine.class);
        if (statusLine != null) {
            if (message != null) {
                statusLine.setMessage(true, message, null);
            } else {
                statusLine.setMessage(true, null, null);
            }
        }
        if (message != null && CoreUtil.getActiveWorkbenchShell() != null) {
            CoreUtil.getActiveWorkbenchShell().getDisplay().beep();
        }
    }

    public static IProject getCurlProjectFromSelection(ISelection selection)
    {
        if (selection == null || selection.isEmpty() || !(selection instanceof IStructuredSelection)) {
            return null;
        }
        Object firstElement = ((IStructuredSelection)selection).getFirstElement();
        if (! (firstElement instanceof IAdaptable)) {
            return null;
        }
        Object adapter = ((IAdaptable)(firstElement)).getAdapter(
                IResource.class);
        if (adapter == null) {
            return null;
        }
        IProject project = ((IResource)adapter).getProject();
        if (!project.isOpen()) {
            return null;
        }
        try {
            return project.isNatureEnabled(CurlUIIDs.ID_CURL_NATURE) ? project : null;
        } catch (CoreException e) {
            // Nothing to do; pass on exception to eclipse.
            throw new RuntimeException(e);
        }
    }

    public static void displayError(final String title, final String msg)
    {
        if (Display.getCurrent() == null) {
            Display.getDefault().asyncExec(new Runnable() {
                public void run()
                {
                    MessageDialog.openError(CoreUtil.getActiveWorkbenchShell(), title, msg);
                }
            });
        } else {
            MessageDialog.openError(CoreUtil.getActiveWorkbenchShell(), title, msg);
        }
    }

    public static String convertPathToCurlURL(IPathEditorInput editorInput)
    {
// see note in com.curl.eclipse.remote.EclipseServer.readOpen()
//        String path = CurlUIIDs.CURL_PROXY_PREFIX + editorInput.getPath().toString();
//        return path;
        return editorInput.getPath().toString();
    }

    public static String convertPathToEclipse(
            String urlString)
    {
        if (urlString.startsWith(CurlUIIDs.CURL_PROXY_PREFIX)) {
// see note in com.curl.eclipse.remote.EclipseServer.readOpen()
// this case won't occur until we start using curl://eclipseproxy urls
            urlString = "file:/" + urlString.substring(CurlUIIDs.CURL_PROXY_PREFIX.length()); //$NON-NLS-1$

        }
        return urlString;
    }

    public static void showContextMenu(
            final Menu menu,
            final int x,
            final int y)
    {
        if (Platform.getOS().equals(Platform.OS_LINUX)) {
            // fmisiak's note:
            // On Linux, to show the context menu, we need to 
            // move the mouse pointer away from the curl graphic region.
            // Else the context menu just won't show up!
            // This is a side-effect of embedding the curl view
            // in the SWT container. I couldn't find any explanation
            // for this very strange behavior, though I've seen some
            // postings about SWT/Swing integration with similar issues
            // but with no conclusive explanation/solution.
            final Display display = menu.getDisplay();
            
            // Since we're going to hack the mouse position, we'll have
            // to restore it when the menu is shown
            final MenuListener menuListener = new MenuListener(){

                @Override
                public void menuHidden(MenuEvent e) {
                }

                @Override
                public void menuShown(MenuEvent e) {
                    menu.removeMenuListener(this);
                    postMoveMouseEvent(display, x, y);
                }};
            menu.addMenuListener(menuListener);
            
            // Put mouse outside of curl region
            new Thread() {
                @Override
                public void run()
                {
                    postMoveMouseEvent(display, 0, 0);
                }

            }.start();
            
            // wait a bit and make menu visible at location
            new Thread() {
                @Override
                public void run()
                {
                    try {
                        Thread.sleep(50);
                    } catch (InterruptedException e) {
                    }
                    menu.getDisplay().syncExec(new Runnable() {

                        @Override
                        public void run()
                        {
                            menu.setLocation(x, y);
                            menu.setVisible(true);
                        }
                    });
                }
            }.start();
        }      
        else {
            // not Linux (so it has to be win32 for now), let's be simple :-)
            menu.setLocation(x, y);
            menu.setVisible(true);
        }
    }
    
    private static void postMoveMouseEvent(
            final Display display,
            int x,
            int y)
    {
        Event event = new Event();
        event.type = SWT.MouseMove;
        event.x = x;
        event.y = y;
        display.post(event);
    }
    
    /**
     * Get a status line manager but if you cannot then return null
     * Note currently used.
     */
    public static IStatusLineManager getStatusLineManager()
    {
        IStatusLineManager statusLineManager = null;
        IWorkbenchWindow activeWindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
        if (activeWindow != null) {
            IWorkbenchPage activePage = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
            if (activePage != null) {
                IWorkbenchPart activePart = activePage.getActivePart();
                if (activePart instanceof IViewPart)
                    statusLineManager = ((IViewPart)activePart).getViewSite().getActionBars().getStatusLineManager();
                else if (activePart instanceof IEditorPart)
                    statusLineManager = ((IEditorPart)activePart).getEditorSite().getActionBars().getStatusLineManager();
            }            
        }

        return statusLineManager;
    }
}
