/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
/*******************************************************************************
 * Copyright (c) 2000, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.curl.eclipse.ui;


import org.eclipse.core.runtime.Assert;
import org.eclipse.jface.resource.CompositeImageDescriptor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;

import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlPluginImages;

/**
 * A {@link CurlElementImageDescriptor} consists of a base image and several adornments. The adornments
 * are computed according to the flags either passed during creation or set via the method
 *{@link #setAdornments(int)}. 
 * 
 * <p>
 * This class may be instantiated; it is not intended to be subclassed.
 * </p>
 *
 * @see JavaElementImageDescriptor
 * @since 2.0 
 */
public class CurlElementImageDescriptor extends CompositeImageDescriptor {
	
	/** Flag to render the abstract adornment. */
	public final static int ABSTRACT=      0x001;
	
	/** Flag to render the final adornment. */
	public final static int FINAL=         0x002;
	
	/** Flag to render the constant adornment. */
	public final static int CONSTANT=      0x004;
	
	/** Flag to render the deprecated adornment. */
	public final static int DEPRECATED=    0x008;
	
	/** Flag to render the implicit adornment. */
	public final static int IMPLICIT=      0x010;
	
	/** Flag to render the warning adornment. (for future use) */
	public final static int WARNING=       0x020;
	
	/** Flag to render the error adornment. (for future use) */
	public final static int ERROR=         0x040;
	
	/** Flag to render the 'override' adornment. */
//	public final static int OVERRIDES=     0x080;
	
	/** Flag to render the 'implements' adornment. */
//	public final static int IMPLEMENTS=    0x100;
	
	/** Flag to render the 'constructor' adornment. */
	public final static int CONSTRUCTOR=   0x200;
	
    /** Flag to render the 'inline' adornment. */
    public final static int INLINE=        0x400;

    /** Flag to render the 'sealed' adornment. */
    public final static int SEALED=        0x800;
    
    /** Flag to render the 'weak' adornment. */
    public final static int WEAK=         0x1000;
    	

	private final ImageDescriptor fBaseImage;
	private Point fSize = new Point(16, 16);

    private int fFlags;

	/**
	 * Creates a new JavaElementImageDescriptor.
	 * 
	 * @param baseImage an image descriptor used as the base image
	 * @param adornmentFlags flags indicating which adornments are to be rendered. See {@link #setAdornments(int)}
	 * 	for valid values.
	 */
    public CurlElementImageDescriptor(
            ImageDescriptor baseImage,
            int adornmentFlags)
    {
		fBaseImage= baseImage;
		Assert.isNotNull(fBaseImage);
		fFlags = adornmentFlags;
	}
	
    /**
	 * Sets the descriptors adornments.
	 * @param adornments the image descriptors adornments
	 */
	public void setAdornments(int adornments) {
		Assert.isTrue(adornments >= 0);
		fFlags= adornments;
	}

	/**
	 * Returns the current adornments.
	 * 
	 * @return the current adornments
	 */
	public int getAdronments() {
		return fFlags;
	}

	/**
	 * Sets the size of the image created by calling {@link #createImage()}.
	 * 
	 * @param size the size of the image returned from calling {@link #createImage()}
	 */
	public void setImageSize(Point size) {
		Assert.isNotNull(size);
		Assert.isTrue(size.x >= 0 && size.y >= 0);
		fSize= size;
	}
	
	/**
	 * Returns the size of the image created by calling {@link #createImage()}.
	 * 
	 * @return the size of the image created by calling {@link #createImage()}
	 */
	public Point getImageSize() {
		return new Point(fSize.x, fSize.y);
	}
	
	/* (non-Javadoc)
	 * Method declared in CompositeImageDescriptor
	 */
	@Override
    protected Point getSize() {
		return fSize;
	}
	
	/* (non-Javadoc)
	 * Method declared on Object.
	 */
	@Override
    public boolean equals(Object object) {
		if (object == null || !CurlElementImageDescriptor.class.equals(object.getClass()))
			return false;
			
		CurlElementImageDescriptor other= (CurlElementImageDescriptor)object;
		return (fBaseImage.equals(other.fBaseImage) && fFlags == other.fFlags && fSize.equals(other.fSize));
	}
	
	/* (non-Javadoc)
	 * Method declared on Object.
	 */
	@Override
    public int hashCode() {
		return fBaseImage.hashCode() | fFlags | fSize.hashCode();
	}
	
	/* (non-Javadoc)
	 * Method declared in CompositeImageDescriptor
	 */
	@Override
    protected void drawCompositeImage(int width, int height) {
		ImageData bg= getImageData(fBaseImage);
			
		if ((fFlags & DEPRECATED) != 0) { // draw *behind* the full image
			Point size= getSize();
			ImageData data= getImageData(CurlPluginImages.DESC_OVR_DEPRECATED);
			drawImage(data, 0, size.y - data.height);
		}
		drawImage(bg, 0, 0);
				
		drawTopRight();
		drawBottomRight();
//		drawBottomLeft();
		

	}
	
	private ImageData getImageData(ImageDescriptor descriptor) {
		ImageData data= descriptor.getImageData(); // see bug 51965: getImageData can return null
		if (data == null) {
			data= DEFAULT_IMAGE_DATA;
			CoreUtil.logError("Image data not available: " + descriptor.toString()); //$NON-NLS-1$
		}
		return data;
	}
	
	private void addTopRightImage(ImageDescriptor desc, Point pos) {
		ImageData data= getImageData(desc);
		int x= pos.x - data.width;
		if (x >= 0) {
			drawImage(data, x, pos.y);
			pos.x= x;
		}
	}
	
	private void addBottomRightImage(ImageDescriptor desc, Point pos) {
		ImageData data= getImageData(desc);
		int x= pos.x - data.width;
		int y= pos.y - data.height;
		if (x >= 0 && y >= 0) {
			drawImage(data, x, y);
			pos.x= x;
		}
	}
	
//	private void addBottomLeftImage(ImageDescriptor desc, Point pos) {
//		ImageData data= getImageData(desc);
//		int x= pos.x;
//		int y= pos.y - data.height;
//		if (x + data.width < getSize().x && y >= 0) {
//			drawImage(data, x, y);
//			pos.x= x + data.width;
//		}
//	}
	
	
	private void drawTopRight() {
		Point pos= new Point(getSize().x, 0);
		if ((fFlags & ABSTRACT) != 0) {
			addTopRightImage(CurlPluginImages.DESC_OVR_ABSTRACT, pos);
		}
		if ((fFlags & CONSTRUCTOR) != 0) {
			addTopRightImage(CurlPluginImages.DESC_OVR_CONSTRUCTOR, pos);
		}
		if ((fFlags & FINAL) != 0) {
			addTopRightImage(CurlPluginImages.DESC_OVR_FINAL, pos);
		}
        if ((fFlags & IMPLICIT) != 0) {
            addTopRightImage(CurlPluginImages.DESC_OVR_IMPLICIT, pos);
        }
	}		
	
	private void drawBottomRight() {
		Point size= getSize();
		Point pos= new Point(size.x, size.y);

        if ((fFlags & INLINE) != 0) {
            addBottomRightImage(CurlPluginImages.DESC_OVR_INLINE, pos);
        }
        if ((fFlags & SEALED) != 0) {
            addBottomRightImage(CurlPluginImages.DESC_OVR_SEALED, pos);
        }
        if ((fFlags & CONSTANT) != 0) {
            addBottomRightImage(CurlPluginImages.DESC_OVR_CONSTANT, pos);
        }
        if ((fFlags & WEAK) != 0) {
            addBottomRightImage(CurlPluginImages.DESC_OVR_WEAK, pos);
        }

	}		
	
//	private void drawBottomLeft() {
//		Point pos= new Point(0, getSize().y);
//		if ((fFlags & ERROR) != 0) {
//			addBottomLeftImage(CurlPluginImages.DESC_OVR_ERROR, pos);
//		}
//		if ((fFlags & WARNING) != 0) {
//			addBottomLeftImage(CurlPluginImages.DESC_OVR_WARNING, pos);
//		}
//	}		
}
