/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.search;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.ui.IWorkingSet;
import org.eclipse.ui.IWorkingSetManager;
import org.eclipse.ui.PlatformUI;

import com.curl.eclipse.core.SearchForEnum;


class SearchPatternData
{
    private final boolean fIsCaseSensitive;
    private final String fTextPattern;
    private final int fScope;
    private final IWorkingSet[] fWorkingSets;
    private final LimitToEnum fLimitToEnum;
    private final SearchForEnum fSearchForEnum;
    
    public SearchPatternData(
            SearchForEnum searchForEnum,
            LimitToEnum limitToEnum,
            String textPattern,
            boolean isCaseSensitive,
            int scope,
            IWorkingSet[] workingSets)
    {
        this.fSearchForEnum = searchForEnum;
        this.fLimitToEnum = limitToEnum;
        this.fIsCaseSensitive= isCaseSensitive;
        this.fTextPattern= textPattern;
        this.fScope= scope;
        this.fWorkingSets= workingSets; // can be null
    }
    
    public void store(IDialogSettings settings) {
        settings.put("searchForEnum", fSearchForEnum.toString()); //$NON-NLS-1$
        settings.put("limitToEnum", fLimitToEnum.toString()); //$NON-NLS-1$
        settings.put("ignoreCase", !fIsCaseSensitive); //$NON-NLS-1$
        settings.put("textPattern", fTextPattern); //$NON-NLS-1$
        settings.put("scope", fScope); //$NON-NLS-1$
        if (fWorkingSets != null) {
            String[] wsIds= new String[fWorkingSets.length];
            for (int i= 0; i < fWorkingSets.length; i++) {
                wsIds[i]= fWorkingSets[i].getLabel();
            }
            settings.put("workingSets", wsIds); //$NON-NLS-1$
        } else {
            settings.put("workingSets", new String[0]); //$NON-NLS-1$
        }

    }
    
    public static SearchPatternData create(IDialogSettings settings) {
        String textPattern= settings.get("textPattern"); //$NON-NLS-1$
        String[] wsIds= settings.getArray("workingSets"); //$NON-NLS-1$
        IWorkingSet[] workingSets= null;
        if (wsIds != null && wsIds.length > 0) {
            IWorkingSetManager workingSetManager= PlatformUI.getWorkbench().getWorkingSetManager();
            workingSets= new IWorkingSet[wsIds.length];
            for (int i= 0; workingSets != null && i < wsIds.length; i++) {
                workingSets[i]= workingSetManager.getWorkingSet(wsIds[i]);
                if (workingSets[i] == null) {
                    workingSets= null;
                }
            }
        }
        try {
            String searchForStr = settings.get("searchForEnum"); //$NON-NLS-1$
            SearchForEnum searchForEnum = SearchForEnum.type;
            try {
                if (searchForStr != null) {
                    searchForEnum = SearchForEnum.valueOf(searchForStr);
                }
            } catch (IllegalArgumentException e) {
                // possible when migrating to newer version
            }

            String limitToStr = settings.get("limitToEnum"); //$NON-NLS-1$
            LimitToEnum limitToEnum  = LimitToEnum.DEFINITIONS;
            try {
                if (limitToStr != null) {
                    limitToEnum = LimitToEnum.valueOf(limitToStr);
                }
            } catch (IllegalArgumentException e) {
                // possible when migrating to newer version
            }
            int scope= settings.getInt("scope"); //$NON-NLS-1$
            boolean ignoreCase= settings.getBoolean("ignoreCase"); //$NON-NLS-1$
            return  new SearchPatternData(searchForEnum, limitToEnum, textPattern, !ignoreCase, scope, workingSets);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    public String getPattern() {
        return fTextPattern;
    }

    public boolean isCaseSensitive() {
        return fIsCaseSensitive;
    }

    public String getTextPattern()
    {
        return fTextPattern;
    }

    public int getScope()
    {
        return fScope;
    }

    public IWorkingSet[] getWorkingSets()
    {
        return fWorkingSets;
    }

    public LimitToEnum getLimitToEnum()
    {
        return fLimitToEnum;
    }

    public SearchForEnum getSearchForEnum()
    {
        return fSearchForEnum;
    }
}
