/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.search;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.jface.viewers.AbstractTreeViewer;
import org.eclipse.jface.viewers.ITreeContentProvider;

import com.curl.eclipse.core.BindingInfo;
import com.curl.eclipse.core.CurlElement;
import com.curl.eclipse.core.SearchForEnum;
import com.curl.eclipse.util.CoreUtil;

/**
 * @author fmisiak
 * see org.eclipse.jdt.internal.ui.search.LevelTreeContentProvider
 */
class LevelTreeContentProvider extends CurlSearchContentProvider
    implements ITreeContentProvider
{
    private Map<Object, Set<Object>> fChildrenMap;
    
    // TODO: implement grouping action a la JDT when we have cross-ref search with CPA
    @SuppressWarnings("unused") 
    private int fCurrentLevel;

    public LevelTreeContentProvider(
            CurlSearchResultPage curlSearchResultPage,
            int level)
    {
        super(curlSearchResultPage);
        fCurrentLevel= level;
        
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
     */
    public Object getParent(Object child) {
        if (child instanceof IProject) {
            return null;
        }
        if (child instanceof IContainer) {
            IContainer container = (IContainer)child;
            return container.getParent();
        }
        if (child instanceof IFile) {
            IFile file = (IFile)child;
            return file.getParent();
        }
        if (child instanceof CurlElement) {
            CurlElement el = (CurlElement)child;
            if (el.getType() == null)
            {
                return el.getFile().getParent();
            }
            switch(el.getType())
            {
            case constructor: // fall through
            case field: // fall through
            case method:
                // dynamically build a node for the class/enum parent corresponding to the member
                return createCurlElement(el.getFile(), el.getParentType(), el.getNameOfType(), el.getNameOfType(), el.getParentRow(), el.getParentCol(),
                        el.getInfoParent());
                
            case classCurl: // fall through
            case enumCurl: // fall through
            case globalvar: // fall through
            case macro: // fall through
            case proc:
                return el.getFile();
//                return createCurlElement(el.getFile(), null, "", null);
                
            case type:  // fall thru
            case all:
                CoreUtil.logError("Unexpected 'all' type"); //$NON-NLS-1$
                return null;
            }
        }
        return null;
        // TODO
//        Object possibleParent = null;
        //        Object possibleParent= fContentProvider.getParent(child);
//        if (possibleParent instanceof IJavaElement) {
//            IJavaElement javaElement= (IJavaElement) possibleParent;
//            for (int j= fCurrentLevel; j < MAX_LEVEL + 1; j++) {
//                for (int i= 0; i < JAVA_ELEMENT_TYPES[j].length; i++) {
//                    if (javaElement.getElementType() == JAVA_ELEMENT_TYPES[j][i]) {
//                        return null;
//                    }
//                }
//            }
//        } else if (possibleParent instanceof IResource) {
//            IResource resource= (IResource) possibleParent;
//            for (int j= fCurrentLevel; j < MAX_LEVEL + 1; j++) {
//                for (int i= 0; i < RESOURCE_TYPES[j].length; i++) {
//                    if (resource.getType() == RESOURCE_TYPES[j][i]) {
//                        return null;
//                    }
//                }
//            }
//        }
//        if (fCurrentLevel != LEVEL_FILE && child instanceof IType) {
//            IType type= (IType) child;
//            if (possibleParent instanceof ICompilationUnit
//                    || possibleParent instanceof IClassFile)
//                possibleParent= type.getPackageFragment();
//        }
//        return possibleParent;
    }
    
    private CurlElement createCurlElement(
            IFile file,
            SearchForEnum type,
            String nameOfType,
            String symbolName, 
            int row, 
            int col, 
            BindingInfo bindingInfo)
    {
        return new CurlElement(file, type, nameOfType, symbolName, row, col, bindingInfo);
    }

    /* (non-Javadoc)
     * @see com.curl.eclipse.search.CurlSearchContentProvider#initialize(com.curl.eclipse.search.CurlSearchResult)
     */
    @Override
    protected synchronized void initialize(CurlSearchResult result)
    {
        super.initialize(result);
        fChildrenMap= new HashMap<Object, Set<Object>>();
        if (result != null) {
            Object[] elements= result.getElements();
            for (Object element : elements) {
                if (fPage.getDisplayedMatchCount(element) > 0) {
                    insert(null, null, element);
                }
            }
        }
    }
    
    protected void insert(
            Map<Object, Set<Object>> toAdd,
            Set<Object> toUpdate,
            Object child)
    {
        Object parent= getParent(child);
        while (parent != null) {
            if (insertChild(parent, child)) {
                if (toAdd != null)
                    insertInto(parent, child, toAdd);
            } else {
                if (toUpdate != null)
                    toUpdate.add(parent);
                return;
            }
            child= parent;
            parent= getParent(child);
        }
        if (insertChild(fResult, child)) {
            if (toAdd != null)
                insertInto(fResult, child, toAdd);
        }
    }
    
    private boolean insertChild(
            Object parent,
            Object child)
    {
        return insertInto(parent, child, fChildrenMap);
    }

    private boolean insertInto(
            Object parent,
            Object child,
            Map<Object, Set<Object>> map) 
    {
        Set<Object> children= map.get(parent);
        if (children == null) {
            children= new HashSet<Object>();
            map.put(parent, children);
        }
        return children.add(child);
    }
    
    protected void remove(
            Set<Object> toRemove,
            Set<Object> toUpdate,
            Object element) 
    {
        // precondition here:  fResult.getMatchCount(child) <= 0
    
        if (hasChildren(element)) {
            if (toUpdate != null)
                toUpdate.add(element);
        } else {
            if (fPage.getDisplayedMatchCount(element) == 0) {
                fChildrenMap.remove(element);
                Object parent= getParent(element);
                if (parent != null) {
                    if (removeFromSiblings(element, parent)) {
                        remove(toRemove, toUpdate, parent);
                    }
                } else {
                    if (removeFromSiblings(element, fResult)) {
                        if (toRemove != null)
                            toRemove.add(element);
                    }
                }
            } else {
                if (toUpdate != null) {
                    toUpdate.add(element);
                }
            }
        }
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
     */
    public Object[] getElements(Object inputElement) {
        return getChildren(inputElement);
    }
    
    /**
     * @param element
     * @param parent
     * @return returns true if it really was a remove (i.e. element was a child of parent).
     */
    private boolean removeFromSiblings(Object element, Object parent) {
        Set<Object> siblings=fChildrenMap.get(parent);
        if (siblings != null) {
            return siblings.remove(element);
        } else {
            return false;
        }
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
     */
    public Object[] getChildren(
            Object parentElement)
    {
        Set<Object> children= fChildrenMap.get(parentElement);
        if (children == null)
            return EMPTY_ARR;
        int limit= fPage.getElementLimit().intValue();
        if (limit != -1 && limit < children.size()) {
            Object[] limitedArray= new Object[limit];
            Iterator<Object> iterator= children.iterator();
            for (int i= 0; i < limit; i++) {
                limitedArray[i]= iterator.next();
            }
            return limitedArray;
        }
        
        return children.toArray();
    }

    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
     */
    public boolean hasChildren(Object element) {
        Set<Object> children= fChildrenMap.get(element);
        return children != null && !children.isEmpty();
    }
    
    /* (non-Javadoc)
     * @see com.curl.eclipse.search.CurlSearchContentProvider#clear()
     */
    @Override
    public void clear() {
        initialize(fResult);
        fPage.getViewer().refresh();
    }

    /* (non-Javadoc)
     * @see com.curl.eclipse.search.CurlSearchContentProvider#elementsChanged(java.lang.Object[])
     */
    @Override
    public synchronized void elementsChanged(Object[] updatedElements) {
        if (fResult == null)
            return;
        
        AbstractTreeViewer viewer= (AbstractTreeViewer) fPage.getViewer();

        Set<Object> toRemove= new HashSet<Object>();
        Set<Object> toUpdate= new HashSet<Object>();
        Map<Object, Set<Object>> toAdd= new HashMap<Object, Set<Object>>();
        for (Object element : updatedElements) {
            if (fPage.getDisplayedMatchCount(element) > 0)
                insert(toAdd, toUpdate, element);
            else
                remove(toRemove, toUpdate, element);
        }
        
        viewer.remove(toRemove.toArray());
        for (Object parent : toAdd.keySet()) {
            Set<Object> children= toAdd.get(parent);
            viewer.add(parent, children.toArray());
        }
        for (Object object : toUpdate) {
            viewer.refresh(object);
        }
        
    }

    public void setLevel(int level) {
        fCurrentLevel= level;
        initialize(fResult);
        fPage.getViewer().refresh();
    }
    
}
