/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.search;

import java.util.HashMap;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.search.ui.IContextMenuConstants;
import org.eclipse.search.ui.ISearchResultViewPart;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.search.ui.text.AbstractTextSearchViewPage;
import org.eclipse.search.ui.text.Match;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.ActionContext;
import org.eclipse.ui.dialogs.PreferencesUtil;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.IPageSite;
import org.eclipse.ui.texteditor.ITextEditor;

import com.curl.eclipse.actions.EditorOpener;
import com.curl.eclipse.core.CurlElement;
import com.curl.eclipse.editors.CurlEditor;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;

/**
 * Eclipse search result extension. See other implementations:
 * org.eclipse.search.internal.ui.text.FileSearchPage
 * org.eclipse.jdt.internal.ui.refactoring.nls.search.NLSSearchResultPage
 * 
 * @author fmisiak
 * 
 */
public class CurlSearchResultPage extends AbstractTextSearchViewPage
{

    /**
     */
    private static final class ViewerSorter extends ViewerComparator
    {

        private final ILabelProvider fLabelProvider;

        ViewerSorter(
                ILabelProvider labelProvider)
        {
            fLabelProvider = labelProvider;
        }

        @Override
        @SuppressWarnings("unchecked") 
        public int compare(
                Viewer viewer,
                Object e1,
                Object e2)
        {
            String name1 = fLabelProvider.getText(e1);
            String name2 = fLabelProvider.getText(e2);
            if (name1 == null)
                name1 = "";//$NON-NLS-1$
            if (name2 == null)
                name2 = "";//$NON-NLS-1$
            return getComparator().compare(name1, name2);
        }
    }

    private final EditorOpener fEditorOpener = new EditorOpener();

    private NewSearchViewActionGroup fActionGroup;
    private CurlSearchContentProvider fContentProvider;

    private int fCurrentGrouping;

    private static final int DEFAULT_ELEMENT_LIMIT = 1000;

    private static final String GROUP_FILTERING = "com.curl.eclipse.search.resultpage.filtering"; //$NON-NLS-1$

    public CurlSearchResultPage()
    {
        super(FLAG_LAYOUT_FLAT | FLAG_LAYOUT_TREE);
        setElementLimit(new Integer(DEFAULT_ELEMENT_LIMIT));

    }

    @Override
    protected StructuredViewer getViewer()
    {
        return super.getViewer();
    }

    @Override
    protected void configureTableViewer(
            TableViewer viewer)
    {
        viewer.setUseHashlookup(true);
        fContentProvider = new ContentProvider(this);
        viewer.setContentProvider(fContentProvider);
        CurLabelProvider labelProvider = new CurLabelProvider(false);
        viewer.setLabelProvider(labelProvider);
        viewer.setComparator(new ViewerSorter(labelProvider));
        // todo: to allow drag gesture of items from search result
        // addDragAdapters(viewer);
    }

    @Override
    protected void configureTreeViewer(
            TreeViewer viewer)
    {
        viewer.setUseHashlookup(true);

        CurLabelProvider labelProvider = new CurLabelProvider(true);
        viewer.setLabelProvider(labelProvider);

        fContentProvider = new LevelTreeContentProvider(this, fCurrentGrouping);
        viewer.setContentProvider(fContentProvider);
        viewer.setComparator(new ViewerSorter(labelProvider));

        // todo: to allow drag gesture of items from search result
        // addDragAdapters(viewer);
    }

    /*
     * @see org.eclipse.search.ui.text.AbstractTextSearchViewPage#showMatch(org.eclipse.search.ui.text.Match,
     *      int, int)
     */
    @Override
    protected void showMatch(
            Match match,
            int offset,
            int length,
            boolean activate) throws PartInitException
    {
        if (match.getElement() instanceof CurlElement) {
            CurlElement el = (CurlElement)match.getElement();
            IFile file = el.getFile();
            IEditorPart editor = fEditorOpener.open(file, activate, NewSearchUI.reuseEditor());
            if (editor instanceof CurlEditor && match instanceof CurlMatch) {
                CurlEditor curlEditor = (CurlEditor)editor;
                // in this case the offset is "fake" see CurlMatch.computeDummyOffset()
                curlEditor.selectAndReveal(el);
            } else if (editor instanceof ITextEditor) {
                ITextEditor textEditor = (ITextEditor)editor;
                textEditor.selectAndReveal(offset, length);
            } else if (editor != null) {
                // not a text based editor, let's do what the JDT editor would do...
                showWithMarker(editor, file, offset, length);
            }
        }
    }

    private void showWithMarker(
            IEditorPart editor,
            IFile file,
            int offset,
            int length) throws PartInitException
    {
        IMarker marker = null;
        try {
            marker = file.createMarker(NewSearchUI.SEARCH_MARKER);
            HashMap<String, Integer> attributes = new HashMap<String, Integer>(4);
            attributes.put(IMarker.CHAR_START, new Integer(offset));
            attributes.put(IMarker.CHAR_END, new Integer(offset + length));
            marker.setAttributes(attributes);
            IDE.gotoMarker(editor, marker);
        } catch (CoreException e) {
            throw new PartInitException(SearchMessages.FileSearchPage_error_marker, e);
        } finally {
            if (marker != null)
                try {
                    marker.delete();
                } catch (CoreException e) {
                    // ignore
                }
        }
    }

    @Override
    protected void fillContextMenu(
            IMenuManager mgr)
    {
        super.fillContextMenu(mgr);

        // addSortActions(mgr); TODO provide Curl sorting/grouping buttons

        fActionGroup.setContext(new ActionContext(getSite().getSelectionProvider().getSelection()));
        fActionGroup.fillContextMenu(mgr);
    }

    // TODO: provide Curl sorting/grouping buttons
    // private void addSortActions(IMenuManager mgr) {
    // if (getLayout() != FLAG_LAYOUT_FLAT)
    // return;
    // MenuManager sortMenu= new MenuManager(SearchMessages.FileSearchPage_sort_by_label);
    // sortMenu.add(fSortByNameAction);
    // sortMenu.add(fSortByPathAction);
    //        
    // fSortByNameAction.setChecked(fCurrentSortOrder == fSortByNameAction.getSortOrder());
    // fSortByPathAction.setChecked(fCurrentSortOrder == fSortByPathAction.getSortOrder());
    //        
    // mgr.appendToGroup(IContextMenuConstants.GROUP_VIEWER_SETUP, sortMenu);
    // }

    @Override
    public void setViewPart(
            ISearchResultViewPart part)
    {
        super.setViewPart(part);
        fActionGroup = new NewSearchViewActionGroup(part);
    }

    @Override
    public void init(
            IPageSite site)
    {
        super.init(site);
        IMenuManager menuManager = site.getActionBars().getMenuManager();
        menuManager.insertBefore(IContextMenuConstants.GROUP_PROPERTIES, new Separator(GROUP_FILTERING));
        fActionGroup.fillActionBars(site.getActionBars());
        menuManager.appendToGroup(IContextMenuConstants.GROUP_PROPERTIES, new Action(
                SearchMessages.CurlSearchResultPage_preferences_label) {
            @Override
            public void run()
            {
                String pageId = "org.eclipse.search.preferences.SearchPreferencePage"; //$NON-NLS-1$
                PreferencesUtil.createPreferenceDialogOn(
                        CoreUtil.getActiveWorkbenchShell(),
                        pageId,
                        null,
                        null).open();
            }
        });
    }

    @Override
    public void dispose()
    {
        fActionGroup.dispose();
        super.dispose();
    }

    @Override
    protected void elementsChanged(
            Object[] objects)
    {
        if (fContentProvider != null)
            fContentProvider.elementsChanged(objects);
    }

    @Override
    protected void clear()
    {
        if (fContentProvider != null)
            fContentProvider.clear();
    }

    // TODO: for future grouping actions
    // public void setSortOrder(int sortOrder) {
    // fCurrentSortOrder= sortOrder;
    // DecoratingLabelProvider lpWrapper= (DecoratingLabelProvider) getViewer().getLabelProvider();
    // ((FileLabelProvider) lpWrapper.getLabelProvider()).setOrder(sortOrder);
    // getViewer().refresh();
    // getSettings().put(KEY_SORTING, fCurrentSortOrder);
    // }

    // TODO: for future grouping state
    @Override
    public void restoreState(
            IMemento memento)
    {
        super.restoreState(memento);
        // try {
        // fCurrentSortOrder= getSettings().getInt(KEY_SORTING);
        // } catch (NumberFormatException e) {
        // fCurrentSortOrder= fSortByNameAction.getSortOrder();
        // }
        // int elementLimit= DEFAULT_ELEMENT_LIMIT;
        // try {
        // elementLimit= getSettings().getInt(KEY_LIMIT);
        // } catch (NumberFormatException e) {
        // }
        // if (memento != null) {
        // Integer value= memento.getInteger(KEY_SORTING);
        // if (value != null)
        // fCurrentSortOrder= value.intValue();
        //            
        // value= memento.getInteger(KEY_LIMIT);
        // if (value != null)
        // elementLimit= value.intValue();
        // }
        // setElementLimit(new Integer(elementLimit));
    }

    @Override
    public void saveState(
            IMemento memento)
    {
        super.saveState(memento);
        // memento.putInteger(KEY_SORTING, fCurrentSortOrder);
        // memento.putInteger(KEY_LIMIT, getElementLimit().intValue());
    }

    @Override
    protected void handleOpen(
            OpenEvent event)
    {
        Object firstElement = ((IStructuredSelection)event.getSelection()).getFirstElement();
        if (firstElement instanceof CurlElement) {
            if (getDisplayedMatchCount(firstElement) == 0) {
                try {
                    fEditorOpener.openElement((CurlElement)firstElement, NewSearchUI.reuseEditor());
                } catch (CoreException e) {
                    MultiStatus status = new MultiStatus(
                            CurlUIIDs.ID_PLUGIN,
                            IStatus.ERROR,
                            SearchMessages.CurlSearchopen_editor_error_message,
                            null);
                    ErrorDialog.openError(
                            getSite().getShell(),
                            SearchMessages.OpenAction_label,
                            SearchMessages.OpenAction_error_message,
                            status);
                }
                return;
            }
        }
        super.handleOpen(event);
    }
}
