/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.search;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.search.ui.ISearchPage;
import org.eclipse.search.ui.ISearchPageContainer;
import org.eclipse.search.ui.ISearchQuery;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.search.ui.text.FileTextSearchScope;
import org.eclipse.search.ui.text.TextSearchQueryProvider;
import org.eclipse.search.ui.text.TextSearchQueryProvider.TextSearchInput;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkingSet;
import org.eclipse.ui.PlatformUI;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.core.SearchForEnum;
import com.curl.eclipse.util.CurlUIIDs;

/**
 * @author fmisiak 
 * implementation derived from org.eclipse.jdt.internal.ui.search.JavaSearchPage
 * 
 */
public class CurlSearchPage extends DialogPage implements ISearchPage
{
    private static final int HISTORY_SIZE = 12;
    // Dialog store id constants
    private static final String PAGE_NAME = "CurlSearchPage"; //$NON-NLS-1$
//    private static final String STORE_CASE_SENSITIVE = "CASE_SENSITIVE"; //$NON-NLS-1$
    private static final String STORE_HISTORY = "HISTORY"; //$NON-NLS-1$
    private static final String STORE_HISTORY_SIZE = "HISTORY_SIZE"; //$NON-NLS-1$
    private boolean fFirstTime = true;
    private final List<SearchPatternData> fPreviousSearchPatterns;
    private Combo fPattern;
    private Button fCaseSensitive;
//    private boolean fIsCaseSensitive;
    private Button[] fSearchFor;
    private Button[] fLimitTo;
    private ISearchPageContainer fContainer;

    public CurlSearchPage()
    {
        fPreviousSearchPatterns = new ArrayList<SearchPatternData>();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.jface.dialogs.DialogPage#dispose()
     */
    @Override
    public void dispose()
    {
        writeConfiguration();
        super.dispose();
    }

    public boolean performAction()
    {
        SearchPatternData data = getPatternData();
        final FileTextSearchScope scope = createTextSearchScopeAndDescription();
        
        final ISearchQuery query;
        
        try {
            boolean caseSensitive = data.isCaseSensitive();
            SearchForEnum searchForEnum = data.getSearchForEnum();
            String pattern = data.getPattern();
            switch (data.getLimitToEnum()) {
            // original Curl reg expr for "identifier" search:
            // pattern = "(?<![\\w_?-])" + Pattern.quote(pattern) + "(?![\\w_?-])";
            case IDENTIFIER_LIKE: {
                query = createPlainTextSearchQuery(
                        pattern,
                        caseSensitive,
                        scope);
                break;
            }
            
            case DEFINITIONS:
                query = createDefinitionSearchQuery(scope, caseSensitive, searchForEnum, pattern);
                break;
    
            default:
                return false;
            }
        }
        catch (CoreException e) {
            ErrorDialog.openError(
                    getShell(),
                    SearchMessages.TextSearchPage_searchproblems_title,
                    SearchMessages.TextSearchPage_searchproblems_message,
                    e.getStatus());
            return false;
        }
        NewSearchUI.runQueryInBackground(query);
        return true;
    }

    static public ISearchQuery createDefinitionSearchQuery(
            final FileTextSearchScope scope,
            boolean caseSensitive,
            SearchForEnum searchForEnum,
            String pattern)
    {
        final ISearchQuery query;
        DefinitionQuerySpecification querySpec = new DefinitionQuerySpecification(
                searchForEnum.toString(),
                pattern,
                scope,
                scope.getDescription(),   // note: as we get more sophisticated, 
                                          // we'll may allow searching in selected 
                                          // curl elements, this description will have 
                                          // to mimic the jdt way
                caseSensitive);
        query= new CurlSearchDefinitionQuery(querySpec);
        return query;
    }

    static private String buildRegExpForIdentifierLikeSearch(
            String pattern)
    {
        final String searchedPattern;
        StringBuilder finalPattern = new StringBuilder();
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < pattern.length(); i++) {
            char c = pattern.charAt(i);
            switch (c) {
            case '*':
                quoteAndAppendFrag(finalPattern, sb);
                finalPattern.append("[\\w_?\\-.]*"); //$NON-NLS-1$
                break;
            case '!':
                quoteAndAppendFrag(finalPattern, sb);
                finalPattern.append("[\\w_?\\-.]{1}"); //$NON-NLS-1$
                break;
            default:
                sb.append(c);
                break;
            }
        }
        quoteAndAppendFrag(finalPattern, sb);
        searchedPattern = "(?<![\\w_])" + finalPattern.toString() + "(?![\\w_?\\-])"; //$NON-NLS-1$ //$NON-NLS-2$
        return searchedPattern;
    }

    public static ISearchQuery createPlainTextSearchQuery(
            final String plainPattern,
            final boolean isCaseSensitive,
            final FileTextSearchScope scope) throws CoreException
    {
        final String searchedPattern = buildRegExpForIdentifierLikeSearch(plainPattern);

        TextSearchInput input = new TextSearchInput() {
            @Override
            public FileTextSearchScope getScope()
            {
                return scope;
            }

            @Override
            public String getSearchText()
            {
                return searchedPattern;
            }

            @Override
            public boolean isCaseSensitiveSearch()
            {
                return isCaseSensitive;
            }

            @Override
            public boolean isRegExSearch()
            {
                return true;
            }
        };
        // delegate the search and display to eclipse file search plugin
        return TextSearchQueryProvider.getPreferred().createQuery(input);
    }

    private FileTextSearchScope createTextSearchScopeAndDescription()
    {
        final FileTextSearchScope scope;
        CurlSearchScopeFactory factory= CurlSearchScopeFactory.getInstance();
        
        switch (getContainer().getSelectedScope()) {
        case ISearchPageContainer.SELECTION_SCOPE:
            scope = factory.getSelectedResourcesScope(getContainer().getSelection());
            break;
            
        case ISearchPageContainer.SELECTED_PROJECTS_SCOPE:
            String[] projectNames= getContainer().getSelectedProjectNames();
            scope = factory.getEnclosingProjectScope(projectNames);
            break;
            
        case ISearchPageContainer.WORKING_SET_SCOPE:
            IWorkingSet[] workingSets = getContainer().getSelectedWorkingSets();
            scope = factory.createCurlSearchScope(workingSets);
            break;
            
        case ISearchPageContainer.WORKSPACE_SCOPE:
        default:    // fall through
            scope = factory.createWorkspaceScope();
            break;
        }
        return scope;
    }

    

    /**
     * Append to finalPattern a quoted version of sb, and reset content of sb.
     * 
     * @param finalPattern
     * @param sb
     */
    private static void quoteAndAppendFrag(
            StringBuilder finalPattern,
            StringBuilder sb)
    {
        finalPattern.append(Pattern.quote(sb.toString()));
        sb.delete(0, sb.length());
    }

    private SearchPatternData findInPrevious(
            String pattern)
    {
        for (SearchPatternData element : fPreviousSearchPatterns) {
            if (pattern.equals(element.getPattern())) {
                return element;
            }
        }
        return null;
    }

    /**
     * Return search pattern data and update previous searches. An existing entry will be updated.
     * 
     * @return the pattern data
     */
    private SearchPatternData getPatternData()
    {
        String pattern = getPattern();
        SearchPatternData match = findInPrevious(pattern);
        if (match != null) {
            fPreviousSearchPatterns.remove(match);
        }
        match = new SearchPatternData(
                getSearchFor(),
                getLimitTo(),
                pattern,
                fCaseSensitive.getSelection(),
                getContainer().getSelectedScope(),
                getContainer().getSelectedWorkingSets());
        fPreviousSearchPatterns.add(0, match); // insert on top
        return match;
    }

    public void setContainer(
            ISearchPageContainer container)
    {
        fContainer = container;
    }

    /**
     * Returns the search page's container.
     * 
     * @return the search page container
     */
    private ISearchPageContainer getContainer()
    {
        return fContainer;
    }

    public void createControl(
            Composite parent)
    {
        initializeDialogUnits(parent);
        readConfiguration();
        Composite result = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        layout.horizontalSpacing = 10;
        result.setLayout(layout);
        Control expressionComposite = createExpression(result);
        expressionComposite.setLayoutData(new GridData(GridData.FILL, GridData.CENTER, true, false, 2, 1));
        Label separator = new Label(result, SWT.NONE);
        separator.setVisible(false);
        GridData data = new GridData(GridData.FILL, GridData.FILL, false, false, 2, 1);
        data.heightHint = convertHeightInCharsToPixels(1) / 3;
        separator.setLayoutData(data);
        Control searchFor = createSearchFor(result);
        searchFor.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, false, 1, 1));
        Control limitTo = createLimitTo(result);
        limitTo.setLayoutData(new GridData(GridData.FILL, GridData.FILL, true, false, 1, 1));
        setControl(result);
        Dialog.applyDialogFont(result);
        PlatformUI.getWorkbench().getHelpSystem().setHelp(result, CurlUIIDs.CURL_SEARCH_PAGE_CONTEXT_ID);
    }

    private String[] getPreviousSearchPatterns()
    {
        // Search results are not persistent
        int patternCount = fPreviousSearchPatterns.size();
        String[] patterns = new String[patternCount];
        for (int i = 0; i < patternCount; i++)
            patterns[i] = fPreviousSearchPatterns.get(i).getPattern();
        return patterns;
    }

    @Override
    public void setVisible(
            boolean visible)
    {
        if (visible && fPattern != null) {
            if (fFirstTime) {
                fFirstTime = false;
                // Set item and text here to prevent page from resizing
                fPattern.setItems(getPreviousSearchPatterns());
                initSelections();
            }
            fPattern.setFocus();
        }
        updateOKStatus();
        super.setVisible(visible);
    }

    private void handlePatternSelected()
    {
        int selectionIndex = fPattern.getSelectionIndex();
        if (selectionIndex < 0 || selectionIndex >= fPreviousSearchPatterns.size())
            return;
        SearchPatternData patternData = fPreviousSearchPatterns.get(selectionIndex);
        if (!fPattern.getText().equals(patternData.getTextPattern()))
            return;
        fCaseSensitive.setSelection(patternData.isCaseSensitive());
        setLimitTo(patternData.getLimitToEnum());
        setSearchFor(patternData.getSearchForEnum());
        fPattern.setText(patternData.getTextPattern());
        enableSearchForGroup();
        if (patternData.getWorkingSets() != null)
            getContainer().setSelectedWorkingSets(patternData.getWorkingSets());
        else
            getContainer().setSelectedScope(patternData.getScope());
    }

    private void updateOKStatus()
    {
        boolean isValid = isValidSearchPattern();
        getContainer().setPerformActionEnabled(isValid);
    }

    private boolean isValidSearchPattern()
    {
        if (getPattern().length() == 0) {
            return false;
        }
        return true;
    }

    private String getPattern()
    {
        return fPattern.getText();
    }

    private void initSelections()
    {
        SearchPatternData initData = getDefaultInitValues();
        setLimitTo(initData.getLimitToEnum());
        if (initData.getLimitToEnum() == LimitToEnum.DEFINITIONS) {
            setSearchFor(initData.getSearchForEnum());
        }
        fCaseSensitive.setSelection(initData.isCaseSensitive());
        fPattern.setText(initData.getPattern());
        getContainer().setSelectedScope(initData.getScope());
        IWorkingSet[] workingSets = initData.getWorkingSets();
        if (workingSets != null) {
            getContainer().setSelectedWorkingSets(workingSets);
        }
        enableSearchForGroup();
    }

    private SearchPatternData getDefaultInitValues()
    {
        ISelection selection= fContainer.getSelection();
        if (selection instanceof ITextSelection && !selection.isEmpty()) 
        {
            String text= ((ITextSelection) selection).getText();
            if (text != null) 
            {
                return new SearchPatternData(
                        SearchForEnum.type, // dummy here
                        LimitToEnum.IDENTIFIER_LIKE,
                        text,
                        false,
                        ISearchPageContainer.WORKSPACE_SCOPE, 
                        null); 
            }
        }
        // TODO:the jdt also considers the structured selection (like definitions view) 
        
        if (!fPreviousSearchPatterns.isEmpty()) {
            return fPreviousSearchPatterns.get(0);
        }
        return new SearchPatternData(
                SearchForEnum.type,
                LimitToEnum.DEFINITIONS,
                "*", false, ISearchPageContainer.WORKSPACE_SCOPE, null); //$NON-NLS-1$
    }

    private SearchForEnum getSearchFor()
    {
        for (Button element : fSearchFor) {
            if (element.getSelection()) {
                return SearchForEnum.valueOf((String)element.getData());
            }
        }
        throw new RuntimeException();
    }

    private LimitToEnum getLimitTo()
    {
        for (Button element : fLimitTo) {
            if (element.getSelection()) {
                return LimitToEnum.valueOf((String)element.getData());
            }
        }
        throw new RuntimeException();
    }

    private void setLimitTo(
            LimitToEnum limitToEnum)
    {
        for (Button element : fLimitTo) {
            element.setSelection(LimitToEnum.valueOf((String)element.getData()) == limitToEnum);
        }
    }

    private void setSearchFor(
            SearchForEnum searchForEnum)
    {
        for (Button element : fSearchFor) {
            element.setSelection(SearchForEnum.valueOf((String)element.getData()) == searchForEnum);
        }
    }

    private void enableSearchForGroup()
    {
        boolean enable = getLimitTo() == LimitToEnum.DEFINITIONS;
        for (Button element : fSearchFor) {
            element.setEnabled(enable);
        }
    }

    private Control createLimitTo(
            Composite parent)
    {
        Group result = new Group(parent, SWT.NONE);
        result.setText(SearchMessages.SearchPage_limitTo_label);
        result.setLayout(new GridLayout(1, true));
        fLimitTo = new Button[] {
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_limitTo_definitions,
                        LimitToEnum.DEFINITIONS.toString(),
                        false),
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_limitTo_identifier_like,
                        LimitToEnum.IDENTIFIER_LIKE.toString(),
                        true), };
        SelectionAdapter curlElementInitializer = new SelectionAdapter() {
            @Override
            public void widgetSelected(
                    SelectionEvent event)
            {
                enableSearchForGroup();
            }
        };
        for (Button element : fLimitTo) {
            element.addSelectionListener(curlElementInitializer);
        }
        return result;
    }

    private Control createSearchFor(
            Composite parent)
    {
        Group result = new Group(parent, SWT.NONE);
        result.setText(SearchMessages.SearchPage_searchFor_label);
        result.setLayout(new GridLayout(2, true));
        fSearchFor = new Button[] {
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_searchFor_type,
                        SearchForEnum.type.toString(),
                        true),
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_searchFor_constructor,
                        SearchForEnum.constructor.toString(),
                        false),
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_searchFor_method,
                        SearchForEnum.method.toString(),
                        false),
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_searchFor_field,
                        SearchForEnum.field.toString(),
                        false),
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_searchFor_globalvar,
                        SearchForEnum.globalvar.toString(),
                        false),
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_searchFor_proc,
                        SearchForEnum.proc.toString(),
                        false),
                createButton(
                        result,
                        SWT.RADIO,
                        SearchMessages.SearchPage_searchFor_macro,
                        SearchForEnum.macro.toString(),
                        false)
        };
        // Fill with dummy radio buttons
        Label filler = new Label(result, SWT.NONE);
        filler.setVisible(false);
        filler.setLayoutData(new GridData(SWT.FILL, SWT.FILL, false, false, 1, 1));
        return result;
    }

    private Button createButton(
            Composite parent,
            int style,
            String text,
            String data,
            boolean isSelected)
    {
        Button button = new Button(parent, style);
        button.setText(text);
        button.setData(data);
        button.setLayoutData(new GridData());
        button.setSelection(isSelected);
        return button;
    }

    private Control createExpression(
            Composite parent)
    {
        Composite result = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout(2, false);
        layout.marginWidth = 0;
        layout.marginHeight = 0;
        result.setLayout(layout);
        // Pattern text + info
        Label label = new Label(result, SWT.LEFT);
        label.setText(SearchMessages.SearchPage_expression_label);
        label.setLayoutData(new GridData(GridData.FILL, GridData.FILL, false, false, 2, 1));
        // Pattern combo
        fPattern = new Combo(result, SWT.SINGLE | SWT.BORDER);
        fPattern.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(
                    SelectionEvent e)
            {
                handlePatternSelected();
                updateOKStatus();
            }
        });
        fPattern.addModifyListener(new ModifyListener() {
            public void modifyText(
                    ModifyEvent e)
            {
                updateOKStatus();
            }
        });
        // TextFieldNavigationHandler.install(fPattern);
        GridData data = new GridData(GridData.FILL, GridData.FILL, true, false, 1, 1);
        data.widthHint = convertWidthInCharsToPixels(50);
        fPattern.setLayoutData(data);
        // Ignore case checkbox
        fCaseSensitive = new Button(result, SWT.CHECK);
        fCaseSensitive.setText(SearchMessages.SearchPage_expression_caseSensitive);
//        fCaseSensitive.addSelectionListener(new SelectionAdapter() {
//            public void widgetSelected(
//                    SelectionEvent e)
//            {
//                fIsCaseSensitive = fCaseSensitive.getSelection();
//            }
//        });
        fCaseSensitive.setLayoutData(new GridData(GridData.FILL, GridData.FILL, false, false, 1, 1));
        return result;
    }

    /**
     * Returns the page settings for this Text search page.
     * 
     * @return the page settings to be used
     */
    private IDialogSettings getDialogSettings()
    {
        return CurlPlugin.getDefault().getDialogSettingsSection(PAGE_NAME);
    }

    /**
     * Initializes itself from the stored page settings.
     */
    private void readConfiguration()
    {
        IDialogSettings s = getDialogSettings();
//        fIsCaseSensitive = s.getBoolean(STORE_CASE_SENSITIVE);
        try {
            int historySize = s.getInt(STORE_HISTORY_SIZE);
            for (int i = 0; i < historySize; i++) {
                IDialogSettings histSettings = s.getSection(STORE_HISTORY + i);
                if (histSettings != null) {
                    SearchPatternData data = SearchPatternData.create(histSettings);
                    if (data != null) {
                        fPreviousSearchPatterns.add(data);
                    }
                }
            }
        } catch (NumberFormatException e) {
            // ignore
        }
    }

    /**
     * Stores it current configuration in the dialog store.
     */
    private void writeConfiguration()
    {
        IDialogSettings s = getDialogSettings();
//        s.put(STORE_CASE_SENSITIVE, fIsCaseSensitive);
        int historySize = Math.min(fPreviousSearchPatterns.size(), HISTORY_SIZE);
        s.put(STORE_HISTORY_SIZE, historySize);
        for (int i = 0; i < historySize; i++) {
            IDialogSettings histSettings = s.addNewSection(STORE_HISTORY + i);
            SearchPatternData data = fPreviousSearchPatterns.get(i);
            data.store(histSettings);
        }
    }
}
