/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.search;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.search.ui.ISearchQuery;
import org.eclipse.search.ui.ISearchResult;
import org.eclipse.search.ui.NewSearchUI;

import com.curl.eclipse.core.CurlElement;
import com.curl.eclipse.core.SearchForEnum;
import com.curl.eclipse.remote.ICurlSearchDefinitionQueryResultHandler;
import com.curl.eclipse.remote.SearchResultProxyData;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlPluginImages;
import com.curl.eclipse.util.Messages;

/**
 * @author fmisiak
 * see org.eclipse.jdt.internal.ui.search.JavaSearchQuery
 */
class CurlSearchDefinitionQuery implements ISearchQuery
{
    private CurlSearchResult fResult;

    private final DefinitionQuerySpecification fQuerySpec;

    private IProgressMonitor fProgressMonitor;

    volatile private int fNumberOfScannedFiles;

    private int fNumberOfFilesToScan;

//    private TextSearchResultCollector fCollector;

    private MultiStatus fStatus;

    volatile private boolean fisCurlSearchInProgress;

    private SearchResultProxyData fProxyData;

    CurlSearchDefinitionQuery(
            DefinitionQuerySpecification querySpec)
    {
       fQuerySpec  = querySpec;
    }

    public boolean canRerun()
    {
        return true;
    }

    public boolean canRunInBackground()
    {
        return true;
    }

    public String getLabel()
    {
        return  SearchMessages.CurlSearchQuery_label;
    }

    public ISearchResult getSearchResult()
    {
        if (fResult == null) {
            CurlSearchResult result= new CurlSearchResult(this);
            new SearchResultUpdater(result);
            fResult= result;
        }
        return fResult;
    }

    public IStatus run(
            IProgressMonitor monitor) throws OperationCanceledException
    {
        final CurlSearchResult textResult= (CurlSearchResult) getSearchResult();
        textResult.removeAll();

        fProgressMonitor= monitor == null ? new NullProgressMonitor() : monitor;
        fNumberOfScannedFiles= 0;
        fStatus = new MultiStatus(NewSearchUI.PLUGIN_ID, IStatus.OK, SearchMessages.TextSearchEngine_statusMessage, null);
        IFile[] files = fQuerySpec.getScope().evaluateFilesInScope(fStatus);
        fNumberOfFilesToScan= files.length;

//        fCollector= new TextSearchResultCollector(textResult);

        launchQuery(fQuerySpec.getSearchFor(), fQuerySpec.getRegExpPattern(), fQuerySpec.isCaseSensitive(), files);

        // TODO: we could wait a bit before launching the progress bar, to see if it's really worth it.
        Job monitorUpdateJob= new Job(SearchMessages.TextSearchVisitor_progress_updating_job) {
            private int fLastNumberOfScannedFiles= 0;

            @Override
            public IStatus run(IProgressMonitor inner) {
                while (!inner.isCanceled()) {
                        int steps= fNumberOfScannedFiles - fLastNumberOfScannedFiles;
                        fProgressMonitor.worked(steps);
                        fLastNumberOfScannedFiles += steps;
                        // uncomment this for testing
//                        matchingDefinitionsFound();
                    try {
                        Thread.sleep(100);
                    } catch (InterruptedException e) {
                    }
                }
                return Status.OK_STATUS;
            }
        };

        try {
            String taskName = Messages.format(
                    SearchMessages.TextSearchVisitor_textsearch_task_label,
                    fQuerySpec.getSearchText());
            fProgressMonitor.beginTask(taskName, fNumberOfFilesToScan);
            monitorUpdateJob.setSystem(true);
            monitorUpdateJob.schedule();
            try {
                waitUntilCurlSearchComplete();
                return fStatus;
            } finally {
                monitorUpdateJob.cancel();
            }
        } finally {
            fProgressMonitor.done();
//            fCollector.endReporting();
        }
    }

    private void waitUntilCurlSearchComplete()
    {
        while (!fProgressMonitor.isCanceled() && fisCurlSearchInProgress) {
            try {
                Thread.sleep(100);
            } catch (InterruptedException e) {
            }
        }
        if (fProgressMonitor.isCanceled()) {
            fProxyData.cancelQuery();
            fProxyData.destroy();
        }
    }

    private void launchQuery(
            String searchFor,
            String searchText,
            boolean caseSensitive,
            IFile[] files)
    {
        fisCurlSearchInProgress = true;
        fProxyData = new SearchResultProxyData(new ICurlSearchDefinitionQueryResultHandler() {
            @Override
            public void matchingDefinitionFound(Match searchMatch)
            {
                IFile file = CoreUtil.getFile(searchMatch.fileURL);
                SearchForEnum type = SearchForEnum.valueOf(searchMatch.curlElementType);
                SearchForEnum parentType = searchMatch.curlElementParentType == null ?
                        null : SearchForEnum.valueOf(searchMatch.curlElementParentType);
                CurlElement curlElement = new CurlElement(
                        file,
                        type,
                        searchMatch.nameOfType,
                        searchMatch.symbolName,
                        searchMatch.row,
                        searchMatch.col,
                        searchMatch.parentRow,
                        searchMatch.parentCol,
                        searchMatch.info,
                        searchMatch.infoParent,
                        parentType);
                org.eclipse.search.ui.text.Match match = new CurlMatch(curlElement);
                fResult.addMatch(match);
            }

            @Override
            public void searchCompleted()
            {
                fisCurlSearchInProgress = false;
            }

            @Override
            public void setNumberOfFileScanned(
                    int numberOfFileScanned)
            {
                fNumberOfScannedFiles = numberOfFileScanned;
            }
        });
        fProxyData.queryDefinitions(searchFor, searchText, caseSensitive, files);
    }

//    /**
//     * only needed for testing, simulates response from curl
//     */
//    private void matchingDefinitionsFound()
//    {
//        IFile file = CoreUtil.getFile("file://C:/runtime-New_configuration/MyFirstProject/start.curl");
//        Match match = new Match(file, 1, 10);
//        fResult.addMatch(match);
//
//        file = CoreUtil.getFile("file://C:/runtime-New_configuration/MyFirstProject/start3.curl");
//        match = new Match(file, 1, 10);
//        fResult.addMatch(match);
//
//        // if this is last chunk of results
//        fisCurlSearchInProgress = false;
//    }

    String getResultLabel(
            int nMatches)
    {
        if (nMatches == 1) {
            return Messages.format(
                    SearchMessages.CurlSearchOperation_singularDeclarationsPostfix,
                    getSearchPatternDescription(),
                    fQuerySpec.getScopeDescription());
        } else {
            return Messages.format(
                    SearchMessages.CurlSearchOperation_pluralDeclarationsPostfix,
                    getSearchPatternDescription(),
                    new Integer(nMatches),
                    fQuerySpec.getScopeDescription());
        }
    }

    private String getSearchPatternDescription()
    {
        return fQuerySpec.getSearchText();
    }

    public ImageDescriptor getImageDescriptor()
    {
        return CurlPluginImages.DESC_OBJS_SEARCH_DECL;
    }
}
