/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.search;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.model.WorkbenchLabelProvider;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.core.AccessEnum;
import com.curl.eclipse.core.BindingInfo;
import com.curl.eclipse.core.CurlElement;
import com.curl.eclipse.core.SearchForEnum;
import com.curl.eclipse.ui.CurlElementImageDescriptor;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlPluginImages;

/**
 * @author fmisiak
 *
 * TODO: move to a more general package when reused
 */
class CurLabelProvider extends LabelProvider
{
    private final static ILabelProvider fDecoratingWorkbenchLabelProvider;
    static {
        fDecoratingWorkbenchLabelProvider = WorkbenchLabelProvider.getDecoratingWorkbenchLabelProvider();
    }
    private final boolean fIsTreeView;

    // TODO: probably refactor to use a ImageDescriptorRegistry (see JDT)
    CurLabelProvider(boolean isTreeView)
    {
        
        fIsTreeView = isTreeView;
    }

    private ImageDescriptor createImageDescriptor(String imageURI)
    {
        ImageDescriptor imageDescriptor = CoreUtil.createImageDescriptor(
                CurlPlugin.getDefault().getBundle(),
                CurlPluginImages.ICONS_PATH.append(imageURI),
                true);
        return imageDescriptor;
    }
        
    /* (non-Javadoc)
     * @see org.eclipse.jface.viewers.LabelProvider#getImage(java.lang.Object)
     */
    @Override
    public Image getImage(
            Object element)
    {
        ImageDescriptor imageDescriptor = null;
        
        if (element instanceof IResource) {
            return fDecoratingWorkbenchLabelProvider.getImage(element);
        }
        if (element instanceof CurlElement) {
            CurlElement el = (CurlElement)element;
            if (el.getType() != null) {
                ImageDescriptor baseImageDescriptor = null;
                switch (el.getType()) {
                case constructor: // fall thru
                case method:
                    baseImageDescriptor = getBaseImageForMethodOrConstructor(el.getInfo());
                    break;
                case field:
                    baseImageDescriptor = getBaseImageForField(el.getInfo());
                    break;
                case globalvar:
                    baseImageDescriptor = getBaseImageForGlobalVar(el.getInfo());
                    break;
                case macro:
                    baseImageDescriptor = getBaseImageForMacro(el.getInfo());
                    break;
                case proc:
                    baseImageDescriptor = getBaseImageForProc(el.getInfo());
                    break;
                case classCurl:
                    baseImageDescriptor = getBaseImageForClass(el.getInfo());
                    break;
                case enumCurl:
                    baseImageDescriptor = getBaseImageForEnum(el.getInfo());
                    break;
                case type:
                case all:
                    CoreUtil.logError("unexpected element type 'all'"); //$NON-NLS-1$
                }
                int adornmentFlags= computeCurlAdornmentFlags(el);
                imageDescriptor = new CurlElementImageDescriptor(baseImageDescriptor, adornmentFlags);
            } else {
                // should not happen
                CoreUtil.logWarning("Should not happen:" + this.getClass().getName() + ".getImage(), element=" + element.toString()); //$NON-NLS-1$ //$NON-NLS-2$
            }
            
        }
        if (imageDescriptor == null) {
            imageDescriptor = createImageDescriptor("error.gif"); //FIXME  //$NON-NLS-1$
        }
        return CurlPlugin.getImageDescriptorRegistry().get(imageDescriptor);
    }

    
    private int computeCurlAdornmentFlags(
            CurlElement el)
    {
        int flags= 0;
        if (el.getType() == SearchForEnum.constructor)
            flags |= CurlElementImageDescriptor.CONSTRUCTOR;
            
        BindingInfo info = el.getInfo();
        // see Curl doc BindingInfo
        if (! el.isData() && info.isAbstract()) {
            flags |= CurlElementImageDescriptor.ABSTRACT;
        }
        if (el.isData() && /*info.isData() &&*/ info.isConstant()) {
            flags |= CurlElementImageDescriptor.CONSTANT;
        }
        if (info.isDeprecated()) {
            flags |= CurlElementImageDescriptor.DEPRECATED;
        }
        if (!el.isData() && info.isFinal()) {
            flags |= CurlElementImageDescriptor.FINAL;
        }
        if (!el.isData() && info.isImplicit()) {
            flags |= CurlElementImageDescriptor.IMPLICIT;
        }
        if (!el.isData() && info.isInline()) {
            flags |= CurlElementImageDescriptor.INLINE;
        }
        if (! el.isData() && info.isSealed()) {
            flags |= CurlElementImageDescriptor.SEALED;
        }
        if (el.isData() && info.isWeak()) {
            flags |= CurlElementImageDescriptor.WEAK;
        }
        return flags;
    }

    private ImageDescriptor getBaseImageForProc(
            BindingInfo info)
    {
        AccessEnum access = info.access;
        switch(access) {
        case accessHidden:
            return CurlPluginImages.DESC_PROC_HIDDEN;
        case accessPackage:
            return CurlPluginImages.DESC_PROC_PACKAGE;
        case accessProtected:
            // not possible in Curl
            CoreUtil.logError("unexpected accessPackage for a proc icon"); //$NON-NLS-1$
            return null;
        case accessPrivate:
            return CurlPluginImages.DESC_PROC_PRIVATE;
        case accessPublic:
            return CurlPluginImages.DESC_PROC_PUBLIC;
        }
        return null;
    }

    private ImageDescriptor getBaseImageForMacro(
            BindingInfo info)
    {
        AccessEnum access = info.access;
        switch(access) {
        case accessHidden:
            return CurlPluginImages.DESC_MACRO_HIDDEN;
        case accessPackage:
            return CurlPluginImages.DESC_MACRO_PACKAGE;
        case accessPrivate: // fall thru
        case accessProtected:
            // not possible in Curl
            CoreUtil.logError("unexpected access for a macro icon"); //$NON-NLS-1$
            return null;
        case accessPublic:
            return CurlPluginImages.DESC_MACRO_PUBLIC;
        }
        return null;
    }

    private ImageDescriptor getBaseImageForGlobalVar(
            BindingInfo info)
    {
        AccessEnum access = info.access;
        switch(access) {
        case accessHidden:
            return CurlPluginImages.DESC_FIELD_HIDDEN;
        case accessPackage:
            return CurlPluginImages.DESC_FIELD_PACKAGE;
        case accessPrivate: // fall thru
        case accessProtected:
            // not possible in Curl
            return null;
        case accessPublic:
            return CurlPluginImages.DESC_FIELD_PUBLIC;
        }
        return null;
    }

    private ImageDescriptor getBaseImageForClass(
            BindingInfo info)
    {
        AccessEnum access = info.access;
        switch(access) {
        case accessHidden:
            return CurlPluginImages.DESC_CLASS_HIDDEN;
        case accessPackage:
            return CurlPluginImages.DESC_CLASS_PACKAGE;
        case accessPrivate:
            // not possible in Curl
            return null;
        case accessProtected:
            return CurlPluginImages.DESC_CLASS_PROTECTED;
        case accessPublic:
            return CurlPluginImages.DESC_CLASS_PUBLIC;
        }
        return null;
    }
    
    private ImageDescriptor getBaseImageForEnum(
            BindingInfo info)
    {
        AccessEnum access = info.access;
        switch(access) {
        case accessHidden:
            return CurlPluginImages.DESC_ENUM_HIDDEN;
        case accessPackage:
            return CurlPluginImages.DESC_ENUM_PACKAGE;
        case accessPublic:
            return CurlPluginImages.DESC_ENUM_PUBLIC;
        case accessPrivate: // fall thru
        case accessProtected:
            return  null;
        }
        return null;
    }
    
    private ImageDescriptor getBaseImageForMethodOrConstructor(
            BindingInfo info)
    {
        AccessEnum access = info.access;
        switch(access) {
        case accessHidden:
            return CurlPluginImages.DESC_METH_HIDDEN;
        case accessPackage:
            return CurlPluginImages.DESC_METH_PACKAGE;
        case accessPrivate:
            return CurlPluginImages.DESC_METH_PRIVATE;
        case accessProtected:
            return CurlPluginImages.DESC_METH_PROTECTED;
        case accessPublic:
            return CurlPluginImages.DESC_METH_PUBLIC;
        }
        return null;
    }

    private ImageDescriptor getBaseImageForField(
            BindingInfo info)
    {
        // for fields use whatever scope is the most visible
        // TODO: might want to come up with some visual icons 
        // for getter/setter visibility
        AccessEnum access = info.writeAccess.ordinal() < info.access.ordinal() ?
                info.writeAccess : info.access;
        switch(access) {
        case accessHidden:
            return CurlPluginImages.DESC_FIELD_HIDDEN;
        case accessPackage:
            return CurlPluginImages.DESC_FIELD_PACKAGE;
        case accessPrivate:
            return CurlPluginImages.DESC_FIELD_PRIVATE;
        case accessProtected:
            return CurlPluginImages.DESC_FIELD_PROTECTED;
        case accessPublic:
            return CurlPluginImages.DESC_FIELD_PUBLIC;
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.eclipse.jface.viewers.LabelProvider#getText(java.lang.Object)
     */
    @Override
    public String getText(
            Object element)
    {
        if (fIsTreeView) {
            if (element instanceof IProject) {
                IProject project = (IProject)element;
                return project.getName();
            }
            if (element instanceof IFolder) {
                IFolder folder = (IFolder)element;
                return folder.getName();
            }
            if (element instanceof IFile) {
                IFile file = (IFile)element;
                return file.getName();
            }
            if (element instanceof CurlElement) {
                CurlElement el = (CurlElement)element;
                if (el.getType() != null) {
                    return el.getSymbolName();
                } else {
                    return el.getFile().getName();
                }
            }
        } else {
            if (element instanceof CurlElement) {
                CurlElement el = (CurlElement)element;
                return getSymbolname(el);
            }
        }
    
        return super.getText(element);
    }

    private String getSymbolname(
            CurlElement el)
    {
        StringBuilder sb = new StringBuilder();
        IFile file = el.getFile();
        String fn = file.getFullPath().toString();
        sb.append(el.getSymbolName())
            .append(" - ") //$NON-NLS-1$
            .append(fn);
        switch(el.getType())
        {
        case constructor: 
        case method: // fall through
        case field: // fall through
            sb.append(" - ") //$NON-NLS-1$
            .append(el.getNameOfType());
            break;
            
        case classCurl:// fall through
        case enumCurl:// fall through
        case globalvar:// fall through
        case macro:// fall through
        case  proc:// fall through
            // nothing to append in these cases
            break;
        case type:
        case all:
            CoreUtil.logError("unexpect type 'all'"); //$NON-NLS-1$
        }
        return sb.toString();
    }
}
