/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.INullSelectionListener;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.curl.eclipse.editors.CurlContentOutlinePage;
import com.curl.eclipse.editors.CurlEditor;
import com.curl.eclipse.editors.CurlView;
import com.curl.eclipse.util.CoreUtil;

/**
 * TODO
 */
public class WorkbenchWindowProxy extends ProxyData
    implements INotifyCommandChangedProxy
{
    private static final String COM_CURL_ECLIPSE = "com.curl.eclipse."; //$NON-NLS-1$

    /**
     * Listens to any selection made in the workbench for any Curl specific
     * work.
     */
    private class WorkbenchWindowSelectionListener implements ISelectionListener, INullSelectionListener
    {
        // TODO: provide rational for "synchronized", fmisiak's note: I don't see any!
        // also note that this run's in the UI thread.
        synchronized public void selectionChanged(
                IWorkbenchPart part,
                ISelection selection)
        {            
            WorkbenchWindowProxy.this.selectionChanged(part, selection);
            /*
             * Notify mediator that focus has been given to a view other than a
             * Curl view.
             */
            if (part instanceof CurlEditor ||
                 part instanceof CurlContentOutlinePage ||
                 part instanceof CurlView) {
                
                // It is us!!
                // Do not send a "focus to some eclipse view" command. Classes
                // above send a set focus command.
                return;
            }
            // TODO: should fix this so that we don't send redundant
            // setFocusToSomeEclipseView messages.
            // FIXME: should only do that if WorkbenchWindowProxy.this.selectionChanged() really send a new 
            // selection to Mediator.
            setFocusToSomeEclipseView();
        }
    }
    
    // 200 is the start value for these operations. We don't
    // want to collide with any Eclipse operation numbers.
    private static final int CURL_WORKBENCH_OPERATIONS = 200;

    /**
     * Note that the name of each enum member is the suffix of a corresponding
     * action id defined in the plugin.xml.
     * Maintainer, please be careful with spelling and casing.
     * Also note that SourceViewProxy uses a similar mechanism to update
     * menu items, TODO: refactor with SourceViewProxy.
     * 
     * If we mark something modal here, we must add the corresponding entry 
     * in WorkbenchWindowAgent.modal-operations.
     */
    public enum OPERATIONS
    {
        project_manifest_addfile(0, ModalMode.modal),
        project_manifest_adddirectory(1, ModalMode.modal),
        project_manifest_addpackage(2, ModalMode.modal),
        project_manifest_adddelegateto(3, ModalMode.modal),
        project_manifest_removeselected(4),
//        project_manifest_changeAPIVersion(5, ModalMode.modal), (enablement managed using <visibleWhen>
        project_editcomponentsettings(6, ModalMode.modal),
        project_showhidetargetsettings(7),
        project_deploy(8, ModalMode.modal),
        project_deployforocc(9, ModalMode.modal),
        tools_OpenClassBrowser(10),
        tools_OpenCodeCoverage(11),
        tools_OpenConfigureEditors(12, ModalMode.modal),
        tools_OpenHttpMonitor(13),
        tools_OpenProfiler(14),
        tools_OpenVLE(15),
        tools_OpenVLEEE(16),
        ;

        private enum ModalMode {
            modal, modeless;
        }

        /** instead of using ordinal(), use explicit opcode since Curl mediator will duplicate these definitions */
        private final Integer fOperationCode;
        
        private final ModalMode fModalMode;
        
        OPERATIONS(int number) {
            this(number, ModalMode.modeless);
        }
        
        OPERATIONS(int number, ModalMode modalMode) {
            fOperationCode = Integer.valueOf(number + CURL_WORKBENCH_OPERATIONS);
            fModalMode = modalMode;
        }
        
        Integer getOperationCode()
        {
            return fOperationCode;
        }

        public boolean isModal()
        {
            return fModalMode == ModalMode.modal;
        }
    }
    
    private final WorkbenchWindowSelectionListener fWorkbenchWindowSelectionListener;
    // We cache the current selection to avoid pointless notifications.
    private String fCurrentSelectionUrlString;
    private boolean fCurrentPartIsEditor = false;
    private ProjectTargets fCurrentProjectTargets;
    private final HashMap<Integer, Boolean> fOperationStatus = new HashMap<Integer, Boolean>();
    private final Map<Integer, IAction> fActions = new HashMap<Integer, IAction>();
    private final List<IActionRegisterListener> fActionRegisterListeners = new ArrayList<IActionRegisterListener>();

    public WorkbenchWindowProxy(
            WorkbenchProxy workbenchProxy, 
            IWorkbenchWindow workbenchWindow)
    {
        super(workbenchWindow);
        for (OPERATIONS operation : OPERATIONS.values()) {
            if (fOperationStatus.put(operation.getOperationCode(), false) != null) {
                CoreUtil.logError("Sanity check, duplicate operation code=" + operation); //$NON-NLS-1$
            }
        }
        workbenchProxy.createWorkbenchWindowProxy(
                fProxyID,
                workbenchWindow);

        fWorkbenchWindowSelectionListener = new WorkbenchWindowSelectionListener();
        workbenchWindow.getSelectionService().addSelectionListener(
                fWorkbenchWindowSelectionListener);
    }
    
    @Override
    public IWorkbenchWindow getProxyOf()
    {
        return (IWorkbenchWindow) super.getProxyOf();
    }
    

    public void registerAction(
            IAction action)
    {
        OPERATIONS operation = fromAction(action);
        if (operation != null) {
            if (fActions.containsKey(operation.getOperationCode())) {
                // the customize perspective dialog may instantiate our
                // actionSets actions later, we must ignore these
                // dummy instances.
                return;
            }
            fActions.put(operation.getOperationCode(), action);
            updateOperation(operation.getOperationCode());
            for(IActionRegisterListener listener : fActionRegisterListeners) {
                listener.actionRegistered(action, operation);
            }
        }
    }
    
    private OPERATIONS fromAction(
            IAction action)
    {
        String actionDefinitionId = action.getActionDefinitionId();
        if (actionDefinitionId.startsWith(COM_CURL_ECLIPSE)) {
            String enumName = actionDefinitionId.substring(COM_CURL_ECLIPSE.length());
            return OPERATIONS.valueOf(enumName);
        } else {
            CoreUtil.logError("OPERATIONS not found for :" + actionDefinitionId); //$NON-NLS-1$
            return null;
        }
    }

    public void doOperation(
            IAction action)
    {
        OPERATIONS operation = fromAction(action);
        doOperation(operation);
    }
    
    /**
     * Notify the mediator that the selection has changed.
     * As a result of sending the new workspace selection,
     * we expect the mediator to call us back with the set
     * of related targets, see {@link #projectTargetsChanged()}
     * @param part
     * @param selection
     */
    public void selectionChanged(
            IWorkbenchPart part,
            ISelection selection)
    {
        URI uri = CoreUtil.getSelectedResourceURI(part, selection);
        boolean partIsEditor = part instanceof IEditorPart;

        if ((uri != null && (!uri.toString().equals(fCurrentSelectionUrlString) || partIsEditor != fCurrentPartIsEditor))
            || (uri == null && fCurrentSelectionUrlString != null)) 
        {
            fCurrentSelectionUrlString = uri == null ? "" : uri.toString(); //$NON-NLS-1$
            fCurrentPartIsEditor = partIsEditor;
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(
                    MediatorConnection.commandWorkbenchWindowSelectionChanged) {
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(fCurrentSelectionUrlString);
                    write(fCurrentPartIsEditor);
                }
            });
        }
    }
    
    private void updateOperation(
            final int opcode)
    {
        final IAction action = fActions.get(opcode);
        if (action !=  null) {
            final boolean enabled = canDoOperation(opcode);
            PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
                public void run()
                {
                    // This operation must be performed in the UI thread.
                    action.setEnabled(enabled);
                }
            });
        } else {
            // Action didn't register yet, that's ok see #registerAction()
        }
    }
    
    public void doOperation(
            OPERATIONS operation)
    {
        Integer operationCode = operation.getOperationCode();
        if (fOperationStatus.containsKey(operationCode) && fOperationStatus.get(operationCode)) {
            // Send operation to mediator to perform the operation.
            if (operation.isModal()) {
                sendDoOperationModal(operationCode);
            } else {
                sendDoOperation(operationCode);
            }
        }
    }

    public boolean canDoOperation(
            int opcode)
    {
        if (fOperationStatus.containsKey(opcode))
            return fOperationStatus.get(opcode);
        else {
            CoreUtil.logError("Missing action status for operation=" + opcode); //$NON-NLS-1$
            return false;
        }
    }

    @Override
    public void setOperationStatus(
            int opcode,
            boolean enabled,
            boolean value,
            String label)
    {
        // TODO: put "value" and "label" in fOperationStatus too.
        fOperationStatus.put(opcode, enabled);
        updateOperation(opcode);
    }

   /**
     * Notify the mediator to set focus to some eclipse view
     */
    public void setFocusToSomeEclipseView()
    {
        fMediatorConnection.async.execute(
                new AsynchronousRemoteOperation(MediatorConnection.commandSetFocusSomeEclipseView) {
                    @Override
                    protected void writeArguments()
                    {
                        write(fProxyID);
                    }
                });
    }

    public void windowActivated()
    {
        fMediatorConnection.async.execute(
                new AsynchronousRemoteOperation(MediatorConnection.commandWindowActivated) {
                    @Override
                    protected void writeArguments()
                    {
                        write(fProxyID);
                    }
                });
        
        setFocusToSomeEclipseView();
    }
    
    public void windowDeactivated()
    {
        fMediatorConnection.async.execute(
                new AsynchronousRemoteOperation(MediatorConnection.commandWindowDeactivated) {
                    @Override
                    protected void writeArguments()
                    {
                        write(fProxyID);
                    }
                });
    }

    @Override
    public void destroy()
    {
        getProxyOf().getSelectionService().removeSelectionListener(fWorkbenchWindowSelectionListener);
        super.destroy();
    }

    public static class ProjectTargets
    {
        final String current;
        final String[] names;

        ProjectTargets(
                String current,
                String[] names)
        {
            super();
            this.current = current;
            this.names = names;
        }

        public String getCurrent()
        {
            return current;
        }

        public String[] getNames()
        {
            return names;
        }
    }

    private static class ProjectTargetsChangedCommand
    {
        public String[] names;
        public String current;
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandProjectTargetsChanged, new ProjectTargetsChangedCommandHandler());
    }
    public static class ProjectTargetsChangedCommandHandler extends ProxyCommandHandler<WorkbenchWindowProxy, ProjectTargetsChangedCommand>
    {
        @Override
        ProjectTargetsChangedCommand decode(
                DataInputStream sockIn)
        {
            ProjectTargetsChangedCommand command = new ProjectTargetsChangedCommand();
            int nb = DataIO.readInt(sockIn);
            command.names = new String[nb];
            for (int i = 0; i < nb; i++) {
                command.names[i] = DataIO.readString(sockIn);
            }
            command.current = DataIO.readString(sockIn);
            return command;
        }

        @Override
        void execute(
                final ProjectTargetsChangedCommand command,
                final WorkbenchWindowProxy proxy,
                DataOutputStream sockOut)
        {
            proxy.fCurrentProjectTargets = new ProjectTargets(command.current, command.names);
        }
    }

    

    public void setCurrentTarget(
            final String currentTargetName)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(
                MediatorConnection.commandSetCurrentTarget) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(currentTargetName);
            }
        });
    }

    public ProjectTargets getProjectTargets()
    {
        return fCurrentProjectTargets;
    }

    public void addActionRegisterListener(
            IActionRegisterListener actionRegisterListener)
    {
        /*
         * In case actions were already registered before this method is called
         */
        for(Entry<Integer, IAction> entry : fActions.entrySet()) {
            actionRegisterListener.actionRegistered(entry.getValue(), fromAction(entry.getValue()));
        }

        /*
         * anyway, the listener needs to know about any action dynamically registered
         */
        fActionRegisterListeners.add(actionRegisterListener);
    }

    public void removeActionRegisterListener(
            IActionRegisterListener actionListener)
    {
        fActionRegisterListeners.remove(actionListener);
    }

}
