/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IWorkbenchWindow;

import com.curl.eclipse.debug.CurlBreakpointDescriptor;
import com.curl.eclipse.debug.CurlExceptionBreakpointDescriptor;
import com.curl.eclipse.debug.CurlLineBreakpointDescriptor;
import com.curl.eclipse.editors.CurlEditor;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.WindowHandle;
import com.curl.eclipse.util.WorkbenchOperations;
import com.ibm.icu.text.MessageFormat;

/**
 * The WorkbenchProxy is a connection to the mediator for sending and receiving commands that are global to
 * the Eclipse workbench.
 */
public class WorkbenchProxy extends ProxyData
{

    /** potentially concurrent access to this map */
    private final Hashtable<URI, IRemoteSourceQueryMonitor> fRemoteSourceMonitorMap = new Hashtable<URI, IRemoteSourceQueryMonitor>();


    public WorkbenchOperations getWorkbenchOperations()
    {
        return (WorkbenchOperations)getProxyOf();
    }

    public WorkbenchProxy(
            WorkbenchOperations proxyOf)
    {
        super(proxyOf);
        createRemoteAgent();
    }

    /**
     * Process list of tool descriptions from mediator,
     * and create menu items in "Tools" menu (id "com.curl.eclipse.CurlToolsMenu")
     */
    public static class ToolEntry
    {
        public final String fCurlEditorID;
        public final String fName;

        public ToolEntry(
                String editorID,
                String name)
        {
            fCurlEditorID = editorID;
            fName = name;
        }
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandSetTools, new SetToolsCommandHandler());
    }
    public static class SetToolsCommandHandler extends ProxyCommandHandler<WorkbenchProxy, List<ToolEntry>>
    {
        @Override
        List<ToolEntry> decode(
                DataInputStream sockIn)
                {
            List<ToolEntry> list = new ArrayList<ToolEntry>();
            int nb = DataIO.readInt(sockIn);
            for (int i = 0; i < nb; i++) {
                list.add(new ToolEntry(
                        DataIO.readString(sockIn),
                        DataIO.readString(sockIn)));
            }
            return list;
                }

        @Override
        void execute(
                final List<ToolEntry> toolsSet,
                final WorkbenchProxy proxy,
                DataOutputStream sockOut)
        {
            proxy.getWorkbenchOperations().setTools(toolsSet);
        }
    }
    
    static
    {
        EclipseServer.register(MediatorConnection.commandSavedURL, new SavedURLCommandHandler());
    }
    public static class SavedURLCommandHandler extends ProxyCommandHandler<WorkbenchProxy, String>
    {
        @Override
        String decode(
                DataInputStream sockIn)
        {
            String savedURLString = DataIO.readString(sockIn);
            return savedURLString;
        }

        @Override
        void execute(
                final String savedURLString,
                final WorkbenchProxy proxy,
                DataOutputStream sockOut)
        {
            proxy.getWorkbenchOperations().savedURLResponse(savedURLString);
        }
    }

    /**
     * Close this project
     */
    public void closeProject(
            final String project)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandCloseProjectAction){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(project);
                write(false);  // Not closing to delete
            }
        });
    }

    /**
     * Close the project in order to later delete it.
     */
    public void closeToDelete(
            final String project)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandCloseProjectAction){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(project);
                write(true);  // Closing to delete
            }
        });
    }

    /**
     * Display documentation help for given symbol
     * in Curl Documentation Viewer
     * 
     * @param symbol
     */
    public void showDocumentationLanguage(
            final String symbol)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandShowDocumentationLanguage){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(symbol);
            }
        });
    }
    
    /**
     * Open Curl Class Browser 
     * on given symbol
     * 
     * @param symbol
     */
    public void showInClassBrowser(
            final String symbol)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandShowInClassBrowser){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(symbol);
            }
        });
    }
    
    /**
     * Open this project
     */
    public void openProject(
            final URL curlProjectURL,
            final String projectName)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandOpenProjectAction) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(curlProjectURL.toString());
                write(projectName);
            }
        });
    }

    public void launchToolForId(
            final String id,
            final URL fileURL)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandLaunchToolForId) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(id);
                write(fileURL == null ? "" : fileURL.toString()); //$NON-NLS-1$
            }
        });
    }

    public void launchAppletForDebugging(
            final String applet)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandLaunchAppletForDebugging){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(applet);
            }
        });
    }

    public void addBreakpoint(
            final CurlBreakpointDescriptor bpDesc)
    {
        if (bpDesc instanceof CurlLineBreakpointDescriptor) {
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandAddLineBreakpoint){
                @Override
                protected void writeArguments()
                {
                    CurlLineBreakpointDescriptor bplDesc = (CurlLineBreakpointDescriptor)bpDesc;
                    String fileName = ""; //$NON-NLS-1$
                    try {
                        fileName = bplDesc.getLocationURI().toURL().toString();
                    } catch (MalformedURLException e) {
                        CoreUtil.logError("Malformed URL", e); //$NON-NLS-1$
                    }
                    write(fProxyID);
                    write(bpDesc.isEnabled());
                    write(fileName);
                    write(bplDesc.fLineNumber);
                    write(bplDesc.fCondition);

                }
            });
        } else {
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandAddExceptionBreakpoint){
                @Override
                protected void writeArguments()
                {
                    CurlExceptionBreakpointDescriptor bpeDesc = (CurlExceptionBreakpointDescriptor)bpDesc;
                    write(fProxyID);
                    write(bpDesc.isEnabled());
                    write(bpeDesc.fExceptionTypeName);
                }
            });
        }
    }


    public void removeBreakpoint(
            CurlBreakpointDescriptor bpDesc)
    {
        if (bpDesc instanceof CurlLineBreakpointDescriptor) {
            final CurlLineBreakpointDescriptor bplDesc = (CurlLineBreakpointDescriptor)bpDesc;
            URI uri = bplDesc.getLocationURI();
            if (uri == null) {
                // The project could be closed by now!
                // We can get called when the project is being closed or deleted.
                CoreUtil.logInfo("The resource for which we are to remove breakpoint is not accessible!"); //$NON-NLS-1$
                return;
            }
            final String fileName;
            try {
                fileName = uri.toURL().toString();
            } catch (MalformedURLException e) {
                CoreUtil.logError("Malformed URL", e); //$NON-NLS-1$
                return;
            }
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandRemoveLineBreakpoint){
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(fileName);
                    write(bplDesc.fLineNumber);
                }
            });
        } else {
            final CurlExceptionBreakpointDescriptor bpeDesc = (CurlExceptionBreakpointDescriptor)bpDesc;
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandRemoveExceptionBreakpoint){
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(bpeDesc.fExceptionTypeName);
                }
            });
        }
    }

    public void updateBreakpointEnableState(
            final CurlBreakpointDescriptor bpDesc)
    {
        if (bpDesc instanceof CurlLineBreakpointDescriptor) {
            final CurlLineBreakpointDescriptor bplDesc = (CurlLineBreakpointDescriptor)bpDesc;
            final String fileName;
            fileName = bplDesc.getLocationURI().toString();
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandUpdateLineBreakpointEnableState){
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(bpDesc.isEnabled());
                    write(fileName);
                    write(bplDesc.fLineNumber);
                }
            });
        }
        else {
            final CurlExceptionBreakpointDescriptor bpeDesc = (CurlExceptionBreakpointDescriptor)bpDesc;
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandUpdateExceptionBreakpointEnableState){
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(bpDesc.isEnabled());
                    write(bpeDesc.fExceptionTypeName);
                }
            });
        }
    }

    public String[] getStandardExceptionBreakpointNames()
    {
        final String[][] res = new String[1][];
        fMediatorConnection.sync.executeInModalContext(new SynchronousRemoteOperation(MediatorConnection.commandGetStandardExceptionBreakpointNames){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
            }
            @Override
            protected void receiveResponse()
            {
                int size = readInt();
                String[] exceptionTypeNames = new String[size];
                for (int i = 0; i < size; i++) {
                    exceptionTypeNames[i] = readString();
                }
                res[0] = exceptionTypeNames;
            }
        });
        return res[0];
    }

    public List<URI> queryCurlStartFiles(
            final URL curlProjectURL)
    {
        final List<URI> startFiles = new ArrayList<URI>();
        fMediatorConnection.sync.executeInModalContext(new SynchronousRemoteOperation(MediatorConnection.commandGetCurlStartFiles){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(curlProjectURL.toExternalForm());
            }

            @Override
            protected void receiveResponse()
            {
                int nbItems = readInt();
                for (int i = 0; i < nbItems; i++) {
                    startFiles.add(URI.create(readString()));
                }
            }
        });
        return startFiles;
    }

    public boolean isFileInProject(
            final URL curlProjectURL,
            final String curlFileURL)
    {
        final boolean[] result = new boolean[1];
        fMediatorConnection.sync.executeInModalContext(new SynchronousRemoteOperation(MediatorConnection.commandIsFileInProject){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(curlProjectURL.toExternalForm());
                write(curlFileURL);
            }

            @Override
            protected void receiveResponse()
            {
                result[0] = readBoolean();
            }
        });
        return result[0];
    }

    public boolean canRun(
            URL curlProjectURL,
            final String curlFileURL)
    {
        final boolean[] result = new boolean[1];
        fMediatorConnection.sync.executeInModalContext(new SynchronousRemoteOperation(MediatorConnection.commandCanRun){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(curlFileURL);
            }

            @Override
            protected void receiveResponse()
            {
                result[0] = readBoolean();
            }
        });
        return result[0];
    }

    /**
     *
     */
    public static class ProjectVersionInfo
    {
        protected String currentVersion;
        protected ArrayList<String> availableVersions;
        protected String curlVersionNumber2;
        protected String runtimeVleComponentURLString;
        
        public String getCurrentVersion()
        {
            return currentVersion;
        }
        
        /** see curl documentation guide for 'curl-version-number-2' */
        public String getCurlVersionNumber2()
        {
            return curlVersionNumber2;
        }
        
        public List<String> getAvailableVersions()
        {
            return availableVersions;
        }
        public String getRuntimeVleComponentURLString()
        {
            return runtimeVleComponentURLString;
        }
        
    }
    
    /**
     * @param curlProjectURL
     * @return
     */
    public ProjectVersionInfo queryProjectVersionInfo(
            final URL curlProjectURL)
    {
        final ProjectVersionInfo[] res = new ProjectVersionInfo[1];
        fMediatorConnection.sync.executeInModalContext(new SynchronousRemoteOperation(MediatorConnection.commandGetProjectVersionInfo){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(curlProjectURL.toExternalForm());
            }

            @Override
            protected void receiveResponse()
            {
                ProjectVersionInfo info = new ProjectVersionInfo();
                info.currentVersion = readString();
                info.curlVersionNumber2 = readString();
                info.runtimeVleComponentURLString = readString();
                info.availableVersions = new ArrayList<String>();
                int nbItems = readInt();
                for (int i = 0; i < nbItems; i++) {
                    info.availableVersions.add(readString());
                }
                res[0] = info;
            }
        });
        return res[0];
    }

    /**
     * @param curlProjectURL
     * @return
     */
    public int updateProjectVersionInfo(
            final URL curlProjectURL,
            final String newAPIVersion)
    {
        final int[] res = new int[1];
        fMediatorConnection.sync.executeInModalContext(new SynchronousRemoteOperation(MediatorConnection.commandSetProjectVersionInfo){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(curlProjectURL.toExternalForm());
                write(newAPIVersion);
            }

            @Override
            protected void receiveResponse()
            {
                res[0] = readInt();
            }
        });
        return res[0];
    }
    
    public void createRemoteAgent()
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(
                MediatorConnection.commandCreateWorkbenchAgent) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
            }
        });
    }

    public SourceViewProxy createSourceView(
            CurlEditor curlEditor,
            final DocumentProxy curlDocumentProxy,
            final Composite editorPaneContent,
            IProgressMonitor progressMonitor)
    {
        final SourceViewProxy sourceViewProxy = new SourceViewProxy(
                curlEditor,
                editorPaneContent,
                progressMonitor);
        fMediatorConnection.async.execute(
                sourceViewProxy.new CreateViewRemoteOperation(
                        MediatorConnection.commandMakeEditorView,
                        editorPaneContent) {
                    @Override
                    protected void writeViewArguments()
                    {
                        write(curlDocumentProxy.getProxyID());
                    };
                });
        sourceViewProxy.setViewExists(true);
        return sourceViewProxy;
    }

    public void shutdown()
    {
        asynchronousNoArgsOperation(MediatorConnection.commandDestroyWorkbenchAgent);
    }


    public void createDocumentAgent(
            final int documentAgentID,
            final String url,
            final String content)
    {
        fMediatorConnection.async.execute(
                new AsynchronousRemoteOperation(
                        MediatorConnection.commandCreateDocumentAgent) {

                    @Override
                    protected void writeArguments()
                    {
                        write(fProxyID);
                        write(documentAgentID);
                        write(url);
                        write(content);
                    };
                });
    }
    
    public void queryRemoteSource(final URI uri, IRemoteSourceQueryMonitor remoteSourceQueryMonitor)
    {
        fRemoteSourceMonitorMap.put(uri, remoteSourceQueryMonitor);
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(
                MediatorConnection.commandQueryRemoteSource) {

            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(uri.toString());
            }
        });
    }

    /**
     * Respond to mediator command that openning a project has failed.
     */
    static
    {
        EclipseServer.register(MediatorConnection.commandOpenProjectFailed, new OpenProjectFailedCommandHandler());
    }

    public static class OpenProjectFailedProxyCommand
    {
        public String projectName;
        public String exceptionMessage;
    }

    public static class OpenProjectFailedCommandHandler 
        extends ProxyCommandHandler<WorkbenchProxy, OpenProjectFailedProxyCommand>
    {        
        @Override
        OpenProjectFailedProxyCommand decode(
                DataInputStream sockIn)
        {
            OpenProjectFailedProxyCommand command = new OpenProjectFailedProxyCommand();
            command.projectName = DataIO.readString(sockIn);
            command.exceptionMessage = DataIO.readString(sockIn);
            return command;
        }
        
        @Override
        void execute(
                final OpenProjectFailedProxyCommand command,
                final WorkbenchProxy proxy,
                DataOutputStream sockOut)
        {
            try {
                IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(command.projectName);                
                CoreUtil.displayError(
                        RemoteMessages.OpenCurlProjectFailed,
                        MessageFormat.format(
                                RemoteMessages.OpenCurlProjectFailedMessage,
                                new String[] { command.exceptionMessage }));
                project.close(new NullProgressMonitor());
            } catch (Exception e) {
                CoreUtil.logError("Failed to report open project failure for project: " + command.projectName, e); //$NON-NLS-1$
            }
        }
    }
    
    private static class QueryRemoteSourceResponseCommand
    {

        private String fContent;
        private URI fURI;

        public URI getURI()
        {
            return fURI;
        }

        public String getContent()
        {
            return fContent;
        }

        public void setURI(URI aURI)
        {
            fURI = aURI;
        }

        public void setContent(String content)
        {
            fContent = content;
        }

        private QueryRemoteSourceResponseCommand()
        {
        }

        QueryRemoteSourceResponseCommand(QueryRemoteSourceResponseCommand queryremotesourceresponsecommand)
        {
            this();
        }
    }

    static {
        EclipseServer.register(MediatorConnection.commandQueryRemoteSourceResponse, new QueryRemoteSourceResponseCommandHandler());
    }

    public static class QueryRemoteSourceResponseCommandHandler extends ProxyCommandHandler<WorkbenchProxy, QueryRemoteSourceResponseCommand>
    {

        @Override
        QueryRemoteSourceResponseCommand decode(DataInputStream sockIn)
        {
            QueryRemoteSourceResponseCommand command = new QueryRemoteSourceResponseCommand(null);
            command.setURI(URI.create(DataIO.readString(sockIn)));
            command.setContent(DataIO.readString(sockIn));
            return command;
        }

        @Override
        void execute(QueryRemoteSourceResponseCommand command, WorkbenchProxy proxy, DataOutputStream sockOut)
        {
            // no longer need to track monitor after processing this response.
            IRemoteSourceQueryMonitor monitor = proxy.fRemoteSourceMonitorMap.remove(command.getURI());
            if (monitor != null) {
                monitor.contentReceived(command.getContent());
            }
            else {
                CoreUtil.logError("Unexpected extra query remote source response, content length=" + command.getContent().length()); //$NON-NLS-1$
            }
        }
    }

    public void createWorkbenchWindowProxy(
            final int windowWorkbenchAgentId,
            final IWorkbenchWindow workbenchWindow)
    {
        fMediatorConnection.async.execute(
                new AsynchronousRemoteOperation(
                        MediatorConnection.commandCreateWorkbenchWindowAgent) {

                    @Override
                    protected void writeArguments()
                    {
                        write(fProxyID);
                        write(windowWorkbenchAgentId);
                        WindowHandle handle = new WindowHandle(workbenchWindow.getShell());
                        write(handle.getHandle());
                    };
                });    
    }

    public void breakpointManagerEnablementChanged(
            final boolean enabled)
    {
        fMediatorConnection.async.execute(
                new AsynchronousRemoteOperation(
                        MediatorConnection.commandNotifySkipAllBreakpoints) {

                    @Override
                    protected void writeArguments()
                    {
                        write(fProxyID);
                        write(!enabled);
                    };
                });    
    }
    
    static
    {
        EclipseServer.register(MediatorConnection.commandRaiseActiveWindow, new RaiseWindowEventHandler());
    }
    public static class RaiseWindowEventHandler extends NoArgsProxyCommandHandler<WorkbenchProxy>
    {
        @Override
        void execute(
                final WorkbenchProxy proxy,
                DataOutputStream sockOut)
        {
            proxy.getWorkbenchOperations().raiseActiveWindow();
        }
    }
}
