/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.HashMap;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.AnnotationModelEvent;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.jface.text.source.IAnnotationModelListener;
import org.eclipse.jface.text.source.IAnnotationModelListenerExtension;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ST;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.actions.ActionFactory;

import com.curl.eclipse.debug.CurlInstructionPointerAnnotation;
import com.curl.eclipse.debug.CurlStackFrame;
import com.curl.eclipse.editors.CurlEditor;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.ICurlEditorActionDefinitionIds;

/**
 * Manage a proxy for an editor's source view.
 */
public class SourceViewProxy extends ProxyView 
implements IAnnotationModelListener, IAnnotationModelListenerExtension, INotifyCommandChangedProxy
{
    
    // 100 is an arbitrary start value for these operations. We don't
    // want to collide with any Eclipse operation numbers.
    private static final int CURL_EDITOR_OPERATIONS = 100;
    public static final int FIX_INDENTATION_OPERATION = CURL_EDITOR_OPERATIONS;
    public static final int COMMENT_OPERATION = CURL_EDITOR_OPERATIONS + 1;
    public static final int UNCOMMENT_OPERATION = CURL_EDITOR_OPERATIONS + 2;
    public static final int BALANCE_OPERATION = CURL_EDITOR_OPERATIONS + 3;
    public static final int SHOW_COMPLETIONS_OPERATION = CURL_EDITOR_OPERATIONS + 4;
    public static final int TOGGLE_BREAKPOINT_OPERATION = CURL_EDITOR_OPERATIONS + 5;
    public static final int CONTINUE_TO_HERE_OPERATION = CURL_EDITOR_OPERATIONS + 6;
    public static final int COPY_TO_WATCH_OPERATION = CURL_EDITOR_OPERATIONS + 7;
    public static final int EDITOR_GOTO_SYMBOL = CURL_EDITOR_OPERATIONS + 8;
    public static final int LINE_UP = CURL_EDITOR_OPERATIONS + 9;
    public static final int LINE_DOWN = CURL_EDITOR_OPERATIONS + 10;
    public static final int LINE_START = CURL_EDITOR_OPERATIONS + 11;
    public static final int LINE_END = CURL_EDITOR_OPERATIONS + 12;
    public static final int COLUMN_PREVIOUS = CURL_EDITOR_OPERATIONS + 13;
    public static final int COLUMN_NEXT = CURL_EDITOR_OPERATIONS + 14;
    public static final int PAGE_UP = CURL_EDITOR_OPERATIONS + 15;
    public static final int PAGE_DOWN = CURL_EDITOR_OPERATIONS + 16;
    public static final int WORD_PREVIOUS = CURL_EDITOR_OPERATIONS + 17;
    public static final int WORD_NEXT = CURL_EDITOR_OPERATIONS + 18;
    public static final int TEXT_START = CURL_EDITOR_OPERATIONS + 19;
    public static final int TEXT_END = CURL_EDITOR_OPERATIONS + 20;
    public static final int WINDOW_START = CURL_EDITOR_OPERATIONS + 21;
    public static final int WINDOW_END = CURL_EDITOR_OPERATIONS + 22;
    
    public static final int SELECT_ALL = CURL_EDITOR_OPERATIONS + 23;
    public static final int SELECT_LINE_UP = CURL_EDITOR_OPERATIONS + 24;
    public static final int SELECT_LINE_DOWN = CURL_EDITOR_OPERATIONS + 25;
    public static final int SELECT_LINE_START = CURL_EDITOR_OPERATIONS + 26;
    public static final int SELECT_LINE_END = CURL_EDITOR_OPERATIONS + 27;
    public static final int SELECT_COLUMN_PREVIOUS = CURL_EDITOR_OPERATIONS + 28;
    public static final int SELECT_COLUMN_NEXT = CURL_EDITOR_OPERATIONS + 29;
    public static final int SELECT_PAGE_UP = CURL_EDITOR_OPERATIONS + 30;
    public static final int SELECT_PAGE_DOWN = CURL_EDITOR_OPERATIONS + 31;
    public static final int SELECT_WORD_PREVIOUS = CURL_EDITOR_OPERATIONS + 32;
    public static final int SELECT_WORD_NEXT = CURL_EDITOR_OPERATIONS + 33;
    public static final int SELECT_TEXT_START = CURL_EDITOR_OPERATIONS + 34;
    public static final int SELECT_TEXT_END = CURL_EDITOR_OPERATIONS + 35;
    public static final int SELECT_WINDOW_START = CURL_EDITOR_OPERATIONS + 36;
    public static final int SELECT_WINDOW_END = CURL_EDITOR_OPERATIONS + 37;
    
    public static final int PAGE_LEFT = CURL_EDITOR_OPERATIONS + 38;
    public static final int PAGE_RIGHT = CURL_EDITOR_OPERATIONS + 39;
    public static final int SELECT_PAGE_LEFT = CURL_EDITOR_OPERATIONS + 40;
    public static final int SELECT_PAGE_RIGHT = CURL_EDITOR_OPERATIONS + 41;
    public static final int SCROLL_PAGE_LEFT = CURL_EDITOR_OPERATIONS + 42;
    public static final int SCROLL_PAGE_RIGHT = CURL_EDITOR_OPERATIONS + 43;
    public static final int SCROLL_COLUMN_LEFT = CURL_EDITOR_OPERATIONS + 44;
    public static final int SCROLL_COLUMN_RIGHT = CURL_EDITOR_OPERATIONS + 45;
    public static final int INDENTED_NEW_LINE = CURL_EDITOR_OPERATIONS + 46;
    
    public static final int UNIT_UP = CURL_EDITOR_OPERATIONS + 47;
    public static final int UNIT_DOWN = CURL_EDITOR_OPERATIONS + 48;
    public static final int SELECT_UNIT_UP = CURL_EDITOR_OPERATIONS + 49;
    public static final int SELECT_UNIT_DOWN = CURL_EDITOR_OPERATIONS + 50;
    public static final int SELECT_UNIT_LEFT = CURL_EDITOR_OPERATIONS + 51;
    public static final int SELECT_UNIT_RIGHT = CURL_EDITOR_OPERATIONS + 52;
    public static final int EXPRESSION_UP = CURL_EDITOR_OPERATIONS + 53;
    public static final int EXPRESSION_DOWN = CURL_EDITOR_OPERATIONS + 54;
    public static final int EXPRESSION_LEFT = CURL_EDITOR_OPERATIONS + 55;
    public static final int EXPRESSION_RIGHT = CURL_EDITOR_OPERATIONS + 56;
    public static final int SELECT_EXPRESSION_UP = CURL_EDITOR_OPERATIONS + 57;
    public static final int SELECT_EXPRESSION_DOWN = CURL_EDITOR_OPERATIONS + 58;
    public static final int SELECT_EXPRESSION_LEFT = CURL_EDITOR_OPERATIONS + 59;
    public static final int SELECT_EXPRESSION_RIGHT = CURL_EDITOR_OPERATIONS + 60;
    public static final int INSERT_START_BRACE = CURL_EDITOR_OPERATIONS + 61;
    public static final int INSERT_END_BRACE = CURL_EDITOR_OPERATIONS + 62;
    public static final int CENTER_CURSOR = CURL_EDITOR_OPERATIONS + 63;
    public static final int SWITCH_POINT_AND_ANCHOR = CURL_EDITOR_OPERATIONS + 64;
    public static final int SELECT_BLOCK = CURL_EDITOR_OPERATIONS + 65;
    public static final int INSERT_TAB = CURL_EDITOR_OPERATIONS + 66;
    public static final int CANCEL_SELECTION = CURL_EDITOR_OPERATIONS + 67;
  
    public class OperationStatus
    {
        private boolean fEnabled;
        private String fLabel;
        private boolean fValue;
        
        OperationStatus()
        {
        }
        
        public boolean isEnabled()
        {
            return fEnabled;
        }
        
        public void setEnabled(
                boolean enabled)
        {
            fEnabled = enabled;
        }

        public void setValue(
                boolean value)
        {
            fValue = value;
        }

        public boolean isValue()
        {
            return fValue;
        }
        
        public void setLabel(
                String label)
        {
            fLabel = label;
        }
        
        public String getLabel()
        {
            return fLabel;
        }
    }
    private final HashMap<Integer, OperationStatus> fOperationStatus = new HashMap<Integer, OperationStatus>();

    private CurlEditor getEditor()
    {
        return (CurlEditor)fProxyOf;
    }

    public SourceViewProxy(
            CurlEditor curlEditor, Composite editorPaneContent, IProgressMonitor progressMonitor)
    {
        super(curlEditor,
                null // focus is managed in this class see #activateParentView()
                ); 
                
        int[] operations = {
                ITextOperationTarget.SHIFT_RIGHT,
                ITextOperationTarget.SHIFT_LEFT, ITextOperationTarget.PRINT, ITextOperationTarget.PREFIX,
                ITextOperationTarget.STRIP_PREFIX, FIX_INDENTATION_OPERATION, COMMENT_OPERATION,
                UNCOMMENT_OPERATION, BALANCE_OPERATION, SHOW_COMPLETIONS_OPERATION,
                TOGGLE_BREAKPOINT_OPERATION, 
                CONTINUE_TO_HERE_OPERATION, 
                COPY_TO_WATCH_OPERATION, EDITOR_GOTO_SYMBOL,
                // Navigation.
                LINE_UP, LINE_DOWN, LINE_START, LINE_END,
                COLUMN_PREVIOUS, COLUMN_NEXT, PAGE_UP, PAGE_DOWN,
                WORD_PREVIOUS, WORD_NEXT, TEXT_START, TEXT_END,
                WINDOW_START, WINDOW_END,
                // Selection.
                // SELECT_ALL, 
                SELECT_LINE_UP, SELECT_LINE_DOWN, 
                SELECT_LINE_START, SELECT_LINE_END,
                SELECT_COLUMN_PREVIOUS, SELECT_COLUMN_NEXT,
                SELECT_PAGE_UP, SELECT_PAGE_DOWN,
                SELECT_WORD_PREVIOUS, SELECT_WORD_NEXT,
                SELECT_TEXT_START, SELECT_TEXT_END,
                SELECT_WINDOW_START, SELECT_WINDOW_END,
                PAGE_LEFT, PAGE_RIGHT, SELECT_PAGE_LEFT, SELECT_PAGE_RIGHT,
                SCROLL_PAGE_LEFT, SCROLL_PAGE_RIGHT,
                SCROLL_COLUMN_LEFT, SCROLL_COLUMN_RIGHT,
                INDENTED_NEW_LINE,
                UNIT_UP, UNIT_DOWN, SELECT_UNIT_UP, SELECT_UNIT_DOWN,
                SELECT_UNIT_LEFT, SELECT_UNIT_RIGHT,
                EXPRESSION_UP, EXPRESSION_DOWN, 
                EXPRESSION_LEFT, EXPRESSION_RIGHT,
                SELECT_EXPRESSION_UP, SELECT_EXPRESSION_DOWN,
                SELECT_EXPRESSION_LEFT, SELECT_EXPRESSION_RIGHT,
                INSERT_START_BRACE, INSERT_END_BRACE,
                CENTER_CURSOR, SWITCH_POINT_AND_ANCHOR,
                SELECT_BLOCK, INSERT_TAB, CANCEL_SELECTION};
        
        for (int operation : operations) {
            fOperationStatus.put(new Integer(operation), new OperationStatus());
        }
        fComposite = editorPaneContent;
    }
    
    
    @Override
    public CurlEditor getProxyOf()
    {
        return (CurlEditor) super.getProxyOf();
    }

    public void reloadBufferContent(final DocumentProxy documentProxy)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandLoadFileIntoEditor) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(documentProxy.getProxyID());
            }
        });
    }

    public void setSelectionRange(
            final int offset,
            final int length)
    {
        if (fViewExists) {
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandEditorNavigate) {
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(offset);
                    write(length);
                }
            });
        }
    }
    
    public void sendAutoCompleteMessage(final char ch) {
        if (fViewExists) {
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandAutoComplete) {
                @Override
                protected void writeArguments()
                {       
                    write(fProxyID);
                    write("" + ch); //$NON-NLS-1$
                }
            });
        }
    }

    public void setSelectionByRowLine(
            final int lineNumber,
            final int columnNumber,
            final boolean selectToEOL)
    {
        if (fViewExists) {
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandEditorNavigateRowColumn) {
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(lineNumber);
                    write(columnNumber);
                    write(selectToEOL);
                }
            });
        }
    }

    @Override
    protected Widget getKeyPressNotifier()
    {
        final ISourceViewer sourceViewer =      
            getEditor().getCurlEditorSourceViewer();
        return sourceViewer.getTextWidget();
    }
    
    @Override
    protected boolean forwardsToCurlMediatorIfRequired(
            Event event)
    {
        final ISourceViewer sourceViewer =      
            getEditor().getCurlEditorSourceViewer();
        StyledText textWidget = sourceViewer.getTextWidget();
        
        int action = getAction(event, textWidget);
        final Integer value = fActionTable.get(action);
        if (value != null) {
            // forward to curl mediator
            doOperation(value.intValue());
            return true;
        }
        else {
            return false;
        }
    }
    
    /**
     * Shamelessly copied/adapted from super class StyledText
     * @param e
     * @param textWidget 
     * @return
     */
    private int getAction(
            Event e, StyledText textWidget)
    {
        int action;
        if (e.keyCode != 0) {
            // special key pressed (e.g., F1)
            action = textWidget.getKeyBinding(e.keyCode | e.stateMask);
        } else {
            // character key pressed
            action = textWidget.getKeyBinding(e.character | e.stateMask);
            if (action == SWT.NULL) {
                // see if we have a control character
                if ((e.stateMask & SWT.CTRL) != 0 && (e.character >= 0) && e.character <= 31) {
                    // get the character from the CTRL+char sequence, the control
                    // key subtracts 64 from the value of the key that it modifies
                    int c = e.character + 64;
                    action = textWidget.getKeyBinding(c | e.stateMask);
                }
            }
        }
        return action;
    }


    private static final HashMap<Integer, Integer> fActionTable = new HashMap<Integer, Integer>();
    static {
        // Navigation Commands.
//      fActionTable.put(ST.LINE_UP, SourceViewProxy.LINE_UP);
//      fActionTable.put(ST.LINE_DOWN, SourceViewProxy.LINE_DOWN);
//        fActionTable.put(ST.LINE_START, SourceViewProxy.LINE_START);
//        fActionTable.put(ST.LINE_END, SourceViewProxy.LINE_END);
//      fActionTable.put(ST.COLUMN_PREVIOUS, SourceViewProxy.COLUMN_PREVIOUS);
//      fActionTable.put(ST.COLUMN_NEXT, SourceViewProxy.COLUMN_NEXT);
        fActionTable.put(ST.PAGE_UP, SourceViewProxy.PAGE_UP);
        fActionTable.put(ST.PAGE_DOWN, SourceViewProxy.PAGE_DOWN);
//        fActionTable.put(ST.WORD_PREVIOUS, SourceViewProxy.WORD_PREVIOUS);
//        fActionTable.put(ST.WORD_NEXT, SourceViewProxy.WORD_NEXT);
//        fActionTable.put(ST.TEXT_START, SourceViewProxy.TEXT_START);
//        fActionTable.put(ST.TEXT_END, SourceViewProxy.TEXT_END);
        fActionTable.put(ST.WINDOW_START, SourceViewProxy.WINDOW_START);
        fActionTable.put(ST.WINDOW_END, SourceViewProxy.WINDOW_END);

        // Selection Commands.
        // fActionTable.put(ST.SELECT_ALL, SourceViewProxy.SELECT_ALL);
        // FIXME: if comment these, then selection is broken, investigate....
//        fActionTable.put(ST.SELECT_LINE_UP, SourceViewProxy.SELECT_LINE_UP);
//        fActionTable.put(ST.SELECT_LINE_DOWN, SourceViewProxy.SELECT_LINE_DOWN);
//        fActionTable.put(ST.SELECT_LINE_START, SourceViewProxy.SELECT_LINE_START);
//        fActionTable.put(ST.SELECT_LINE_END, SourceViewProxy.SELECT_LINE_END);
//        fActionTable.put(ST.SELECT_COLUMN_PREVIOUS, SourceViewProxy.SELECT_COLUMN_PREVIOUS );
//        fActionTable.put(ST.SELECT_COLUMN_NEXT, SourceViewProxy.SELECT_COLUMN_NEXT );
        fActionTable.put(ST.SELECT_PAGE_UP, SourceViewProxy.SELECT_PAGE_UP);
        fActionTable.put(ST.SELECT_PAGE_DOWN, SourceViewProxy.SELECT_PAGE_DOWN);
//        fActionTable.put(ST.SELECT_WORD_PREVIOUS, SourceViewProxy.SELECT_WORD_PREVIOUS);
//        fActionTable.put(ST.SELECT_WORD_NEXT, SourceViewProxy.SELECT_WORD_NEXT);
//        fActionTable.put(ST.SELECT_TEXT_START, SourceViewProxy.SELECT_TEXT_START);
//        fActionTable.put(ST.SELECT_TEXT_END, SourceViewProxy.SELECT_TEXT_END);
        fActionTable.put(ST.SELECT_WINDOW_START, SourceViewProxy.SELECT_WINDOW_START);
        fActionTable.put(ST.SELECT_WINDOW_END, SourceViewProxy.SELECT_WINDOW_END);
    }
            
    /**
     * Respond to mediator command that selection has changed.
     * 
     * @param sockIn
     * @param sockOut
     */
    private static class EditorSelectionChangedCommand
    {
        public int selectionChangedOffset;
        public int selectionChangedLength;
        public int cursorPositionRow;
        public int cursorPositionColumn;
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandEditorSelectionChanged, new EditorSelectionChangedCommandHandler());
    }
    public static class EditorSelectionChangedCommandHandler extends ProxyCommandHandler<SourceViewProxy, EditorSelectionChangedCommand>
    {
        @Override
        EditorSelectionChangedCommand decode(
                DataInputStream sockIn)
        {
            EditorSelectionChangedCommand command = new EditorSelectionChangedCommand();
            command.selectionChangedOffset = DataIO.readInt(sockIn);
            command.selectionChangedLength = DataIO.readInt(sockIn);
            command.cursorPositionRow = DataIO.readInt(sockIn);
            command.cursorPositionColumn = DataIO.readInt(sockIn);
            return command;
        }

        @Override
        void execute(
                final EditorSelectionChangedCommand command,
                final SourceViewProxy proxy,
            DataOutputStream sockOut)
        {
            final CurlEditor curlEditor = proxy.getEditor();
            curlEditor.getDisplay().asyncExec(new Runnable() {
                public void run()
                {
                    curlEditor.getCurlEditorSourceViewer().selectionChangedResponse(
                            command.selectionChangedOffset,
                            command.selectionChangedLength);
                }
            });
        }
    }

    /**
     * Respond to mediator command for the status bar message.
     */
    public static class StatusBarMessageProxyCommand
    {
        public String msg;
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandOnStatusBarMessage, new StatusBarMessageCommandHandler());
    }
    public static class StatusBarMessageCommandHandler extends ProxyCommandHandler<SourceViewProxy, StatusBarMessageProxyCommand>
    {
        
        @Override
        StatusBarMessageProxyCommand decode(
                DataInputStream sockIn)
        {
            StatusBarMessageProxyCommand command = new StatusBarMessageProxyCommand();
            command.msg = DataIO.readString(sockIn);
            return command;
        }
        
        @Override
        void execute(
                final StatusBarMessageProxyCommand command,
                final SourceViewProxy proxy,
                DataOutputStream sockOut)
        {
            final CurlEditor curlEditor = proxy.getEditor();
            curlEditor.getDisplay().asyncExec(new Runnable() {
                public void run()
                {
                    curlEditor.setStatusText(command.msg);
                }
            });
        }
    }

    public boolean isSupportedOperation(
            int operation)
    {
        return fOperationStatus.containsKey(new Integer(operation));
    }

    public void doOperation(
            int operation)
    {
        Integer key = new Integer(operation);
        if (fOperationStatus.containsKey(key) && fOperationStatus.get(key).isEnabled()) {
            // Send operation to mediator to perform the operation.
            sendDoOperation(operation);

        }
    }

    public String getLabel(
            int operation)
    {
        Integer key = new Integer(operation);
        if (fOperationStatus.containsKey(key))
            return fOperationStatus.get(key).getLabel();
        else
            return null;
    }
    
    public boolean canDoOperation(
            int operation)
    {
        Integer key = new Integer(operation);
        if (fOperationStatus.containsKey(key))
            return fOperationStatus.get(key).isEnabled();
        else
            return false;
    }

    @Override
    public void setOperationStatus(
            int operation,
            boolean status,
            boolean value,
            String label)
    {
        OperationStatus operationStatus = fOperationStatus.get(operation);
        operationStatus.setEnabled(status);
        operationStatus.setValue(value);
        operationStatus.setLabel(label);
        String action = null;
        String action2 = null;
        String action3 = null;
        switch (operation) {
        case ITextOperationTarget.CUT:
            action = ActionFactory.CUT.getId();
            break;
        case ITextOperationTarget.COPY:
            action = ActionFactory.COPY.getId();
            break;
        case ITextOperationTarget.PASTE:
            action = ActionFactory.PASTE.getId();
            break;
        case ITextOperationTarget.DELETE:
            action = ActionFactory.DELETE.getId();
            break;
        case ITextOperationTarget.SELECT_ALL:
            action = ActionFactory.SELECT_ALL.getId();
            break;
        case FIX_INDENTATION_OPERATION:
            // FIXME: make the action cases all data driven
            action = ICurlEditorActionDefinitionIds.FixIndentationActionId;
            break;
        case COMMENT_OPERATION:
            action = ICurlEditorActionDefinitionIds.CommentActionId;
            break;
        case UNCOMMENT_OPERATION:
            action = ICurlEditorActionDefinitionIds.UncommentActionId;
            break;
        case BALANCE_OPERATION:
            action = ICurlEditorActionDefinitionIds.BalanceActionId;
            break;
        case SHOW_COMPLETIONS_OPERATION:
            action = ICurlEditorActionDefinitionIds.ShowCompletionsActionId;
            break;
        case TOGGLE_BREAKPOINT_OPERATION:
            action = ICurlEditorActionDefinitionIds.ToggleBreakpointActionId;
            break;
        case CONTINUE_TO_HERE_OPERATION:
            action = ICurlEditorActionDefinitionIds.ContinueToHereActionId;
            break;
        case COPY_TO_WATCH_OPERATION:
            action = ICurlEditorActionDefinitionIds.CopyToWatchActionId;
            break;
        case EDITOR_GOTO_SYMBOL:
            action = ICurlEditorActionDefinitionIds.SearchDefinitionInWSActionId;
            action2 = ICurlEditorActionDefinitionIds.SearchReferenceInWSActionId;
            action3 = ICurlEditorActionDefinitionIds.GotoDefinitionInWSActionId;
            break;
        case PAGE_LEFT:
            action = ICurlEditorActionDefinitionIds.PageLeftActionId;
            break;
        case PAGE_RIGHT:
            action = ICurlEditorActionDefinitionIds.PageRightActionId;
            break;
        case SELECT_PAGE_LEFT:
            action = ICurlEditorActionDefinitionIds.SelectPageLeftActionId;
            break;
        case SELECT_PAGE_RIGHT:
            action = ICurlEditorActionDefinitionIds.SelectPageRightActionId;
            break;
        case SCROLL_PAGE_LEFT:
            action = ICurlEditorActionDefinitionIds.ScrollPageLeftActionId;
            break;
        case SCROLL_PAGE_RIGHT:
            action = ICurlEditorActionDefinitionIds.ScrollPageRightActionId;
            break;
        case SCROLL_COLUMN_LEFT:
            action = ICurlEditorActionDefinitionIds.ScrollColumnLeftActionId;
            break;
        case SCROLL_COLUMN_RIGHT:
            action = ICurlEditorActionDefinitionIds.ScrollColumnRightActionId;
            break;
        case INDENTED_NEW_LINE:
            action = ICurlEditorActionDefinitionIds.IndentedNewLineActionId;
            break;
        case UNIT_UP:
            action = ICurlEditorActionDefinitionIds.UnitUpActionId;
            break;
        case UNIT_DOWN:
            action = ICurlEditorActionDefinitionIds.UnitDownActionId;
            break;
        case SELECT_UNIT_UP:
            action = ICurlEditorActionDefinitionIds.SelectUnitUpActionId;
            break;
        case SELECT_UNIT_DOWN:
            action = ICurlEditorActionDefinitionIds.SelectUnitDownActionId;
            break;
        case SELECT_UNIT_LEFT:
            action = ICurlEditorActionDefinitionIds.SelectUnitLeftActionId;
            break;
        case SELECT_UNIT_RIGHT:
            action = ICurlEditorActionDefinitionIds.SelectUnitRightActionId;
            break;
        case EXPRESSION_UP:
            action = ICurlEditorActionDefinitionIds.ExpressionUpActionId;
            break;
        case EXPRESSION_DOWN:
            action = ICurlEditorActionDefinitionIds.ExpressionDownActionId;
            break;
        case EXPRESSION_LEFT:
            action = ICurlEditorActionDefinitionIds.ExpressionLeftActionId;
            break;
        case EXPRESSION_RIGHT:
            action = ICurlEditorActionDefinitionIds.ExpressionRightActionId;
            break;
        case SELECT_EXPRESSION_UP:
            action = ICurlEditorActionDefinitionIds.SelectExpressionUpActionId;
            break;
        case SELECT_EXPRESSION_DOWN:
            action = ICurlEditorActionDefinitionIds.SelectExpressionDownActionId;
            break;
        case SELECT_EXPRESSION_LEFT:
            action = ICurlEditorActionDefinitionIds.SelectExpressionLeftActionId;
            break;
        case SELECT_EXPRESSION_RIGHT:
            action = ICurlEditorActionDefinitionIds.SelectExpressionRightActionId;
            break;
        case INSERT_START_BRACE:
            action = ICurlEditorActionDefinitionIds.InsertStartBraceActionId;
            break;
        case INSERT_END_BRACE:
            action = ICurlEditorActionDefinitionIds.InsertEndBraceActionId;
            break;
        case CENTER_CURSOR:
            action = ICurlEditorActionDefinitionIds.CenterCursorActionId;
            break;
        case SWITCH_POINT_AND_ANCHOR:
            action = ICurlEditorActionDefinitionIds.SwitchPointAndAnchorActionId;
            break;
        case SELECT_BLOCK:
            action = ICurlEditorActionDefinitionIds.SelectBlockActionId;
            break;
        case INSERT_TAB:
            action = ICurlEditorActionDefinitionIds.InsertTabActionId;
            break;
        case CANCEL_SELECTION:
            action = ICurlEditorActionDefinitionIds.CancelSelectionActionId;
            break;
        }
        if (fViewExists) {
            for (String actionId : new String[]{action, action2, action3}) {
                if (actionId != null) {
                    getEditor().updateAction(actionId);
                }
            }
        }
    }

    public void setViewExists(
            boolean b)
    {
        fViewExists = b;
    }
    
    
    public void modelChanged(
            IAnnotationModel model)
    {
        // No support for older version of the API
        assert false;
    }

    public void modelChanged(
            AnnotationModelEvent event)
    {
        // TODO: could also take care of annotation created by Class Browser
        // whit same approach....
        Annotation[] addedAnnotations = event.getAddedAnnotations();
        for (Annotation annotation : addedAnnotations) {
            if (annotation instanceof CurlInstructionPointerAnnotation) {
                sendHighliteLine(MediatorConnection.commandSetHighlightLine, (CurlInstructionPointerAnnotation) annotation);
            }
        }

// No need to send removal of pc-line highlighting to curl since curl can
// online manage a single pc-line and do what's necessary to cleanup when needed.        
//        Annotation[] removedAnnotations = event.getRemovedAnnotations();
//        for (Annotation annotation : removedAnnotations) {
//            if (annotation instanceof CurlInstructionPointerAnnotation) {
//                sendHighliteLine(MediatorConnection.commandUnsetHighlightLine, (CurlInstructionPointerAnnotation) annotation);
//            }
//        }
    }

    private void sendHighliteLine(
            int commande,
            CurlInstructionPointerAnnotation debugAnnotation)
    {
        IStackFrame frame = debugAnnotation.getFrame();
        try {
            if (frame instanceof CurlStackFrame) {
                final CurlStackFrame curlStackFrame = (CurlStackFrame)frame;
                
                final int lineNumber = curlStackFrame.getLineNumber();
                fMediatorConnection.async.execute(new AsynchronousRemoteOperation(commande){
                    @Override
                    protected void writeArguments()
                    {
                        write(fProxyID);
                        write(lineNumber);
                    }
                });
            }
        } catch (DebugException e) {
            CoreUtil.logError("Sending pc-line highlighting to mediator", e); //$NON-NLS-1$
        }
    }

    @Override
    protected void activateParentView()
    {
        CurlEditor editor = (CurlEditor)fProxyOf;
        editor.activateParentView();
    }
    
    @Override
    protected boolean maybeSendEscKeyToWorkbench() 
    {
        return false;
	}

    @Override
    protected Menu getContextMenu()
    {
        return getProxyOf().getContextMenu();
    }
    
    @Override
    protected void handleCurlFocusOut()
    { 
        CurlEditor editor = getEditor();
        if (editor != null) {
            editor.handleCurlFocusOut();
        }
        super.handleCurlFocusOut();
    }
    
    @Override
    protected void handleCurlPointerPress()
    {
        CurlEditor editor = getEditor();
        if (editor != null) {
            editor.handleCurlPointerPress();
        }
        super.handleCurlPointerPress();
    }
}
