/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.DataInputStream;
import java.io.DataOutputStream;

import org.eclipse.core.runtime.Platform;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Widget;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.IPage;
import org.eclipse.ui.views.contentoutline.ContentOutline;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.SafeComposite;
import com.curl.eclipse.editors.CurlView;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.WindowHandle;


/**
 * Top level class for any proxy. Subclasses that create a particular proxy object can extend this class for
 * customization.
 */
public abstract class ProxyView extends Proxy
{
    /**
     * 
     */
    protected abstract class CreateViewRemoteOperation extends AsynchronousRemoteOperation
    {
        private final int fParentProxyID;
        private final int xCoordinate;
        private final int yCoordinate;
        
        public CreateViewRemoteOperation(
                int commandCode,
                Composite composite)
        {
            this(
                    PlatformUI.getWorkbench().getActiveWorkbenchWindow(),
                    commandCode,
                    composite);
        }

        /**
         * @param workbenchWindow 
         * @param commandCode
         */
        public CreateViewRemoteOperation(
                IWorkbenchWindow workbenchWindow, 
                int commandCode,
                Composite composite)
        {
            super(commandCode);
            WorkbenchWindowProxy windowProxy =
                CurlPlugin.getDefault().getWorkbenchOperations().getWorkbenchWindowProxyFor(workbenchWindow);
            fParentProxyID = windowProxy.fProxyID;
            fComposite = composite;
            xCoordinate = fComposite.getSize().x;
            yCoordinate = fComposite.getSize().y;
            /*
             * Linux/GTK requires a safer protocol to set the parent to the curl view about to be created
             */
            // TODO refactoring: by the time this constructor is invoked, fComposite is supposed to be set
            // the curl view will be created with fSafeHolder as a parent, later on, when Mediator confirms asynchronously,
            // fSafeHolder will be reparented to fComposite
            fSafeHolder = allocateSafeSiblingForParent();

        }

        @Override
        protected final void writeArguments()
        {
            write(fParentProxyID);
            write(fProxyID);
            // the real parent (fComposite) will be attached later
            write(new WindowHandle(fSafeHolder).getHandle());
            write(xCoordinate);
            write(yCoordinate);
            writeViewArguments();
        }

        protected void writeViewArguments()
        {
            // nothing by default
        }
    }

    protected boolean fViewExists;
    protected Composite fComposite;
    private SafeComposite fSafeHolder;
    private final IWindowListener fWindowListener;
    private boolean fResizeEnqueued;
    private final String fViewId;
    private boolean fSetFocusWasRequested;

    private static SafeComposite allocateSafeSiblingForParent()
    {
        return CurlPlugin.getDefault().getSafeCompositeManager().allocateSafeSiblingForParent();
    }

    // TODO: Should define an interface instead of passing an "Object"
    protected ProxyView(
            Object proxyOf,
            String viewId)
    {
        super(proxyOf);
        fViewId = viewId;

        /*
         * To have curl view to get the keyboard focus 
         * when the eclipse windows gets activates
         * TODO: unfortunately this window listener won't see modal
         * dialog activation/deactivation (like project properties for example)
         */
        fWindowListener = new IWindowListener(){
            private boolean fHadFocus;
            public void windowActivated(
                    IWorkbenchWindow window)
            {
                // if had focus before window was deactivated 
                if (isMyWindow(window) && fHadFocus) {                	
                	setFocus();
                }
            }
            /**
             * @return true if the window is associated with our proxied view
             */
            private boolean isMyWindow(
                    IWorkbenchWindow window)
            {
                if (fComposite.isDisposed())
                    return false;

                Shell shell = fComposite.getShell();
                return shell.getData() == window;
            }
            public void windowClosed(
                    IWorkbenchWindow window)
            {}
            public void windowDeactivated(
                    IWorkbenchWindow window)
            {
                if (isMyWindow(window)) {
                    IWorkbenchPage activePage = window.getActivePage();
                    IWorkbenchPart activePart = activePage.getActivePart();
                    if (activePart == fProxyOf) {
                        fHadFocus = true;
                    } else if (activePart instanceof ContentOutline) {
                        ContentOutline contentOutline = (ContentOutline)activePart;
                        IPage currentPage = contentOutline.getCurrentPage();
                        fHadFocus = currentPage == fProxyOf; // FIXME: check this
                    } else {
                        fHadFocus = false;
                    }
                }
            }
            public void windowOpened(
                    IWorkbenchWindow window)
            {}
        };
        CurlPlugin.getDefault().getWorkbench().addWindowListener(fWindowListener);
    }

    @Override
    public void destroy()
    {
        CurlPlugin.getDefault().getWorkbench().removeWindowListener(fWindowListener);

        fViewExists = false; // TODO: Do we really need this?

        // if parent has not been disposed yet then the holder of the curl view
        // can (and must) be reparented since the curl close operation is asynchronous
        // and it could fail/crash if the graphic region is invalid.
        if (! fComposite.isDisposed() && fSafeHolder != null) {
            fSafeHolder.reparentWithSafe();
        }

        super.destroy();
    }

    public void setFocus()
    {
        /*
         * When Eclipse creates a curl view, the view isn't immediately attached to the
         * eclipse graphic hierarchy (because of another issue on X11/Linux that would lead
         * to crashes). There is an asynchronous answer from curl to eclipse when curl is
         * done building its view. But if it takes too long to obtain this response then
         * any setFocus() that were requested before are ineffective, and ultimately the OS
         * decides that the eclipse window has no focus.
         */
        if (fSafeHolder.isAttachedToEclipseParent()) {
            fSetFocusWasRequested = false;
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandSetFocus){
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                }
            });
        } else {
            // if curl didn't confirm that the view is created, differ
            // the set focus request
            fSetFocusWasRequested = true;
        }
    }

    public void releaseFocus()
    {
        fSetFocusWasRequested = false;
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandReleaseFocus){
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
            }
        });
    }


    public void createView(
            IWorkbenchWindow workbenchWindow, 
            final Composite composite,
            final int viewKind)
    {
        
        fMediatorConnection.async.execute(new CreateViewRemoteOperation(workbenchWindow, viewKind, composite) {
        });
        fViewExists = true;
    }

    /*
     * TODO: refactor to avoid passing composite
     * it is a 1-1 relation with this instance...
     */
    public void resizeView(
            final Composite composite)
    {
        if (!composite.isDisposed() && fViewExists) {
            if (!fResizeEnqueued) {
                Display display = composite.getDisplay();
                if (display != null) {
                    fResizeEnqueued = true;
                    display.asyncExec(new Runnable() {
                        @Override
                        public void run()
                        {
                            fResizeEnqueued = false;
                            doResize(composite);
                        }
                    });
                } else {
                    doResize(composite);
                }
            }
        }
    }

    private void doResize(Composite composite)
    {
        if (!composite.isDisposed() && fViewExists) {
            final int x = fComposite.getSize().x;
            final int y = fComposite.getSize().y;
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandResizeView) {
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(x);
                    write(y);
                }
            });
        }
    }


    /**
     * Respond to mediator command that a key (or combination of keys) has been pressed.
     */
    static
    {
        EclipseServer.register(MediatorConnection.commandHandleKeyPress, new HandleKeyPressCommandHandler());
    }
    public static class HandleKeyPressCommandHandler extends ProxyCommandHandler<ProxyView, Event>
    {
        @Override
        Event decode(
                DataInputStream sockIn)
        {
            Event event = new Event();
            event.character = (char)DataIO.readInt(sockIn);
            event.keyCode = DataIO.readInt(sockIn);
            event.stateMask = DataIO.readInt(sockIn);
            event.type = SWT.KeyDown;
            // HACK: synthesize time for event.  As long as two successive events
            // have different times, any value should be OK here.  That is, it is
            // OK if the "time" of two events are in the wrong order.
            event.time = (int) System.currentTimeMillis();
            return event;
        }

        @Override
        void execute(
                final Event event,
                final ProxyView proxy,
                DataOutputStream sockOut)
        {
            final Display display = proxy.fComposite.getDisplay();
            event.display = display;
//            CoreUtil.logInfo("Sending KeyDown event " + event.character + " " + event.keyCode + " " + event.stateMask);
            display.syncExec(new Runnable() {
                public void run() {
                    if (!proxy.fComposite.isDisposed()) {    
                        if (!proxy.processIfTopLevelEvent(event)) {
                            if (proxy.forwardsToCurlMediatorIfRequired(event)) {
                                // nothing more to do
                            }
                            else {
                                proxy.getKeyPressNotifier().notifyListeners(event.type, event);
                            }
                        }
                    }
                };
            });
        }
    }

    protected Widget getKeyPressNotifier()
    {
        return fComposite;   
    }

    protected boolean forwardsToCurlMediatorIfRequired(
            Event event)
    {
        // by default
        return false;
    }

    protected boolean maybeSendEscKeyToWorkbench() {
        return true;
    }
    
    // HACK: If only the ALT modifier key was pressed then see if the 
    // gesture was to drop down the workbench menu. See topLevelMenuItem
    // for details.
    protected boolean processIfTopLevelEvent(Event event)
    {
        if ((event.stateMask == SWT.ALT && topLevelMenuItem(event.character)) ||
            (maybeSendEscKeyToWorkbench() && event.keyCode == SWT.ESC && event.stateMask == 0)) {
            Display display = fComposite.getDisplay();
            fComposite.forceFocus();
            while (!display.readAndDispatch()) {};
            final Event ae = new Event();
            ae.keyCode = event.stateMask;
            ae.type = SWT.KeyDown;
            ae.doit = true;
            if (event.keyCode == SWT.ESC) {
                ae.detail = SWT.TRAVERSE_ESCAPE;
            } else {
                ae.detail = SWT.TRAVERSE_MNEMONIC;
            }
            display.post(ae);
            while (!display.readAndDispatch()) {};
            display.post(event);
            while (!display.readAndDispatch()) {};
            fComposite.setFocus();
            return true;
    	}
    	return false;
    }

    // Note that processIfTopLevelMenuEvent only works for Win32. On
    // Linux if we process say Alt+F we get into an infinite loop. For
    // now to process menu items on Linux you will have to use mouse
    // or use the key shortcuts, if a menu item has one.
    private boolean topLevelMenuItem(char c)
    {
        if (!Platform.getOS().equals(Platform.OS_WIN32)) {
          return false;
        }

        Shell sh = fComposite.getShell();
        if (sh == null) { 
            return false;
        }
        Menu mb = sh.getMenuBar();
        if (mb == null) { 
            return false;
        }
        int count = mb.getItemCount();   
        Character ch = Character.toLowerCase(c);
        for (int i = 0; i < count; i++) {
            MenuItem item = mb.getItem(i);
            if (item.getEnabled()) {
                String str = item.getText();
                int index = str.indexOf('&');
                if (index != - 1 && str.length() > index + 1) {
                    if (Character.toLowerCase(str.charAt(index + 1)) == ch) {
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * respond to open context menu gesture
     */
    private static class OnContextMenuEventResponseCommand
    {
        final int fXAbsoluteOnScreenInPixel;
        final int fYAbsoluteOnScreenInPixel;
        
        OnContextMenuEventResponseCommand(
                int aXAbsoluteOnScreenInPixe,
                int aYAbsoluteOnScreenInPixel)
        {
            fXAbsoluteOnScreenInPixel = aXAbsoluteOnScreenInPixe;
            fYAbsoluteOnScreenInPixel = aYAbsoluteOnScreenInPixel;
        }
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandOnContextMenuEventResponse, new OnContextMenuEventResponseHandler());
    }
    public static class OnContextMenuEventResponseHandler extends ProxyCommandHandler<ProxyView, OnContextMenuEventResponseCommand>
    {
        @Override
        OnContextMenuEventResponseCommand decode(
                DataInputStream sockIn)
        {
            OnContextMenuEventResponseCommand command = new OnContextMenuEventResponseCommand(
                    DataIO.readInt(sockIn),
                    DataIO.readInt(sockIn)
            );
            return command;
        }
        @Override
        void execute(
                final OnContextMenuEventResponseCommand command,
                final ProxyView proxy,
                DataOutputStream sockOut)
        {
            proxy.fComposite.getDisplay().asyncExec(new Runnable() {
                public void run()
                {
                    proxy.onContextMenu(command.fXAbsoluteOnScreenInPixel, command.fYAbsoluteOnScreenInPixel);
                }
            });
        }
    }

    private void onContextMenu(int x, int y)
    {
        CoreUtil.showContextMenu(getContextMenu(), x, y);
    }
    
    /**
     * @return context menu when handling onContextMenu events, else throws RuntimeException
     */
    protected Menu getContextMenu()
    {
        throw new RuntimeException("ERROR on Curl Mediator side"); //$NON-NLS-1$
    }
    
    /**
     * Mediator is telling us that a Curl view has gained or lost 
     * focus.
     * For example, if the Focus is lost then tell the Eclipse view's
     * proxyOf, to cancel any incremental searches, if any.
     * 
     * Note that we do not do this for any dialog proxy views. They do 
     * not need it.
     */
    private static class OnCurlFocusEventResponseCommand
    {
        final boolean focusIn;
        
        OnCurlFocusEventResponseCommand(boolean in)
        {
            focusIn = in;
        }
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandOnCurlFocus, new OnCurlFocusEventResponseHandler());
    }
    public static class OnCurlFocusEventResponseHandler extends ProxyCommandHandler<ProxyView, OnCurlFocusEventResponseCommand>
    {
        @Override
        OnCurlFocusEventResponseCommand decode(
                DataInputStream sockIn)
        {
            OnCurlFocusEventResponseCommand command = new OnCurlFocusEventResponseCommand(DataIO.readBoolean(sockIn));
            
            return command;
        }
        @Override
        void execute(
                final OnCurlFocusEventResponseCommand command,
                final ProxyView proxy,
                DataOutputStream sockOut)
        {
            proxy.fComposite.getDisplay().asyncExec(new Runnable() {
                public void run()
                {
                    if (command.focusIn) {
                        proxy.handleCurlFocusIn();
                    } else {
                        proxy.handleCurlFocusOut();
                    }
                }
            });
        }
    }

    protected void handleCurlFocusIn() {
        // Maybe we should use this instead of the Pointer
        // Click command. But that will need some redesign.
    }
    
    protected void handleCurlFocusOut() {
        // By default does nothing.
    }
    
    /**
     * Mediator is telling us that a Curl view has gained focus. Tell the corresponding Eclipse view, the
     * proxyOf, to set focus. This will cause the Eclipse view to set focus and then send a message to the
     * Mediator to set focus. The mediator will then steal the focus back from Eclipse!
     * 
     * Note that we do not do this for any dialog proxy views. They do not need it.
     */
    static
    {
        EclipseServer.register(MediatorConnection.commandOnPointerClick, new OnPointerClickommandHandler());
    }
    public static class OnPointerClickommandHandler extends ProxyCommandHandler<ProxyView, NullProxyCommand>
    {
        @Override
        void execute(
                final NullProxyCommand command,
                final ProxyView proxy,
                DataOutputStream sockOut)
        {
            proxy.fComposite.getDisplay().asyncExec(new Runnable() {
                public void run()
                {
                    proxy.handleCurlPointerPress();
                    proxy.activateParentView();
                }
            });
        }
    }

    protected void handleCurlPointerPress() {
        // By default it does nothing.
    }

    protected void activateParentView()
    {
        if (fViewId == null) {
            // this is a modal dialog,
            // no need to managed focus so far
            return;
        }
        // Proxies that are not for a CurlView
        // must overwrite activateParentView()
        CurlView curlView = (CurlView) fProxyOf;
        curlView.activateParentView(fViewId);
    }

    /**
     * Curl view has been safely created with a safe "dummy" parent,
     * it is now a time to make this view visible to eclipse.
     */
    static
    {
        EclipseServer.register(MediatorConnection.commandOnViewCreated, new OnViewCreatedCommandHandler());
    }
    public static class OnViewCreatedCommandHandler extends ProxyCommandHandler<ProxyView, NullProxyCommand>
    {
        @Override
        void execute(
                final NullProxyCommand command,
                final ProxyView proxy,
                DataOutputStream sockOut)
        {
            CurlPlugin.getDefault().getWorkbench().getDisplay().asyncExec(new Runnable(){
                public void run()
                {
                    if (proxy.fComposite.isDisposed()) {
                        // if curl side is extremely slow, the user may have
                        // had enough time to close the eclipse editor (or view),
                        // in that case a close-view has been enqueued to the mediator,
                        // and it will be processed to cleanup accordingly, which
                        // also means that ultimately handleOnViewClosed will be called back.
                        // Thus the curl view never had a chance to show up.
                        // Note that at this point fSafeHolder remains parented with
                        // the invisible "safe" Shell.
                    } else {
                        Composite parent = proxy.fComposite;
                        proxy.fSafeHolder.attachToEclipseParent(parent);

                        // If focus was requested before this async message came in,
                        // OS may have since stolen the focus to Eclipse,
                        // now is the good time to give focus to eclipse component
                        if (proxy.fSetFocusWasRequested) {
                            proxy.setFocus();
                        }
                    }
                }
            });
        }
    }

    @Override
    protected void destroyConfirmed()
    {
        super.destroyConfirmed();
        CurlPlugin.getDefault().getWorkbench().getDisplay().asyncExec(new Runnable(){
            @Override
            public void run()
            {
                fSafeHolder.dispose();
                fSafeHolder = null;
            }});
    }

    /**
     * TODO: refactor to get rid of this
     * @param workbenchWindow 
     * @param composite
     */
    public void createRemoteView(IWorkbenchWindow workbenchWindow, Composite composite)
    {
        throw new RuntimeException("Should be overriden by subclass=" + getClass()); //$NON-NLS-1$
    }
}
