/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.DataOutputStream;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.util.CoreUtil;

public abstract class Proxy implements IProxy
{
    protected final MediatorConnection fMediatorConnection;
    protected Object fProxyOf;
    protected final int fProxyID;
    private boolean fDestroyed;

    /**
     * Process destroy confirmation sent by Mediator
     */
    static
    {
        EclipseServer.register(MediatorConnection.commandDestroyedConfirmed, new DestroyedConfirmedCommandHandler());
    }
    public static class DestroyedConfirmedCommandHandler extends ProxyCommandHandler<Proxy, NullProxyCommand>
    {
        @Override
        void execute(
                final NullProxyCommand command,
                final Proxy proxy,
                DataOutputStream sockOut)
        {
            if (! proxy.isDestroyed()) {
                CoreUtil.logWarning("Unsollicited destroy confirmation received, proxy=" + proxy.getClass() //$NON-NLS-1$
                        + ", id=" + proxy.fProxyID); //$NON-NLS-1$
            }
            proxy.destroyConfirmed();
            ProxyFactory.removeProxy(proxy.fProxyID);
        }
    }            

    protected void destroyConfirmed()
    {
        // nothing special to do
    }
    
    protected Proxy(
            Object proxyOf)
    {
        fMediatorConnection = CurlPlugin.getDefault().getMediatorConnection();
        fProxyOf = proxyOf;
        fProxyID = ProxyFactory.addProxy(this);
    }

    public boolean isDestroyed()
    {
        return fDestroyed;
    }
    
    public Object getProxyOf()
    {
        return fProxyOf;
    }

    public int getProxyID()
    {
        return fProxyID;
    }

    public void setProxyOf(
            Object o)
    {
        fProxyOf = o;
    }

    @Override
    protected void finalize() throws Throwable
    {
        if (! fDestroyed) {
            CoreUtil.logWarning("Proxy goes out of scope, but was never explicitely destroyed."); //$NON-NLS-1$
            destroy();
        }
    }
    
    public void destroy()
    {
        fDestroyed = true;
        asynchronousNoArgsOperation(MediatorConnection.commandDestroyAgent);
    }

    protected void sendDoOperationModal(
            final int operation)
    {
        // FIXME:  This should be turned into a dialog like "new file" or "new project"
        fMediatorConnection.sync.executeModal(new SynchronousRemoteOperation(MediatorConnection.commandDoOperation) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(operation);
            }

            @Override
            protected void receiveResponse()
            {
                // The command result is ignored, but we must read it to
                // block until the command is completed.
                readUnsignedByte();
            }
        });
    }

    protected void sendDoOperation(
            final int operation)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandDoOperation) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(operation);
            }
        });
    }

    protected void asynchronousNoArgsOperation(
            int commandCode)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(commandCode) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
            }
        });
    }
}
