/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.net.Socket;
import java.util.LinkedList;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicLong;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.operation.ModalContext;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlProperties;
import com.curl.eclipse.util.CurlUIIDs;
import com.ibm.icu.text.MessageFormat;

/**
 * Manage life-cycle and operations defined on the mediator.
 * Clients are expected to call getMediatorConnection to get a mediator
 * connection.  mediator connection is a singleton.
 */
public class MediatorConnection
{
    /**
     * Commands to the mediator & accepted by the eclipse server.
     * Any changes here MUST ALSO be reflected in the server!
     */

    // Commands from the plugin to the mediator
    static final int commandDone = 0;
    static final int commandMakeEditorView = 1;
    static final int commandMakeEditorOutlineView = 2;
    static final int commandMakeProjectCreationWizardView = 3;
    static final int commandGetProjectVersionInfo = 4;
    static final int commandMakeFileWizardView = 5;
    static final int subcommandForPackage = 1;
    static final int subcommandForSourceFile = 2;
    static final int commandLoadFileIntoEditor = 6;
    static final int commandResizeView = 7;
    static final int commandDisassociateEditorFromOutline = 8;
    static final int commandShowOutlineView = 9;
    static final int commandDoOperation = 10;
    static final int commandSetProjectVersionInfo = 11;
    static final int commandSetFocus = 12;
    static final int commandReleaseFocus = 13;
    static final int commandHideOutlineView = 14;
    static final int commandSetFocusSomeEclipseView = 15;
    static final int commandEditorNavigate = 16;
    static final int commandAutoComplete = 17;

    static final int commandReplaceFileBufferContent = 18;
    static final int commandEditorNavigateRowColumn = 19;
//    static final int unused = 30;
    static final int commandTerminateApplet = 31;
    static final int commandSuspendApplet = 32;
    static final int commandMakeConsoleAgent = 33;
    static final int commandRemoveConsoleAgent = 34;
    static final int commandMakeDefinitionsView = 40;
    static final int commandMakeManifestExplorerView = 41;
    static final int commandMakeStyleSheetProjectPropertiesView = 49;
    static final int commandMakeGeneralPreferencesView = 50;
    static final int commandMakeEditorPreferencesView = 51;
    static final int commandMakeGeneralProjectPropertiesView = 52;
    static final int commandMakeTargetsProjectPropertiesView = 53;
    static final int commandMakeDebugSessionDataProxy = 54;
    static final int comamndMakeLicensePreferencesView = 55;
    static final int commandPerformDialogCommit = 59;
    static final int commandPerformDialogValidation = 60;
    static final int commandCreateResource = 61;
    static final int commandResumeApplet = 62;
    public static final int commandStepOver = 63;
    public static final int commandStepInto = 64;
    public static final int commandStepOut = 65;
    static final int commandCheckVersion = 66;
    static final int commandGetStackVars = 67;
    static final int commandExpandVars = 68;
    static final int commandEvaluateExpression = 69;
    static final int commandShowDocumentationLanguage = 94;
    static final int commandShowInClassBrowser = 95;
    static final int commandPerformQueryDefinitions = 120;
    static final int commandPerformCancelQueryDefinitions = 121;
    static final int commandCreateDocumentAgent = 136;
    static final int commandDestroyAgent = 137;
    public static int commandCreateWorkbenchWindowAgent = 142;
    public static int commandWindowActivated = 143;
    public static int commandWindowDeactivated = 144;
    
    // Commands accepted by the eclipse server.
    static final int commandCompleted = 70;
    static final int commandOpenEditorOnUrl = 71;
    public static final int commandEnableWorkspaceAction = 72;
    static final int commandClientLimitExceeded = 73;
    static final int commandStringInserted = 74;
    static final int commandCharacterDeleted = 75;
    static final int commandDebugAppletTerminated = 76;
    static final int commandDebugLineBreakpointsChanged = 77;
    static final int commandOnPointerClick = 78;
    static final int commandDeclareBufferClean = 79;
    static final int commandDebugAppletStarted = 80;
    static final int commandDebugBreakpointHit = 81;
    public static final int subcommandDebugReasonStoppedAtBreakpoint = 1;
    public static final int subcommandDebugReasonStoppedAtException = 2;
    public static final int subcommandDebugReasonFinishedStepping = 3;
    static final int commandDebugResumed = 82;
    static final int commandHandleKeyPress = 83;
    static final int commandReportProblem = 84;
    public static final int commandReportSyntaxErrorSeverityError = 1;
    public static final int commandReportSyntaxErrorSeverityWarning = 2;
    public static final int commandReportSyntaxErrorSeverityInfo = 3;
    static final int commandClearProblems = 85;
    static final int commandEditorSelectionChanged = 86;
    static final int commandcheckOut = 87;
    static final int commandNotifyCommandChanged = 88;
    static final int commandQueryDefinitionResults = 89;
    static final int commandReadOpen = 90;
    static final int commandBufferResynced = 91;
    static final int commandConsoleOutput = 92;
    public static final int commandOpenProjectFailed = 93;
    public static final int commandProjectTargetsChanged = 119;
    static final int commandEcho = 123;
    public static final int commandSetTools = 126;
    static final int commandVersionMismatch = 135;
    public static final int commandOnViewCreated = 134;
    public static final int commandDestroyedConfirmed = 138;

    // Commands from the plugin to the mediator but for no particular agent
    public static final int commandCloseProjectAction = 100;
    public static final int commandOpenProjectAction = 101;
//    public static final int unused = 102;
//    static final int unused = 103;
//    static final int unused = 104;

    static final int commandLaunchAppletForDebugging = 106;
    static final int commandAddLineBreakpoint = 108;
    static final int commandRemoveLineBreakpoint = 109;
    static final int commandUpdateLineBreakpointEnableState = 110;
    static final int commandAddExceptionBreakpoint = 111;
    static final int commandUpdateExceptionBreakpointEnableState = 112;
    static final int commandRemoveExceptionBreakpoint = 113;
    static final int commandGetStandardExceptionBreakpointNames = 114;
    static final int commandGetCurlStartFiles = 115;
    static final int commandIsFileInProject = 116;
    static final int commandCanRun = 117;
    static final int commandWorkbenchWindowSelectionChanged = 118;
    public static final int commandSetCurrentTarget = 122;
    public static final int commandNotifySkipAllBreakpoints = 124;
    public static final int commandLaunchToolForId = 127;
    public static final int commandCreateWorkbenchAgent = 128;

//    public static final int commandShowContextMenu = 129;
    public static final int commandDestroyWorkbenchAgent = 130;
    public static final int commandSetHighlightLine = 131;
//    public static final int commandUnsetHighlightLine = 132;
    public static final int commandSelectionChanged = 133;
    public static int commandQueryRemoteSource = 139;
    public static int commandQueryRemoteSourceResponse = 140;
    public static int commandOnContextMenuEventResponse = 141;
    public static int commandOnCurlFocus = 145;
    public static int commandSavedURL = 146;
    public static int commandOnStatusBarMessage = 150;
    public static int commandRaiseActiveWindow = 151;
    
    // Commands common to both servers
    static final int commandSuccess = 200;
    static final int commandError = 201;
    static final int commandYes = 202;
    static final int commandNo = 203;

    // To differentiate between version mismatch states
    public enum VersionMismatchStates {NOT_YET_DETERMINED, VERSIONS_NOT_MATCHED, VERSIONS_MATCHED}
    
    private final CommandExecutor fCommandExecutor; 
    public final Sync sync;
    public final Async async;
    
    private Socket fSock;
    private DataOutputStream fSockOut;
    private DataInputStream fSockIn;
    private String fIPName;
    private EclipseServer fEclipseServer;
    private Process fMediatorProcess;

    /**
     * Serializes sending all commands to be executed by the mediator.  By doing
     * so, there is no need for calling threads to synchronize between themselves. 
     * Show a progress monitor in the status line area for as long as a command is
     * being executed by the mediator.
     */
    private static final class CommandExecutor
    {
        /**
         * Monitors mediator's progress in the status line area
         */
        private class MediatorMonitorJob extends Job {
            
            public MediatorMonitorJob() {
                super(RemoteMessages.MediatorProcessing);
                setSystem(false);
                setUser(false);
            }
            
            /**
             * Polls the outstanding commands to be executed by the mediator and when down to zero
             * it simply returns.  In doing so, it keeps the Job alive until there is no more
             * work to be done by the mediator and shows the progress in the status line area.
             * TODO:  We could use sleep() to avoid restarting this job. 
             */
            @Override
            public IStatus run(
                    IProgressMonitor monitor)
            {
                while (fCommandsToBeExecuted > 0) {
                    if (monitor.isCanceled()) {
                        setName(RemoteMessages.MediatorProcessingCannotBeCancelled);
                    }
                    try {
                        Thread.sleep(500);
                    } catch (Exception e) {
                        // Nothing
                    }
                }
                return Status.OK_STATUS;
            }            
         }

        /** A single thread executor that serializes sending commands to the mediator */
        private final ExecutorService fSingleThreadExecutor;
        /** Counter for outstanding mediator commands */
        private int fCommandsToBeExecuted;
        /** To monitor completion of the mediator command.  */
        private final MediatorMonitorJob fMediatorMonitorJob;
        
        /** Note that multiple threads will access this list. */
        private final LinkedList<Long> pendingTransactionIds = new LinkedList<Long>();
        
        private final AtomicLong fTransactionCounter = new AtomicLong();
        
        private CommandExecutor()
        {
            fSingleThreadExecutor = Executors.newSingleThreadExecutor();
            fMediatorMonitorJob = new MediatorMonitorJob();            
        }
        /**
         * Execute the runnable and monitor its completion
         */
        private void executeCommand(long transactionId, 
                final boolean show_progressbar, final Runnable runnable)
        {
            pendingTransactionIds.add(transactionId);

            fSingleThreadExecutor.execute(new Runnable() {
                public void run()
                {
                   fCommandsToBeExecuted++;
                   if (show_progressbar) {
                       fMediatorMonitorJob.schedule(750);
                   }
                   runnable.run();
                }
            });
                
            // for testing... do not delete!
/*            new Thread() {
                @Override
                public void run()
                {
                    try {Thread.sleep(5000);} catch (Exception e) {};
                    commandFinishedExecuting();
                }
            }.start();
*/
            }

        /**
         * Mediator has finished executing the command we sent it.
         * @param txId 
         */
        private void commandFinishedExecuting(long txId)
        {
            /*
             * Dispose head of queue of pending transaction identifiers.
             */
            Long dequeuedTxId;
            synchronized (pendingTransactionIds) {
                dequeuedTxId = pendingTransactionIds.poll();
            }
            if (dequeuedTxId == null) {
                CoreUtil.logError("No transaction pending while ack received for=" + txId); //$NON-NLS-1$
            } else {
                if (dequeuedTxId.longValue() != txId) {
                    CoreUtil.logError("Received ack for txid="  + txId + ", while expecting txid=" + dequeuedTxId); //$NON-NLS-1$ //$NON-NLS-2$
                }
            }
            
            /*
             * manage progress monitor
             */
            fSingleThreadExecutor.execute(new Runnable() {
                public void run()
                {
                    fCommandsToBeExecuted--;
                    
                    if (fCommandsToBeExecuted < 0) {
                        CoreUtil.logError("Outstanding mediator commands should never be negative!"); //$NON-NLS-1$
                        fCommandsToBeExecuted = 0;
                    }
                    
                    if (fCommandsToBeExecuted == 0) {
                        // Perhaps the job is waiting (according to its delay period) to be scheduled.
                        // Tell it to not bother running.
                        fMediatorMonitorJob.cancel();
                    }                    
                }
            });
        }
        private long lastEnqueuedTransactionId()
        {
            synchronized (pendingTransactionIds) {
                if (! pendingTransactionIds.isEmpty()) {
                    return pendingTransactionIds.getLast().longValue();
                }
                return -1L;
            }
        }
        private long allocateTransactionId()
        {
            return fTransactionCounter.getAndIncrement();
        }
    }
    
    /**
     * Provide a set of synchronous operations defined only for the SynchronousRemoteOperation class.
     * Strong argument can be made that these execute operations should really be part of the
     * SynchronousRemoteOperation class itself.
     *
     * To operate safely in the UI thread, operations that do support calls from the UI thread always
     * operate modal and always process the event loop.  Hence, they do not allow user input from any
     * view but do process any other window event.  The assumption is that this guards against any UI
     * subsystem reentry while allowing non-queued events to be processed.
     * FIXME:  This requires further investigation.
     */
    public class Sync
    {
        /**
         * Execute the synchronous operation. Must never be called from the UI thread.
         */
        public void execute(
                final SynchronousRemoteOperation oper)
        {
            if (CoreUtil.amITheUIThread()) {
                CoreUtil.logError("execute called when in UI thread"); //$NON-NLS-1$
                return;
            }

            runQueuedAndWait(
                    oper.fTransactionId,
                    new Runnable() {
                        public void run()
                        {
                            oper.sendReceive();
                        }
                    });
        }

        /**
         * Execute the synchronous operation. Must be called from the UI thread.
         * It processes the UI event loop with a Busy cursor and modal and
         * progress bar.
         */
        public void executeBusyModalWithProgressBar(
                final SynchronousRemoteOperation oper)
        {
            if (!CoreUtil.amITheUIThread()) {
                CoreUtil.logError("executeBusyWithProgressBar called when not in UI thread"); //$NON-NLS-1$
                return;
            }

            runQueuedBusyWhile(
                    oper.fTransactionId,
                    new Runnable() {
                        public void run()
                        {
                            oper.sendReceive();
                        }
                    });
        }

        /**
         * Execute the synchronous operation modal. Must be called from the UI thread.
         */
        public void executeModal(
                final SynchronousRemoteOperation oper)
        {
            if (!CoreUtil.amITheUIThread()) {
                CoreUtil.logError("executeModal called when not in UI thread"); //$NON-NLS-1$
                return;
            }

            runQueuedModalWithEventLoop(
                    oper.fTransactionId,
                    new Runnable() {
                        public void run()
                        {
                            oper.sendReceive();
                        }});
        }

        /**
         * Execute the synchronous operation when we are already in modal context.
         * Must be called from the UI thread.
         */
        public void executeInModalContext(
                final SynchronousRemoteOperation oper)
        {
            if (!CoreUtil.amITheUIThread()) {
                CoreUtil.logError("executeInModalContext called when not in UI thread"); //$NON-NLS-1$
                return;
            }

            try {
                ModalContext.run(
                    new IRunnableWithProgress() {
                        public void run(
                            IProgressMonitor pm) throws InvocationTargetException
                        {
                            runQueuedAndWait(
                                    oper.fTransactionId,
                                    new Runnable() {
                                        public void run() {
                                            oper.sendReceive();
                                        }
                                    });
                        }
                    },
                    true,
                    new NullProgressMonitor(),
                    PlatformUI.getWorkbench().getDisplay());
            } catch (Exception e) {
                CoreUtil.logError("While setting up for modal context", e); //$NON-NLS-1$
            }
        }

        /**
         * Queue the runnable for our single worker thread while modal and processing the
         * event loop with busy cursor.  When the worker thread completes the runnable,
         * then get out of the event loop and return.
         *
         * Note:  This method ignores a Cancel from the progress monitor.
         */
        private void runQueuedBusyWhile(
                final long transactionId,
                final Runnable runnable)
        {
            try {
                PlatformUI.getWorkbench().getProgressService().busyCursorWhile(new IRunnableWithProgress() {
                    public void run(
                            IProgressMonitor monitor) throws InvocationTargetException, InterruptedException
                    {
                        runQueuedAndWait(
                                transactionId,
                                new Runnable() {
                                    public void run()
                                    {
                                        runnable.run();
                                    }
                                });
                    }
                });
            } catch (InterruptedException e) {
            } catch (InvocationTargetException e) {
                CoreUtil.logError("While invoking a runnable", e); //$NON-NLS-1$
            }
        }

        /**
         * Queue the runnable and wait for the runnable to complete, while processing the
         * UI event loop in modal state.  Must be invoked from the UI thread.
         * @param transactionId 
         */
        private void runQueuedModalWithEventLoop(
                long transactionId,
                final Runnable runnable)
        {
            setModal(true);
            try {
                runQueuedWithEventLoop(transactionId, runnable);
            } finally {
                setModal(false);
            }
        }

        /**
         * Queue the runnable and wait for it to return.
         * @param transactionId 
         */
        private void runQueuedAndWait(
                long transactionId,
                final Runnable runnable)
        {
            final Object monitor = new Object();
            final boolean done[] = new boolean[1];
            fCommandExecutor.executeCommand(transactionId, true, new Runnable(){
                public void run()
                {
                    try {
                        runnable.run();
                    } catch (Exception e) {
                        CoreUtil.logError("Error invoking runnable", e); //$NON-NLS-1$
                    } finally {
                        synchronized (monitor) {
                            try {
                                done[0] = true;
                                monitor.notifyAll();
                            } catch (IllegalMonitorStateException e) {
                                CoreUtil.logError("Must own the monitor!", e);  //$NON-NLS-1$
                            }
                        }
                    }
                }
            });

            synchronized (monitor) {
                while (!done[0]) {
                    try {
                        monitor.wait();
                    } catch (InterruptedException e) {
                        // Do nothing...
                    } catch (IllegalMonitorStateException e) {
                        CoreUtil.logError("Must own the monitor!", e);  //$NON-NLS-1$
                    }
                }
            }
        }

        private void runQueuedWithEventLoop(
                long transactionId,
                final Runnable runnable)
        {
            final boolean done[] = new boolean[1];
            final Display display = Display.getCurrent();
            fCommandExecutor.executeCommand(transactionId, false, new Runnable() {
                public void run()
                {
                    try {
                        runnable.run();
                    } catch (Exception e) {
                        CoreUtil.logError("Error invoking runnable", e); //$NON-NLS-1$
                    } finally {
                        done[0] = true;
                    }
                }
            });

            while (!done[0]) {
                if (!display.readAndDispatch())
                    display.sleep();
            }
        }

        private void setModal(boolean makeModal)
        {
            final Shell[] shells = PlatformUI.getWorkbench().getDisplay().getShells();

            if (makeModal) {
                for (Shell element : shells) {
                    element.setEnabled(false);
                }
            } else {
                for (int i = shells.length - 1; i >= 0; i--) {
                    shells[i].setEnabled(true);
                }
            }
        }
    }

    /**
     * Provide a set of asynchronous operations defined only for the AsynchronousRemoteOperation class.
     * Strong argument can be made that these execute operations should really be part of the
     * AsynchronousRemoteOperation class itself.
     */
    public class Async
    {
        /**
         * Execute the asynchronous operation.  The operation is executed serially relative to all
         * other synchronous and asynchronous operations.  Asynchronous operations never wait for
         * data back from the mediator, they send the data and immediately continue.
         */
        public void execute(final AsynchronousRemoteOperation oper)
        {
            fCommandExecutor.executeCommand(oper.fTransactionId, true, new Runnable() {
                public void run()
                {
                    if (getEclipseServer() == null || !getEclipseServer().isServerRunning())
                        // Server is not running... nothing to do
                        return;
            
                    try {
                        oper.sendArguments();
                    } catch (Exception e) {
                        if (!CurlPlugin.getDefault().isStoppedOrStopping()) {
                            CoreUtil.logError("While executing queued mediator operation", e); //$NON-NLS-1$
                        }
                    }
                }
            });
        }
    }

    /**
     * Must be called from plugin start.  Note that this method is not synchronized and depends on the
     * fact that plugin start is not reentered.
     * 
     * @throws Exception
     */
    public MediatorConnection() throws Exception
    {
        fCommandExecutor = new CommandExecutor(); 
        async = new Async();
        sync = new Sync();
        launchMediatorAndStartEclipseServer();
    }
    
    /**
     * Check (validate) the plugin version against the mediator version
     * Note:  This call must be made in the plugin startup immediately after IPC connections
     * between the plugin and the mediator are established and before any commands are exchanged.
     */
    public void versionsMatch() throws Exception
    {
        final String mediatorVersionPluginWasBuiltWith = CurlProperties.getProperty("version", ""); //$NON-NLS-1$ //$NON-NLS-2$
        
        // Tell mediator to check for version consistency then poll for answer
        async.execute(
                new AsynchronousRemoteOperation(MediatorConnection.commandCheckVersion){
            @Override
            protected void writeArguments()
            {
                write(mediatorVersionPluginWasBuiltWith);
            }
        });

        // Poll for version check answer
        final int tryThisManyTimes = 10;
        final int waitThisMuchBetweenTries = 500;
        boolean versionsMatch = false;
        boolean gaveUp = true;
        int tryAgain = tryThisManyTimes;
        while (tryAgain >= 0) {
            switch (fEclipseServer.getPluginAndMediatorVersionCheckState()) {
            case NOT_YET_DETERMINED:
                if (tryAgain > 0) {
                    CoreUtil.logInfo("Waiting for mediator to validate version"); //$NON-NLS-1$
                    try {
                        Thread.sleep(waitThisMuchBetweenTries);
                    } catch (Exception e1) {
                    }
                }
                break;
            case VERSIONS_MATCHED:
                CoreUtil.logInfo("Versions matched"); //$NON-NLS-1$
                versionsMatch = true;
                gaveUp = false;
                tryAgain = 0;
                break;
            case VERSIONS_NOT_MATCHED:
                CoreUtil.logInfo("Versions did not match!"); //$NON-NLS-1$
                gaveUp = false;
                tryAgain = 0;
                break;                    
            }
            tryAgain--;
        }

        // Now that we have the answer on version check, give a message
        // if there is a mismatch.
        if (!versionsMatch) {
            String curlMessage;
            if (gaveUp)
                curlMessage = RemoteMessages.VersionsMatchGaveUp;
            else
                curlMessage = MessageFormat.format(
                        RemoteMessages.VersionsNotMatched,
                        new String[] { mediatorVersionPluginWasBuiltWith, fEclipseServer.getMediatorVersion() });
 
            Status status = new Status(IStatus.ERROR, CurlUIIDs.ID_PLUGIN, IStatus.OK, curlMessage, null);
            CoreUtil.logStatus(status);
            ErrorDialog error = new ErrorDialog(
                    CoreUtil.getActiveWorkbenchShell(),
                    RemoteMessages.MediatorNotAvailable_Title,
                    RemoteMessages.MediatorNotAvailable_Message4,
                    status,
                    IStatus.ERROR);
            error.open();
            closeRemoteConnections();
            throwMediatorNotAvailable();
        }
    }

    /**
     * Tear down the IPC.  Must be called during plugin stop.  Note that this method is not
     * synchronized and depends on plugin stop to not reenter.
     */
    public void closeRemoteConnections()
    {
        if (fEclipseServer != null) {
            fEclipseServer.stopServer();
            fEclipseServer = null;
        }
        
        if (fMediatorProcess != null) {
            try {
                if (fSockOut != null) {
                    fSockOut.close();
                    fSockOut = null;
                }
                if (fSockIn != null) {
                    fSockIn.close();
                    fSockIn = null;
                }
                if (fSock != null) {
                    fSock.close();
                    fSock = null;
                }
            } catch (IOException e) {
                CoreUtil.logError("Error closing connection to " + fIPName, e); //$NON-NLS-1$
            }
            
            fMediatorProcess.destroy();
            fMediatorProcess = null;
        }
    }

    public DataOutputStream getOutputStream()
    {
        return fSockOut;
    }

    public DataInputStream getInputStream()
    {
        return fSockIn;
    }

    public EclipseServer getEclipseServer()
    {
        return fEclipseServer;
    }
    
    public void mediatorCommandCompleted(long txId)
    {
        fCommandExecutor.commandFinishedExecuting(txId);
    }

    /**
     * Start the IPC:  Launch the mediator and start the eclipse server.  Make sure they are operational.
     * Throw an exception if they are not.
     * @throws Exception
     */
    @SuppressWarnings("null")
    private void launchMediatorAndStartEclipseServer() throws Exception
    {
        // Launch the mediator asynchronously.  This takes a while so do it first.
        Ports ports = launchMediator();
        if (ports == null) {
            throwMediatorNotAvailable();
        }
        
        // Start the Eclipse server and wait for it to accept connections.  We have to wait because
        // the mediator does not wait for this socket to be available.
        fEclipseServer = new EclipseServer(ports.fEclipseServerPort);
        while (!fEclipseServer.isServerRunning() && !fEclipseServer.hasFailed()) {
            CoreUtil.logInfo("Waiting for Eclipse server to start..."); //$NON-NLS-1$
            try {
                Thread.sleep(1000);
            } catch (Exception e) {
            }
        }

        // Now that the eclipse server is accepting connections, open a connection to the mediator.  
        // The mediator startup code will connect the mediator to the eclipse server.  
        if (!openMediatorConnection(DataIO.IPName, ports.fMediatorPort)) {
            throwMediatorNotAvailable();
        }

        CoreUtil.logInfo("Successfully connected to the mediator"); //$NON-NLS-1$
    }

    /**
     * Launch the mediator asynchronously according to the property file.  If the launcher
     * specified in the property file cannot be found, do not attempt to launch the mediator.
     * This takes care of the case that the user has uninstalled the IDE without uninstalling
     * the CDE plugin.
     */
    private Ports launchMediator()
    {        
        Ports ports = new Ports();
        ports.fMediatorPort = CurlPlugin.findFreePort();
        ports.fEclipseServerPort = CurlPlugin.findFreePort();
        
        String runType = CurlProperties.getProperty("RUN", "std"); //$NON-NLS-1$ //$NON-NLS-2$
        String launcherPath = CurlProperties.getProperty("CurlDirectory") +  //$NON-NLS-1$
                              CurlProperties.getProperty(runType + ".LauncherExecutable"); //$NON-NLS-1$
        String commandLine = launcherPath + " " + //$NON-NLS-1$
                             String.format(
                                     CurlProperties.getProperty(runType + ".CommandLine"), //$NON-NLS-1$
                                     ports.fMediatorPort,
                                     ports.fEclipseServerPort);
        
        // Make sure the launcher exists
        File fileWithExe = new File(launcherPath + ".exe"); //$NON-NLS-1$
        File filePlain = new File(launcherPath);
        if (!fileWithExe.exists() && !filePlain.exists()) {
            Status status = new Status(
                    IStatus.ERROR,
                    CurlUIIDs.ID_PLUGIN,
                    IStatus.OK,
                    MessageFormat.format(
                            RemoteMessages.LauncherNotFound,
                            new String[] { launcherPath }),
                    null);
            CoreUtil.logStatus(status);
            ErrorDialog error = new ErrorDialog(
                    CoreUtil.getActiveWorkbenchShell(),
                    RemoteMessages.MediatorNotAvailable_Title,
                    RemoteMessages.MediatorNotAvailable_Message1,
                    status,
                    IStatus.ERROR);
            error.open();
            return null;
        }

        // Launch it
        try {
            fMediatorProcess = Runtime.getRuntime().exec(commandLine, null, null);
            // to avoid curl RTE hanging when it displays a long stack trace (when it panics)
            fMediatorProcess.getErrorStream().close();
            CoreUtil.logInfo("Mediator launched"); //$NON-NLS-1$
        } catch (Exception e) {
            CoreUtil.logError("Launching mediator failed", e); //$NON-NLS-1$
            return null;
        }
        
        return ports;
    }

    /**
     * Try to open a connection to the mediator.  Try only so many times and if the mediator
     * is not ready to accept a connection withing an aloted time then give up.
     * @param name
     * @param port
     * @return
     */
    private boolean openMediatorConnection(
            String name,
            int port)
    {
        final int tryThisManyTimes = 
            Integer.valueOf(CurlProperties.getProperty("TRIES-TO-CONNECT-TO-MEDIATOR", "60")).intValue(); //$NON-NLS-1$ //$NON-NLS-2$
        final int waitThisLongBetweenTries = 1000;
        fIPName = name;
        int tryAgain = tryThisManyTimes;
        while (tryAgain >= 0) {
            try {
                fSock = new Socket(name, port);
                tryAgain = -1;
            } catch (java.net.ConnectException e) {

                // Give up immediately if mediator has told us that the client limit has exceeded
                if (fEclipseServer.isClientLimitExceeded()) {
                    popupDialog(
                            RemoteMessages.ClientLimitExceeded,
                            RemoteMessages.MediatorNotAvailable_Message3,
                            "Curl Runtime refused second connection", //$NON-NLS-1$
                            null);
                    return false;
                }

                // Give up if exhausted the number of times we should try to connect
                if (tryAgain == 0) {
                    popupDialog(
                            RemoteMessages.ConnectionFailed1,
                            RemoteMessages.MediatorNotAvailable_Message5,
                            "Curl Runtime is still not up, giving up", //$NON-NLS-1$
                            e);
                    return false;
                }

                // Keep trying to connect...
                CoreUtil.logInfo("Trying to connect to the mediator..."); //$NON-NLS-1$
                try {Thread.sleep(waitThisLongBetweenTries);} catch (Exception e1) {}
                tryAgain--;
                
            } catch (Exception e) {
                popupDialog(
                        RemoteMessages.ConnectionFailed1,
                        RemoteMessages.MediatorNotAvailable_Message1,
                        "Could not open connection to mediator", //$NON-NLS-1$
                        e);
                return false;
            }
        }
        
        // If succeeded in opening a connection to the mediator, then open the streams...
        if (fSock != null) {
            try {
                BufferedOutputStream bos = new BufferedOutputStream(fSock.getOutputStream());
                fSockOut = new DataOutputStream(bos);
                BufferedInputStream bis = new BufferedInputStream(fSock.getInputStream());
                fSockIn = new DataInputStream(bis);
            } catch (Exception e) {
                popupDialog(
                        RemoteMessages.ConnectionFailed1,
                        RemoteMessages.MediatorNotAvailable_Message1,
                        "Could not open stream for connection to mediator", //$NON-NLS-1$
                        e);
                return false;
            }
        } else {
            // Not sure when this can happen
            CoreUtil.logError("Socket is still null");         //$NON-NLS-1$
            return false;
        }

        return true;
    }

    private void popupDialog(
            String error,
            String message,
            String log,
            Exception excp)
    {
        CoreUtil.logError(log, excp);        
        Status status = new Status(
                IStatus.ERROR,
                CurlUIIDs.ID_PLUGIN,
                IStatus.OK,
                error,
                excp);
        ErrorDialog dialog = new ErrorDialog(
                CoreUtil.getActiveWorkbenchShell(),
                RemoteMessages.MediatorNotAvailable_Title,
                message,
                status,
                IStatus.ERROR);
        dialog.open();
    }
    
    /**
     * Used to return or pass the ports as argument.
     */
    private static class Ports
    {
        int fMediatorPort;
        int fEclipseServerPort;
    }
    
    private void throwMediatorNotAvailable() throws Exception
    {
        throw new Exception(RemoteMessages.MediatorNotAvailable_Message1);        
    }

    long lastEnqueuedTransactionId()
    {
        return fCommandExecutor.lastEnqueuedTransactionId();
    }

    long allocateTransactionId()
    {
        return fCommandExecutor.allocateTransactionId();
    }
}
