/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.ui.PlatformUI;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.debug.CurlAppletLaunchDelegate;
import com.curl.eclipse.debug.ManageBreakpoints;
import com.curl.eclipse.debug.ManageProblems;
import com.curl.eclipse.remote.INotifyCommandChangedProxy.NotifyCommandChangedCommandHandler;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlMediatorEvent;
import com.curl.eclipse.util.CurlUIIDs;

/*
 * Eclipse server listens for mediator messages through its input socket
 * and talks back through its output socket.  There is only one eclipse server
 * for a Curl plugin.
 */
public class EclipseServer extends Thread
{
    private static class DifferedCommand
    {
        private final int fAgentId;
        private final Object fCommand;
        private final int fCommandId;
        private final ProxyCommandHandler<IProxy,Object> fProxyCommandHandler;
        private long fTxId;

        public DifferedCommand(
                int agentId,
                Object proxyCommand,
                int commandId, 
                ProxyCommandHandler<IProxy,Object> proxyCommandHandler,
                long txId)
        {
            fAgentId = agentId;
            fCommand = proxyCommand;
            fCommandId = commandId;
            fProxyCommandHandler = proxyCommandHandler;
            fTxId = txId;
        }
    }

    private Boolean fHasFailed = false;
    private ServerSocket fServerSock;
    private Socket fClientSock;
    private DataOutputStream fSockOut;
    private DataInputStream fSockIn;
    private boolean fPollForCommands;
    private boolean fServerIsRunning;
    private boolean fClientLimitExceeded;
    private MediatorConnection.VersionMismatchStates fVersionMismatchState;
    private String fMediatorVersion;
    private final int fEclipseServerPort;
    private final LinkedList<DifferedCommand> fDeferredCommands = new LinkedList<DifferedCommand>();
    
    private static final Map<Integer, ProxyCommandHandler<? extends IProxy,?>> fCommandIdToDecoderMap =
        new HashMap<Integer, ProxyCommandHandler<?,?>>();
    
    static void register(int command, ProxyCommandHandler<?, ?> decoder)
    {
        ProxyCommandHandler<?, ?> previous = fCommandIdToDecoderMap.put(command, decoder);
        if (previous != null) {
            CoreUtil.logError("Duplicate handler declaration for command " + command); //$NON-NLS-1$
        }
    }
    
    /**
     * TODO: commandNotifyCommandChanged is the only command shared by 2 different proxy implementations:
     * SourceViewProxy and WorkbenchProxy, and it can only be registered once.
     * Could we come up with something better
     */
    static
    {
        EclipseServer.register(MediatorConnection.commandNotifyCommandChanged, new NotifyCommandChangedCommandHandler());
    }

    EclipseServer(int eclipseServerPort)
    {
        super("Curl Eclipse Server"); //$NON-NLS-1$
        fEclipseServerPort = eclipseServerPort;
        start();
    }

    public void stopServer()
    {
        fPollForCommands = false;
        fServerIsRunning = false;
        closeConnection();
    }

    public boolean isServerRunning()
    {
        return fServerIsRunning;
    }

    public boolean isClientLimitExceeded()
    {
        return fClientLimitExceeded;
    }
    
    public MediatorConnection.VersionMismatchStates getPluginAndMediatorVersionCheckState()
    {
        return fVersionMismatchState;
    }

    public String getMediatorVersion()
    {
        return fMediatorVersion;
    }

    public boolean hasFailed()
    {
        return fHasFailed;
    }

    @Override
    public void run()
    {
        if (openConnection(fEclipseServerPort)) {
            dispatchServerRequests();
        }
        fServerIsRunning = false;
    }

    /**
     * Open a connection and wait for the mediator to connect.
     */
    private boolean openConnection(
            int port)
    {
        try {
            fServerSock = new ServerSocket(port);
            fClientLimitExceeded = false;
            fVersionMismatchState = MediatorConnection.VersionMismatchStates.NOT_YET_DETERMINED;
            fMediatorVersion = ""; //$NON-NLS-1$
            fServerIsRunning = true;
            fClientSock = fServerSock.accept();
            fSockOut = new DataOutputStream(new BufferedOutputStream(fClientSock.getOutputStream()));
            fSockIn = new DataInputStream(new BufferedInputStream(fClientSock.getInputStream()));
        } catch (Exception e) {
            CoreUtil.logError("opening connection failed", e); //$NON-NLS-1$
            fHasFailed = true;
            return false;
        }
        CoreUtil.logInfo("Eclipse server is running", null); //$NON-NLS-1$
        return true;
    }

    /**
     * Close connection to the server
     */
    private void closeConnection()
    {
        fPollForCommands = false;
        try {
            if (fSockOut != null) {
                fSockOut.close();
                fSockOut = null;
            }
            if (fSockIn != null) {
                fSockIn.close();
                fSockIn = null;
            }
            if (fClientSock != null) {
                fClientSock.close();
                fClientSock = null;
            }
            if (fServerSock != null) {
                fServerSock.close();
                fServerSock = null;
            }
        } catch (Exception e) {
            CoreUtil.logError("Closing connections failed", e); //$NON-NLS-1$
            fHasFailed = true;
        }
    }

    /**
     * Respond to echo request from mediator
     */
    private void echoResponse()
    {
        DataIO.writeInt(fSockOut, MediatorConnection.commandEcho);
        DataIO.flush(fSockOut);
    }

    /**
     * Report a problem
     */
    private void reportProblemResponse()
    {
        final String urlString = DataIO.readString(fSockIn);
        final int line = DataIO.readInt(fSockIn);
        final int column = DataIO.readInt(fSockIn);
        final int severity = DataIO.readInt(fSockIn);
        final String message = DataIO.readString(fSockIn);
        ManageProblems.addProblem(urlString, line, column, severity, message);
    }

    /**
     * clear all problems
     */
    private void clearProblemsResponse()
    {
        ManageProblems.clearProblems();
    }


    private void checkOutResponse()
    {
        int nb = DataIO.readInt(fSockIn);
        ArrayList<IFile> files = new ArrayList<IFile>();
        for (int i = 0; i < nb; i++) {
            String urlStr = DataIO.readString(fSockIn);
            IFile file = CoreUtil.getFile(urlStr);
            if (file != null) {
                files.add(file);
            }
        }
        final IFile[] fileArray = files.toArray(new IFile[files.size()]);
        boolean success = false;
        try {
            PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
                public void run()
                {
                    ResourcesPlugin.getWorkspace().validateEdit(fileArray, CoreUtil.getActiveWorkbenchShell());
                }
            });
            success = true;
        } finally {
            DataIO.writeBoolean(fSockOut, success);
            DataIO.flush(fSockOut);
        }
    }

    /**
     * Open the editor on this url
     */
    private void openEditorOnUrlResponse()
    {
        final String urlString = DataIO.readString(fSockIn);
        final int row = DataIO.readInt(fSockIn);
        final int col = DataIO.readInt(fSockIn);
        final int length = DataIO.readInt(fSockIn);
        final boolean activate = DataIO.readBoolean(fSockIn);
        final String toolID = DataIO.readString(fSockIn);
        PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
            public void run()
            {
                CoreUtil.openEditorOnUrl(CoreUtil.convertPathToEclipse(urlString), row - 1, col, length, activate, toolID);
            }
        });
    }

    /**
     * Enable the specified workspace action
     */
    private void enableWorkspaceResponse()
    {
        final String action = DataIO.readString(fSockIn);
        PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
            public void run()
            {
                CurlMediatorEvent event = new CurlMediatorEvent(
                        MediatorConnection.commandEnableWorkspaceAction,
                        action);
                CurlPlugin.getMediatorEventManager().fireEvent(event);
            }
        });
    }

    /**
     * Client limit is exceeded. We cannot use the mediator!
     */
    private void clientLimitExceededResponse()
    {
        fClientLimitExceeded = true;
    }

    /**
     * Mediator responded to tell us about a version mismatch
     */
    private void versionsMatchingResponse()
    {
        if (DataIO.readBoolean(fSockIn))
            fVersionMismatchState = MediatorConnection.VersionMismatchStates.VERSIONS_MATCHED;
        else
            fVersionMismatchState = MediatorConnection.VersionMismatchStates.VERSIONS_NOT_MATCHED;
        fMediatorVersion = DataIO.readString(fSockIn);
    }

    /**
     * We have to do this outside of the proxy because the mediator did not
     * have the applet-id to proxy-id hashmap setup in time.
     */
    // TODO: figure out the right syntax for ProxyCommandHandler<xxx> declaration
    @SuppressWarnings("unchecked")  
    private void delegateToDebugProxyResponse()
    {
        String appletID = DataIO.readString(fSockIn);
        Proxy proxy = ProxyFactory.getDebugProxyData(appletID);
        ProxyCommandHandler handler = fCommandIdToDecoderMap.get(MediatorConnection.commandDebugBreakpointHit);
        Object proxyCommand = handler.decode(fSockIn);

        if (proxy == null) {
            CoreUtil.logError("delegating to debug proxy cannot find the proxy"); //$NON-NLS-1$
        }
        else {
            handler.execute(proxyCommand, proxy, fSockOut);
        }
    }

    /**
     * Poll the socket and dispatch commands
     */
    // TODO: figure out the right syntax for ProxyCommandHandler<xxx> declaration
    @SuppressWarnings("unchecked")  
    private void dispatchServerRequests()
    {
        fPollForCommands = true;
        while (fPollForCommands) {
            int commandId;
            long transactionId;
            try {
                /** Skip any garbage before synchronization marker
                 * this may happen when purging command parameters for destroyed agent
                 * see note in  Proxy.destroy()
                 */
                byte[] seq = {(byte)0xCA, (byte)0xFE, (byte)0xBA, (byte)0xBE};
                int i = 0;
                int countSkippedChar = 0;
                while(i < seq.length) {
                    byte b = fSockIn.readByte();
                    if (b == seq[i]) {
                        i++;
                    }
                    else {
                        countSkippedChar++;
                    }
                }
                if (countSkippedChar > 0) {
                    CoreUtil.logInfo("EclipseServer, resynchronized stream, bytes skipped=" + countSkippedChar); //$NON-NLS-1$
                }
                commandId = fSockIn.readInt();
                transactionId = DataIO.readLong(fSockIn);

            } catch (Exception e) {
                if (fPollForCommands) {
                    CoreUtil.logError("Curl IDE died unexpectedly"); //$NON-NLS-1$
                    CoreUtil.displayError(RemoteMessages.EclipseServer_11,RemoteMessages.EclipseServer_12);
                }
                break;
            }
            try {
                switch (commandId) {
                //
                // TODO: refactor special cases to make them fit the generic
                // proxy/command pattern
                // Note that these special cases are never going to be deferred (like regular proxy commands),
                // which is just fine for now (the notion of transaction is simply ignored).
                //
                case MediatorConnection.commandEcho:
                    echoResponse();
                    break;
                case MediatorConnection.commandOpenEditorOnUrl:
                    openEditorOnUrlResponse();
                    break;
                case MediatorConnection.commandEnableWorkspaceAction:
                    enableWorkspaceResponse();
                    break;
                case MediatorConnection.commandClientLimitExceeded:
                    clientLimitExceededResponse();
                    break;
                case MediatorConnection.commandVersionMismatch:
                    versionsMatchingResponse();
                    break;
                case MediatorConnection.commandDebugAppletStarted:
                    CurlAppletLaunchDelegate.appletStartedResponse(fSockIn);
                    break;
                case MediatorConnection.commandDebugLineBreakpointsChanged:
                    ManageBreakpoints manageBreakpoints = CurlPlugin.getDefault().getManageBreakpoints();
                    manageBreakpoints.lineBreakpointsChangedResponse(fSockIn);
                    break;
                case MediatorConnection.commandDebugBreakpointHit:
                    delegateToDebugProxyResponse();
                    break;
                case MediatorConnection.commandReportProblem:
                    reportProblemResponse();
                    break;
                case MediatorConnection.commandClearProblems:
                    clearProblemsResponse();
                    break;
                case MediatorConnection.commandcheckOut:
                    checkOutResponse();
                    break;
                case MediatorConnection.commandReadOpen:
                    readOpenResponse();
                    break;
                case MediatorConnection.commandCompleted:
                    CurlPlugin.getDefault().getMediatorConnection().mediatorCommandCompleted(transactionId);
                    executeDeferredCommandsFor(transactionId);
                    break;
                default:
                    // This is where we recognize that the connection to the mediator
                    // has been lost. But, we have to be quiet if this is merely Eclipse exiting.
                    if (fPollForCommands) {
                        if (commandId == -1) {
                            final Status status = new Status(
                                    IStatus.ERROR,
                                    "com.curl.eclipse.plugin", //$NON-NLS-1$
                                    IStatus.OK,
                                    RemoteMessages.EclipseServer_14,
                                    null);
                            CoreUtil.logStatus(status);
                            PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
                                public void run()
                                {
                                    ErrorDialog error = new ErrorDialog(
                                            CoreUtil.getActiveWorkbenchShell(),
                                            RemoteMessages.MediatorNotAvailable_Title,
                                            RemoteMessages.MediatorNotAvailable_Message2,
                                            status,
                                            IStatus.ERROR);
                                    error.open();
                                }
                            });
                            // TODO: What we really need to do is to somehow stop
                            // the plugin all together. But how?
                            stopServer();
                        } else {
                            int agentId = DataIO.readInt(fSockIn);
                            ProxyCommandHandler handler = fCommandIdToDecoderMap.get(commandId);
                            
                            // Decode the command, regardless the corresponding proxy still exist,
                            // or is pending destroyed
                            Object proxyCommand = handler.decode(fSockIn);
                            
                            Proxy proxy = ProxyFactory.getProxy(agentId);
                            if (proxy == null) {
                                CoreUtil.logError("Mediator command is for non-existant proxy: " + agentId + ", command=" + commandId); //$NON-NLS-1$ //$NON-NLS-2$
                            }
                            else if (proxy.isDestroyed()) {
                                if (commandId == MediatorConnection.commandDestroyedConfirmed) {
                                    // destroy confirmed is the only command we must process in that case
                                    handler.execute(proxyCommand, proxy, fSockOut);
                                }
                                // else, just ignore proxyCommand.
                            }
                            else {
                                long lastEnqueuedTransactionId = CurlPlugin.getDefault().getMediatorConnection().lastEnqueuedTransactionId();
                                if (transactionId == -1 && lastEnqueuedTransactionId >= 0) {
                                    // Cannot execute this command immediately, we have to wait
                                    // until transaction with id [lastEnqueuedTransactionId] is acknowledged.
                                    fDeferredCommands.add(new DifferedCommand(agentId, proxyCommand, commandId, handler, lastEnqueuedTransactionId));
                                } else {
                                    // Normal case
                                    handler.execute(proxyCommand, proxy, fSockOut);
                                }
                            }
                        }
                    }
                    break;
                }
                if (fSockOut != null) {
                    fSockOut.flush();
                }
                else {
                    CoreUtil.logWarning("fSockout is null, command=" + commandId); //$NON-NLS-1$
                }
            } catch (Exception e) {
                CoreUtil.logError("Eclipse server caught exception but continuing...", e); //$NON-NLS-1$
            }
        }
    }
    
    private void executeDeferredCommandsFor(
            long txId)
    {
        while (! fDeferredCommands.isEmpty()) {
            DifferedCommand differedCommand = fDeferredCommands.element();
            if (differedCommand.fTxId <= txId) {
                fDeferredCommands.remove();
                
                // Time to execute differed command.
                if (differedCommand.fTxId < txId) {
                    CoreUtil.logError("Missed an ack, current ack=" + txId + ", missed id=" + differedCommand.fTxId); //$NON-NLS-1$ //$NON-NLS-2$
                }
                Proxy proxy = ProxyFactory.getProxy(differedCommand.fAgentId);
                if (proxy == null) {
                    // Deferred command for previously disposed proxy
                } else if (proxy.isDestroyed()) {
                    if (differedCommand.fCommandId == MediatorConnection.commandDestroyedConfirmed) {
                        // destroy confirmed is the only command we must process in that case
                        differedCommand.fProxyCommandHandler.execute(differedCommand.fCommand, proxy, fSockOut);
                    }
                    // else, just ignore deferred proxyCommand.
                } else {
                    differedCommand.fProxyCommandHandler.execute(differedCommand.fCommand, proxy, fSockOut);
                    
                    // above statement may have indirectly enqueued new transactions, hence we must
                    // adjust the expected transaction id acknowledgments for all currently deferred commands
                    long lastEnqueuedTransactionId = CurlPlugin.getDefault().getMediatorConnection().lastEnqueuedTransactionId();
                    if (lastEnqueuedTransactionId != -1 && lastEnqueuedTransactionId != txId) {
                        for(DifferedCommand dc : fDeferredCommands) {
                            dc.fTxId = lastEnqueuedTransactionId;
                        }
                        break;
                    }
                }
                
            } else {
                // Too early for all pending deferred commands, check again when
                // next acknowledgment comes in.
                break;
            }
        }
    }

    /**
     * This is the server for the protocol "curl://eclipseproxy".
     * These URL are used on the curl side to read/write files content
     * such that the real file system i/o is performed in eclipse.
     * 
     * NOTE: No currently used, to reconsider when we fully integrate VLE and other tools
     * This will allow the programatic updates to manifest.curl files to be recorded
     * in the eclipse "local history".
     * 
     * @throws CoreException
     * @throws IOException
     */
    private void readOpenResponse() throws CoreException, IOException
    {
        String curlUrl = DataIO.readString(fSockIn);
        if (! curlUrl.startsWith(CurlUIIDs.CURL_PROXY_PREFIX)) {
            DataIO.writeBoolean(fSockOut, false);
        }
        else {
            String filepath = CoreUtil.convertPathToEclipse(curlUrl);
            IFile file = CoreUtil.getFile(filepath);
            if (file == null) {
                DataIO.writeBoolean(fSockOut, false);
            }
            else {
                Reader reader = null;
                char[] cbuf = new char[4096]; 
                try {
                    String charset = file.getCharset();
                    reader = new BufferedReader(new InputStreamReader(file.getContents(true), charset));
                    StringBuilder sb = new StringBuilder(4096);
                    int lread;
                    while ((lread = reader.read(cbuf)) != -1) {
                        sb.append(cbuf, 0, lread);
                    }
                    DataIO.writeBoolean(fSockOut, true);
                    DataIO.writeString(fSockOut, sb.toString());
                } catch (IOException ioE) {
                    DataIO.writeBoolean(fSockOut, false);
                } finally {
                    if (reader != null) {
                        try {
                            reader.close();
                        } catch (IOException e) {}
                    }
                }
            }
        }
        DataIO.flush(fSockOut);
    }
}
