/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.DataInputStream;
import java.io.DataOutputStream;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentExtension4;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.Region;
import org.eclipse.ui.IEditorPart;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.editors.CurlEditor;
import com.curl.eclipse.editors.CurlSourceViewer;
import com.curl.eclipse.util.CoreUtil;

/**
 * Data proxy for the remote curl DocumentAgent/TextEditBuffer.
 *
 * @author fmisiak
 *
 */
public class DocumentProxy extends ProxyData
{
    private final IDocument fCurlDocument;
    private IDocumentListener fCurlDocumentListener;
    private boolean fEnableReplicationToCurl;
    private boolean fDeclareBufferCleanEventIsPending;
    private long fDeclareBufferCleanTimestamp;
    private boolean fDocumentIsAboutToChange;
    
    public DocumentProxy(
            IDocument curlDocument, 
            String docURL)
    {
        super(curlDocument);
        fCurlDocument = curlDocument;
        addDocumentListener();
        
        CurlPlugin.getDefault().getWorkbenchProxy().createDocumentAgent(
                fProxyID,
                docURL,
                curlDocument.get());
    }
    
    private void addDocumentListener()
    {
        fEnableReplicationToCurl = true;
        final Region[] curlRegion = new Region[1];
        final String[] insertText = new String[1];
        fCurlDocumentListener = new IDocumentListener() {
            public void documentAboutToBeChanged(DocumentEvent event) 
            {
                if (fEnableReplicationToCurl) {
                    // offset/length conversion can only be performed before the document has changed.
                    curlRegion[0] = eclipseRegionToCurlRegion(event.getOffset(), event.getLength());
                    insertText[0] = event.getText();
                    fDocumentIsAboutToChange = true;
                }
            }
            public void documentChanged(DocumentEvent event) {
                if (fEnableReplicationToCurl) {
                    // by deferring until the doc gets changed, this guarantee that eclipse as checked out
                    // the file if it was needed.
                    replaceBufferContent(curlRegion[0].getOffset(), curlRegion[0].getLength(), insertText[0]);
                    fDocumentIsAboutToChange = false;
                    if (fDeclareBufferCleanEventIsPending) {
                        declareBufferClean(fDeclareBufferCleanTimestamp);
                    }
                }
            }
        };
        fCurlDocument.addDocumentListener(fCurlDocumentListener);
    }

    public void replaceBufferContent(
            final int offset,
            final int length,
            final String text)
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandReplaceFileBufferContent) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(offset);
                write(length);
                write(text);
            }
        });
    }
    
    public void bufferResyncedResponse(
            String newContent)
    {
        // No need to transform line delimiters since buffer coming from external tools
        // (for example the VLE) kept the original line delimiters
        replaceLocalOnly(0, fCurlDocument.getLength(), newContent);
    }

    public void characterDeletedResponse(
            final int row,
            final int col,
            final int length)
    {
        try {
            int offsetAtBegOfLine = fCurlDocument.getLineOffset(row);
            int offset = offsetAtBegOfLine + col;
            int lenToBeDeleted = 
                curlSelectionLengthToEclipseSelectionLength(length, offset);
            replaceLocalOnly(offset, lenToBeDeleted, ""); //$NON-NLS-1$
        } catch (BadLocationException e) {
            CoreUtil.logError("Editor insertion response failed", e); //$NON-NLS-1$
        }
    }

    public int curlOffsetToEclipseOffset(
            int curlOffset)
    {
        int eclipseOffset = 0;
        int curlOffsetRemaining = curlOffset;
        try {
            for (int line = 0; curlOffsetRemaining > 0; line++) {
                String lineDelimiter = fCurlDocument.getLineDelimiter(line);
                int delimiterLength = lineDelimiter == null ? 0 : lineDelimiter.length();
                int eclipseLineLength = fCurlDocument.getLineLength(line);
                int curlLineLength = eclipseLineLength - (delimiterLength == 0 ? 0 : delimiterLength - 1);
                if (curlOffsetRemaining >= curlLineLength) {
                    eclipseOffset += eclipseLineLength;
                    curlOffsetRemaining -= curlLineLength;
                } else {
                    eclipseOffset += curlOffsetRemaining;
                    break;
                }
            }
            return eclipseOffset;
        } catch (BadLocationException e) {
            CoreUtil.logError("Convertion to Eclipse offset failed", e); //$NON-NLS-1$
        }
        return 0;
    }

    /**
     * Convert the length of a selection from Curl to Eclipse.  They can be
     * different due to line delimiter differences.
     * 
     * TODO: Could be simplified by combining this with  curlOffsetToEclipseOffset()
     */
    public int curlSelectionLengthToEclipseSelectionLength(
            int selectionChangedLength,
            int eclipseOffset)
    {
        try {
            IRegion lineInformationOfOffset = fCurlDocument.getLineInformationOfOffset(eclipseOffset);
            int lineStartSelection = fCurlDocument.getLineOfOffset(eclipseOffset);
            int colZeroBased = eclipseOffset - lineInformationOfOffset.getOffset();
    
            int curlLengthRemaining = selectionChangedLength;
            int eclipseLength = selectionChangedLength;
            for (int line = lineStartSelection; curlLengthRemaining > 0; line++) {
                String lineDelimiter = fCurlDocument.getLineDelimiter(line);
                int delimiterLength = lineDelimiter == null ? 0 : lineDelimiter.length();
                int lineLength = fCurlDocument.getLineLength(line);
                int curlLineLength = lineLength - (delimiterLength == 0 ? 0 : delimiterLength - 1);
                if (curlLengthRemaining >= (curlLineLength - colZeroBased)) {
                    eclipseLength += (delimiterLength == 0 ? 0 : delimiterLength - 1);
                    curlLengthRemaining -= curlLineLength - colZeroBased;
                    colZeroBased = 0;
                } else {
                    break;
                }
    
            }
            return eclipseLength;
        } catch (BadLocationException e) {
            CoreUtil.logError("Convertion to Eclipse selection length failed", e); //$NON-NLS-1$
        }
        return 0;
    }

    public Region eclipseRegionToCurlRegion(
            int eclipseOffset,
            int eclipseLength)
    {
        try {
            // adjust offset
            int firstLine = fCurlDocument.getLineOfOffset(eclipseOffset);
            int curlOffset = eclipseOffset;
            for (int line = 0; line < firstLine; line++) {
                String delimiter = fCurlDocument.getLineDelimiter(line);
                curlOffset -= (delimiter.length() - 1);
            }
            
            int curlLength = eclipseLength;
            if (curlLength > 0) {
                // adjust length
                int eclipseOffsetEnd = eclipseOffset + eclipseLength;
                int lastLine = fCurlDocument.getLineOfOffset(eclipseOffsetEnd);
                for(int line=firstLine; line <= lastLine; line++) {
                    if (line == lastLine) {
                        IRegion lineInformation = fCurlDocument.getLineInformation(line);
                        if ((eclipseOffsetEnd) <= (lineInformation.getOffset() + lineInformation.getLength())) {
                            // don't consider line delimiter of last line
                            break;
                        }
                    }
                    String delimiter = fCurlDocument.getLineDelimiter(line);
                    curlLength -= delimiter != null ? (delimiter.length() - 1) : 0;
                }
            }
            return new Region(curlOffset, curlLength);
        } catch (BadLocationException e) {
            CoreUtil.logError("Convertion to Curl offset failed", e); //$NON-NLS-1$
        }
        return new Region(0, 0);
    }

    private void replaceLocalOnly(int pos, int length, String text)
    {
        try {
            fEnableReplicationToCurl = false;
            
            // Also disable corresponding selection listener in CurlSourceViewer
            // else upcoming curl selection (coming after char inserted or removed) will be overwritten by StyledText selection
            CurlSourceViewer curlEditorSourceViewer = null;
            
            // Since we're processing a replace event from Mediator, there is necessarily 
            // a corresponding active editor
            IEditorPart activeEditor = CurlPlugin.getDefault().getWorkbench()
                .getActiveWorkbenchWindow().getActivePage().getActiveEditor();
            if (activeEditor instanceof CurlEditor) {
                CurlEditor curlEditor = (CurlEditor)activeEditor;
                curlEditorSourceViewer = curlEditor.getCurlEditorSourceViewer();
            }
            else {
                CoreUtil.logError("No curl active editor found"); //$NON-NLS-1$
            }
            try {
                if (curlEditorSourceViewer != null) {
                    curlEditorSourceViewer.enableSelectionListener(false);
                }
                fCurlDocument.replace(pos, length, text);
            }
            finally {
                if (curlEditorSourceViewer != null) {
                    curlEditorSourceViewer.enableSelectionListener(true);
                }
            }
        } catch (BadLocationException e) {
            CoreUtil.logError("Editor deletion response failed", e); //$NON-NLS-1$
        } finally {
            fEnableReplicationToCurl = true;
        }
    }

    public void stringInsertedResponse(
            final int row,
            final int col,
            final String insertedString)
    {
        try {
            int offset = fCurlDocument.getLineOffset(row) + col;
            String toBeInserted = insertedString;
            // Line terminator in the inserted string will always be '\n'.
            if (insertedString.indexOf('\n') != -1) {
                if (fCurlDocument instanceof IDocumentExtension4) {
                    IDocumentExtension4 documentExtension4 = (IDocumentExtension4)fCurlDocument;
                    String delimeter = documentExtension4.getDefaultLineDelimiter();
                    if(delimeter != "\n") {  //$NON-NLS-1$
                        toBeInserted = insertedString.replaceAll("\n", delimeter); //$NON-NLS-1$
                    }
                }
                else {
                    CoreUtil.logWarning("Could not determine default line delimiter for document"); //$NON-NLS-1$
                }
            }
            replaceLocalOnly(offset, 0, toBeInserted);
        } catch (BadLocationException e) {
            CoreUtil.logError("Editor insertion response failed", e);  //$NON-NLS-1$
        }
    }

    /**
     * Respond to mediator command that a string has been inserted into the editor buffer.
     */
    public static class StringInsertedProxyCommand
    {
        public int row;
        public int col;
        public String insertedString;
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandStringInserted, new StringInsertedCommandHandler());
    }
    public static class StringInsertedCommandHandler extends ProxyCommandHandler<DocumentProxy, StringInsertedProxyCommand>
    {
        
        @Override
        StringInsertedProxyCommand decode(
                DataInputStream sockIn)
        {
            StringInsertedProxyCommand command = new StringInsertedProxyCommand();
            command.row = DataIO.readInt(sockIn);
            command.col = DataIO.readInt(sockIn);
            command.insertedString = DataIO.readString(sockIn);
            return command;
        }
        
        @Override
        void execute(
                final StringInsertedProxyCommand command,
                final DocumentProxy proxy,
                DataOutputStream sockOut)
        {
            asyncExec(new Runnable() {
                public void run()
                {
                    proxy.stringInsertedResponse(command.row, command.col, command.insertedString);
                }
            });
        }
    }
    

    /**
     * Respond to mediator command that the editor status has changed.
     */
    public static class CharacterDeletedProxyCommand
    {
        public int row;
        public int col;
        public int length;
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandCharacterDeleted, new CharacterDeletedCommandHandler());
    }
    public static class CharacterDeletedCommandHandler extends ProxyCommandHandler<DocumentProxy, CharacterDeletedProxyCommand>
    {
        @Override
        CharacterDeletedProxyCommand decode(
                DataInputStream sockIn)
        {
            CharacterDeletedProxyCommand command = new CharacterDeletedProxyCommand();
            command.row = DataIO.readInt(sockIn);
            command.col = DataIO.readInt(sockIn);
            command.length = DataIO.readInt(sockIn);
            return command;
        }
        
        @Override
        void execute(
                final CharacterDeletedProxyCommand command,
                final DocumentProxy proxy,
                DataOutputStream sockOut)
        {
            asyncExec(new Runnable() {
                public void run()
                {
                    proxy.characterDeletedResponse(command.row, command.col, command.length);
                }
            });
        }
    }

    /**
     * Respond to a mediator command that the curl buffer content as been fully
     * resynchronized with the content of an external editor (e.g VLE)
     */
    private static class BufferResyncedProxyCommand
    {
        protected final String fNewContent;
        public BufferResyncedProxyCommand(
                String newContent)
        {
            fNewContent = newContent;
        }
    }
    static
    {
        EclipseServer.register(MediatorConnection.commandBufferResynced, new BufferResyncedCommandHandler());
    }
    static class BufferResyncedCommandHandler extends ProxyCommandHandler<DocumentProxy, BufferResyncedProxyCommand>
    {
        @Override
        BufferResyncedProxyCommand decode(
                DataInputStream sockIn)
        {
            final String newContent = DataIO.readString(sockIn);
            return new BufferResyncedProxyCommand(newContent);
        }
        
        @Override
        void execute(
                final BufferResyncedProxyCommand command,
                final DocumentProxy proxy,
                DataOutputStream sockOut)
        {
            asyncExec(new Runnable() {
                public void run()
                {
                    proxy.bufferResyncedResponse(command.fNewContent);
                }
            });
        }
    }
    

    private static void asyncExec(Runnable runnable) 
    {
        CurlPlugin.getDefault().getWorkbench().getDisplay().asyncExec(runnable);
    }

    public void enableReplicationToCurl(
            boolean enable)
    {
        fEnableReplicationToCurl = enable;
    }

    public void declareBufferClean(final long timestamp)
    {
        if (fDocumentIsAboutToChange) {
            // do nothing now, differ until until we are signaled that the document is really changed.
            fDeclareBufferCleanEventIsPending = true;
            fDeclareBufferCleanTimestamp = timestamp;
        } else {
            fDeclareBufferCleanEventIsPending = false;
            fDeclareBufferCleanTimestamp = -1;
            fMediatorConnection.async.execute(new AsynchronousRemoteOperation(MediatorConnection.commandDeclareBufferClean) {
                @Override
                protected void writeArguments()
                {
                    write(fProxyID);
                    write(timestamp);
                }
            });
        }
    }
}
