/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.remote;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.util.ArrayList;
import java.util.List;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.debug.CurlDebugTarget;
import com.curl.eclipse.debug.CurlEvaluationDescriptor;
import com.curl.eclipse.debug.CurlStackFrameDescriptor;
import com.curl.eclipse.debug.CurlVariableDescriptor;
import com.curl.eclipse.util.CoreUtil;
import com.ibm.icu.text.MessageFormat;


/**
 *
 */
public class DebugProxyData extends ProxyData
{
    private final String fAppletID;
    private volatile boolean fStructurallyComplete;

    static
    {
        EclipseServer.register(MediatorConnection.commandDebugAppletTerminated, new DebugAppletTerminatedCommandHandler());
    }
    
    static class DebugAppletTerminatedCommandHandler extends ProxyCommandHandler<DebugProxyData, NullProxyCommand>
    {
        
        @Override
        void execute(
                NullProxyCommand command,
                DebugProxyData proxy,
                DataOutputStream sockOut)
        {
            proxy.getDebugTarget().fMediatorCommandHandler.terminated();
        }
    }
    
    public static class DebugBreakpointHitProxyCommand
    {
        public String helperEngineID;
        public int reason;
        public boolean canStepInto;
        public boolean canStepOver;
        public boolean canStepOut;
        public boolean canResume;
        public boolean canSuspend;
        public boolean canTerminate;
        public boolean isExceptionBreakpoint;
        public CurlStackFrameDescriptor[] stackFrameDescs;
        public String exceptionBreakpointTypeName;
    }
    
    static
    {
        EclipseServer.register(MediatorConnection.commandDebugBreakpointHit, new DebugBreakpointHitCommandHandler());
    }

    public static class DebugBreakpointHitCommandHandler extends ProxyCommandHandler<DebugProxyData, DebugBreakpointHitProxyCommand>
    {
        
        @Override
        DebugBreakpointHitProxyCommand decode(
                DataInputStream sockIn)
        {
            DebugBreakpointHitProxyCommand command = new DebugBreakpointHitProxyCommand();

            command.helperEngineID = (new Integer(DataIO.readInt(sockIn))).toString();
            command.reason = DataIO.readInt(sockIn);
            command.canStepInto = DataIO.readBoolean(sockIn);
            command.canStepOver = DataIO.readBoolean(sockIn);
            command.canStepOut = DataIO.readBoolean(sockIn);
            command.canResume = DataIO.readBoolean(sockIn);
            command.canSuspend = DataIO.readBoolean(sockIn);
            command.canTerminate = DataIO.readBoolean(sockIn);
            command.isExceptionBreakpoint = DataIO.readBoolean(sockIn);
            String exceptionBreakpointTypeName = ""; //$NON-NLS-1$
            String exceptionBreakpointReason = ""; //$NON-NLS-1$
            if (command.isExceptionBreakpoint) {
                exceptionBreakpointTypeName = DataIO.readString(sockIn);
                exceptionBreakpointReason = DataIO.readString(sockIn);
            }
            final int stackFrameSize = DataIO.readInt(sockIn);
            List<CurlStackFrameDescriptor> frameDescs = new ArrayList<CurlStackFrameDescriptor>(stackFrameSize);
            // We can have missing frames!
            int nextFrameNumber = 0;
            for (int i = 0; i < stackFrameSize; i++) {
                final int frameNumber = DataIO.readInt(sockIn);
                final String functionName = DataIO.readString(sockIn);
                final boolean hasDebugInfo = DataIO.readBoolean(sockIn);
                String fileName = ""; //$NON-NLS-1$
                int lineNumber = 0;
                if (hasDebugInfo) {
                    fileName = DataIO.readString(sockIn);
                    lineNumber = DataIO.readInt(sockIn);
                }
                if (nextFrameNumber != frameNumber) {
                    // Account for missing frames
                    frameDescs.add(CurlStackFrameDescriptor.makeMissingFrameDesc(
                            nextFrameNumber,
                            command.reason,
                            MessageFormat.format(
                                    RemoteMessages.DebugModelMissing_Frame,
                                    new String[] { new Integer(frameNumber - nextFrameNumber).toString() })));
                } else {
                    frameDescs.add(CurlStackFrameDescriptor.makeFullFrameDesc(
                            frameNumber,
                            command.reason,
                            exceptionBreakpointTypeName,
                            exceptionBreakpointReason,
                            functionName,
                            hasDebugInfo,
                            fileName,
                            lineNumber));
                }
                nextFrameNumber = frameNumber + 1;
            }
            command.stackFrameDescs = frameDescs.toArray(new CurlStackFrameDescriptor[stackFrameSize]);
            return command;
        }
        
        @Override
        void execute(
                DebugBreakpointHitProxyCommand command,
                DebugProxyData proxy,
                DataOutputStream sockOut)
        {
            proxy.getDebugTarget().fMediatorCommandHandler.breakpointHit(command);
        }
    }
    
    static
    {
        EclipseServer.register(MediatorConnection.commandDebugResumed, new DebugResumedCommandHandler());
    }


    static class DebugResumedCommandHandler extends ProxyCommandHandler<DebugProxyData, NullProxyCommand>
    {
        @Override
        void execute(
                NullProxyCommand command,
                DebugProxyData proxy,
                DataOutputStream sockOut)
        {
            proxy.getDebugTarget().fMediatorCommandHandler.resumed();
        }
    }
    
    public DebugProxyData(
            final String appletID)
    {
        // Note that this proxy is created incompletely, without a "proxy of".
        // The "proxy of" is set
        // later when the debug session actually starts. As long as this is
        // true, the proxy is not
        // "structurally complete".
        super(null);
        fStructurallyComplete = false;
        fAppletID = appletID;
    }

    // Must not be synchronized, otherwise setStructuallyComplete could not be
    // called.
    public CurlDebugTarget getDebugTarget()
    {
        if (getStructurallyComplete())
            return (CurlDebugTarget)fProxyOf;
        // The "proxy of" is not set yet. But it will be soon.
        CoreUtil.logInfo("Debug Target is not available yet, waiting..."); //$NON-NLS-1$
        int numOfTimesToTry = 20;
        for (int i = 0; i < numOfTimesToTry; i++) {
            if (getStructurallyComplete())
                return (CurlDebugTarget)fProxyOf;
            try {
                CoreUtil.logInfo("Debug proxy is waiting for debug target to be set..."); //$NON-NLS-1$
                Thread.sleep(1000);
            } catch (Exception e) {
            }
        }
        CoreUtil.logWarning("Debug Target was not found.  Giving up.  Message from mediator ignored."); //$NON-NLS-1$
        return null;
    }

    synchronized private boolean getStructurallyComplete()
    {
        return fStructurallyComplete;
    }

    synchronized public void setStructurallyComplete()
    {
        // FIXME: possibly use an override of setProxyOf() instead of providing
        // this.
        fStructurallyComplete = true;
    }

    synchronized public String getAppletID()
    {
        return fAppletID;
    }

    synchronized public void makeDebugSessionProxyData()
    {
        fMediatorConnection.async.execute(new AsynchronousRemoteOperation(
                MediatorConnection.commandMakeDebugSessionDataProxy) {
            @Override
            protected void writeArguments()
            {
                write(CurlPlugin.getDefault().getWorkbenchProxy().getProxyID());
                write(fProxyID);
                write(fAppletID);
            }
        });
    }

    // FIXME: synchronize ?
    synchronized public void terminate()
    {
        asynchronousNoArgsOperation(MediatorConnection.commandTerminateApplet);
    }

    synchronized public void suspend()
    {
        asynchronousNoArgsOperation(MediatorConnection.commandSuspendApplet);
    }

    /**
     * Resume this debugging session
     */
    // FIXME: synchronize ?
    synchronized public void resume()
    {
        asynchronousNoArgsOperation(MediatorConnection.commandResumeApplet);
    }

    /**
     * Step of some kind in this debugging session
     */
    // FIXME: synchronize ?
    synchronized public void step(
            int commandCode)
    {
        asynchronousNoArgsOperation(commandCode);
    }

    /**
     *
     */
    private static abstract class SynchronousRemoteDebugOperation extends SynchronousRemoteOperation
    {
        SynchronousRemoteDebugOperation(
                int commandCode)
        {
            super(commandCode);
        }

        protected CurlVariableDescriptor[] getVarDescs()
        {
            int numOfVars = readInt();
            CurlVariableDescriptor[] varDescs = new CurlVariableDescriptor[numOfVars];
            for (int i = 0; i < numOfVars; i++) {
                varDescs[i] = getVarDesc();
            }
            return varDescs;
        }

        private CurlVariableDescriptor getVarDesc()
        {
            final String typeName = readString();
            final String name = readString();
            final String value = readString();
            final boolean expandable = readBoolean();
            int id = -1;
            if (expandable)
                id = readInt();
            return new CurlVariableDescriptor(typeName, name, value, expandable, id);
        }
    }

    public CurlEvaluationDescriptor evaluateExpression(
            final String expressionText,
            final int frameNumber)
    {
        final Object res[] = new Object[1];
        fMediatorConnection.sync.execute(new SynchronousRemoteDebugOperation(
                MediatorConnection.commandEvaluateExpression) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(expressionText);
                write(frameNumber);
            }

            @Override
            protected void receiveResponse()
            {
                String errorMessage = ""; //$NON-NLS-1$
                CurlVariableDescriptor[] varDescs = null;
                boolean hasErrors = readBoolean();
                if (hasErrors) {
                    errorMessage = readString();
                    res[0] = new CurlEvaluationDescriptor(expressionText, errorMessage);
                } else {
                    varDescs = getVarDescs();
                    res[0] = new CurlEvaluationDescriptor(expressionText, varDescs[0]);
                }
            }
        });
        return (CurlEvaluationDescriptor)res[0];
    }

    public CurlVariableDescriptor[] getStackVars(
            final int frameNumber)
    {
        final Object[] res = new Object[1];
        fMediatorConnection.sync.execute(new SynchronousRemoteDebugOperation(
                MediatorConnection.commandGetStackVars) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(frameNumber);
            }

            @Override
            protected void receiveResponse()
            {
                res[0] = getVarDescs();
            }
        });
        return (CurlVariableDescriptor[])res[0];
    }

    /**
     * Expand and get the variables in this variable
     */
    public CurlVariableDescriptor[] expandVars(
            final int varID)
    {
        final Object[] res = new Object[1];
        fMediatorConnection.sync.execute(new SynchronousRemoteDebugOperation(
                MediatorConnection.commandExpandVars) {
            @Override
            protected void writeArguments()
            {
                write(fProxyID);
                write(varID);
            }

            @Override
            protected void receiveResponse()
            {
                res[0] = getVarDescs();
            }
        });
        return (CurlVariableDescriptor[])res[0];
    }

//    private void receivedBreakpointHit(
//            DebugBreakpointHitCommandHandler command)
//    {
//
//        getDebugTarget().fMediatorCommandHandler.breakpointHit(
//                helperEngineID,
//                reason,
//                isExceptionBreakpoint,
//                exceptionBreakpointTypeName,
//                canStepInto,
//                canStepOver,
//                canStepOut,
//                canResume,
//                canSuspend,
//                canTerminate,
//                frameDescs.toArray(new CurlStackFrameDescriptor[nextFrameNumber]));
//    }
}
