/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.files;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.core.filebuffers.FileBuffers;
import org.eclipse.core.filebuffers.IFileBuffer;
import org.eclipse.core.filebuffers.IFileBufferListener;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jface.text.source.AnnotationModel;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.team.core.history.IFileRevision;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPathEditorInput;
import org.eclipse.ui.IStorageEditorInput;
import org.eclipse.ui.editors.text.ForwardingDocumentProvider;
import org.eclipse.ui.editors.text.TextFileDocumentProvider;
import org.eclipse.ui.ide.FileStoreEditorInput;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.texteditor.IDocumentProvider;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.remote.DocumentProxy;
import com.curl.eclipse.util.CoreUtil;
/**
 * The document provider for Curl source files.
 * 
 * Policy: 1) File operations (creation, deletion, move, etc.) are handled on the Eclipse side and file
 * content as a whole is sent to the Curl side.
 * 
 * 2) File content operations (insertion, analysis, etc.) are handled on the Curl side and sent piece by piece
 * to Eclipse side.
 * 
 * 2) As result, for each document in the editor there are two buffers, one on Eclipse side and one on Curl
 * side, but they are constantly (immediately) kept to be identical.
 * 
 * @see org.eclipse.jdt.internal.ui.javaeditor.CompilationUnitDocumentProvider
 */
public class CurlFileDocumentProvider extends TextFileDocumentProvider
{
    /**
     * Mimic {@link CompilationUnitDocumentProvider#fFakeCUMapForMissingInfo}.
     * This is needed to deal with revision-history elements.
     */
    private final Map<Object, CurlFileInfo> fFakeCurlMapForMissingInfo= new HashMap<Object,CurlFileInfo>();
    private final IFileBufferListener fBufferListener;

    public CurlFileDocumentProvider()
    {
        super();
        IDocumentProvider provider= new TextFileDocumentProvider();
        provider= new ForwardingDocumentProvider(null, new CurlDocumentSetupParticipant(), provider);
        setParentDocumentProvider(provider);

        fBufferListener = new IFileBufferListener() {
            public void bufferCreated(IFileBuffer buffer) {
            }

            public void bufferDisposed(IFileBuffer buffer) {
            }

            public void bufferContentAboutToBeReplaced(IFileBuffer buffer) {
            }

            public void bufferContentReplaced(IFileBuffer buffer) {
            }

            public void stateChanging(IFileBuffer buffer) {
            }

            @SuppressWarnings("unchecked") 
            public void dirtyStateChanged(IFileBuffer buffer, boolean isDirty) {
                if (!isDirty) {
                    for (Iterator<FileEditorInput> iterator = getElements(buffer); iterator.hasNext();) {
                        FileEditorInput editorInput = iterator.next();
                        FileInfo info = getFileInfo(editorInput);
                        if (info instanceof CurlFileInfo)  {
                            ((CurlFileInfo)info).fDocumentProxy.declareBufferClean(buffer.getModificationStamp());
                            return;
                        }
                    }
                }
            }

            public void stateValidationChanged(IFileBuffer buffer, boolean isStateValidated) {
            }

            public void underlyingFileMoved(IFileBuffer buffer, IPath path) {
            }

            public void underlyingFileDeleted(IFileBuffer buffer) {
            }

            public void stateChangeFailed(IFileBuffer buffer) {
            }
        };
        FileBuffers.getTextFileBufferManager().addFileBufferListener(fBufferListener);
    }

    static protected class CurlFileInfo extends FileInfo 
    {
        public DocumentProxy fDocumentProxy;
    }


    @Override
    protected org.eclipse.ui.editors.text.TextFileDocumentProvider.FileInfo createEmptyFileInfo() 
    {
        return new CurlFileInfo();
    };

    @Override
    protected FileInfo createFileInfo(Object element) throws CoreException {
        FileInfo info= super.createFileInfo(element);
        // Do like JDT does....
        if (!(info instanceof CurlFileInfo)) {
            return null;
        }
        setUpSynchronization(info);
        return info;
    }

    @Override
    public void connect(Object element) throws CoreException {
        super.connect(element);
        if (getFileInfo(element) != null)
            return;

        CurlFileInfo info= fFakeCurlMapForMissingInfo.get(element);
        if (info == null) {
            info = new CurlFileInfo();
            info.fElement= element;
            info.fModel= new AnnotationModel();
            fFakeCurlMapForMissingInfo.put(element, info);
        }
        info.fCount++;
    }

    @Override
    public IAnnotationModel getAnnotationModel(Object element) {
        IAnnotationModel model= super.getAnnotationModel(element);
        if (model != null)
            return model;

        FileInfo info= fFakeCurlMapForMissingInfo.get(element);
        if (info != null) {
            if (info.fModel != null)
                return info.fModel;
            if (info.fTextFileBuffer != null)
                return info.fTextFileBuffer.getAnnotationModel();
        }

        return null;
    }

    @Override
    public void disconnect(Object element) {
        CurlFileInfo info= fFakeCurlMapForMissingInfo.get(element);
        if (info != null)  {
            if (info.fCount == 1) {
                fFakeCurlMapForMissingInfo.remove(element);
                info.fModel= null;
                info.fDocumentProxy.destroy();
            } else {
                info.fCount--;
            }
        }
        super.disconnect(element);
    }

    @SuppressWarnings("unchecked") 
    public void shutdown()
    {
        Iterator e = getConnectedElementsIterator();
        while (e.hasNext())
            disconnect(e.next());
        fFakeCurlMapForMissingInfo.clear();
        FileBuffers.getTextFileBufferManager().removeFileBufferListener(fBufferListener);
    }

    @Override
    protected void disposeFileInfo(Object element, org.eclipse.ui.editors.text.TextFileDocumentProvider.FileInfo info)
    {
        super.disposeFileInfo(element, info);
        if (info instanceof CurlFileInfo) {
            CurlFileInfo curlFileInfo = (CurlFileInfo) info;
            curlFileInfo.fDocumentProxy.destroy();
        }
    }

    public DocumentProxy establishDocumentProxy(
            Object element)
    {
        FileInfo fileInfo = getFileInfo(element);
        if (fileInfo == null) {
            fileInfo = fFakeCurlMapForMissingInfo.get(element);
        }
        if (fileInfo instanceof CurlFileInfo) {
            return establishDocumentProxy(element, (CurlFileInfo)fileInfo);
        }
        throw new RuntimeException("fileInfo wasn't created for element=" + element); //$NON-NLS-1$
    }

    private DocumentProxy establishDocumentProxy(
            Object element,
            CurlFileInfo curlFileInfo)
    {
        if (curlFileInfo.fDocumentProxy == null) {
            String docURL;
            if (element instanceof IPathEditorInput) {
                IPathEditorInput pathEditorInput = (IPathEditorInput)element;
                docURL = CoreUtil.convertPathToCurlURL(pathEditorInput);
            } 
            else if (element instanceof FileStoreEditorInput) {
                FileStoreEditorInput fileStoreEditorInput = (FileStoreEditorInput)element;
                docURL = fileStoreEditorInput.getURI().toString();
            } else {
                // we must record this editor input to support outline view navigation
                IEditorInput editorInput = (IEditorInput)element;
                docURL = CurlPlugin.getDefault().getWorkbenchOperations().storeHistoryEditorInput(editorInput);
                assert docURL != null;
            }
            curlFileInfo.fDocumentProxy = new DocumentProxy(getDocument(element), docURL);
        }
        else if (element instanceof IStorageEditorInput) {
            // document proxy was previously while creating another editor
            // but we must record this editor input to support outline view navigation
            IStorageEditorInput editorInput = (IStorageEditorInput)element;
            IFileRevision fileRevision = (IFileRevision)editorInput.getAdapter(IFileRevision.class);
            if (fileRevision != null) {
                CurlPlugin.getDefault().getWorkbenchOperations().storeHistoryEditorInput(editorInput);
            }
        }
        return curlFileInfo.fDocumentProxy;
    }
}
