/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.editors;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.filesystem.EFS;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.remote.ICurlViewSelectionListener;
import com.curl.eclipse.remote.ProxyView;
import com.curl.eclipse.util.CoreUtil;

public abstract class CurlView extends ViewPart implements ICurlViewSelectionListener
{
    protected IStructuredSelection fCurrentSelection;

    // Represents the window Curl uses to create a child window.
    private Composite fComposite;
    // Proxy for communicating with the Curl view.
    private ProxyView fProxy;
    // View's help ID.
    private final String fViewHelpID;

    protected final SelectionProvider fSelectionProvider;

    private Composite fDummyComposite; // FIXME: unused?????


    protected CurlView(String helpID)
    {
        fViewHelpID = helpID;
        fSelectionProvider = new SelectionProvider();
    }

    @Override
    public void dispose()
    {
        fProxy.destroy();
        super.dispose();
    }

    @Override
    public void createPartControl(
            Composite parent)
    {
        fDummyComposite = new Composite(parent, SWT.NONE) {
            @Override
            public boolean isVisible()
            {
                // horrible hack, see org.eclipse.swt.widgets.Control.forceFocus(),
                // somehow, on windows at least, this component is not visible
                // probably because it is stacked below the curl component,
                // anyway we need this to allow forceFocus to do it's job.
                // without this Shell won't be able to track the active control
                // and thus won't fire activation/deactivation when needed.
                return true;
            }
        };
        fComposite = parent;
        // Establish the proxy
        if (fProxy == null) {
            ProxyView proxy = createProxy();
            proxy.createRemoteView(getSite().getWorkbenchWindow(), fComposite);
            fProxy = proxy;
        }
        getSite().setSelectionProvider(fSelectionProvider);
        parent.addControlListener(new ControlAdapter() {
            @Override
            public void controlResized(
                    ControlEvent e)
            {
                setNewSize();
            }

            @Override
            public void controlMoved(
                    ControlEvent e)
            {
                setNewSize();
            }
        });
        setHelpID(fComposite);
    }

    protected abstract ProxyView createProxy();

    @Override
    public void selectionChanged(
            String urlString)
    {
        try {
            if (urlString != null && urlString.length() > 0) {
                IFile[] mappedFiles = ResourcesPlugin.getWorkspace().getRoot()
                .findFilesForLocationURI(new URI(urlString));
                List<IResource> res = new ArrayList<IResource>();
                Collections.addAll(res, mappedFiles);
                IContainer[] mappedContainers = ResourcesPlugin.getWorkspace().getRoot()
                .findContainersForLocationURI(new URI(urlString));
                Collections.addAll(res, mappedContainers);
                Iterator<IResource> iterator = res.iterator();
                IResource resource;
                while(iterator.hasNext() && ((resource=iterator.next()) != null)) {
                    if (! resource.isAccessible()) {
                        iterator.remove();
                    }
                }
                if (res.size() > 0) {
                    fCurrentSelection = new StructuredSelection(res);
                } else {
                    try {
                        IFileStore store = EFS.getStore(URI.create(urlString));
                        fCurrentSelection = new StructuredSelection(store);
                    } catch (CoreException e) {
                        CoreUtil.logInfo("decoding selection=" + urlString, e); //$NON-NLS-1$
                        fCurrentSelection = StructuredSelection.EMPTY;
                    }
                }
            } else {
                fCurrentSelection = StructuredSelection.EMPTY;
            }
            CurlPlugin.getDefault().getWorkbench().getDisplay().asyncExec(
                    new Runnable(){
                        @Override
                        public void run()
                        {
                            // need to clone selection ?!
                            // note that for now this selection will be sent back to the mediator!!
                            fSelectionProvider.fireSelectionChanged(new SelectionChangedEvent(fSelectionProvider, fCurrentSelection));
                        }
                    });
        } catch (URISyntaxException e) {
            CoreUtil.logError("CurlView current selection invalid url", e); //$NON-NLS-1$
        }
    }

    protected void setHelpID(Control control)
    {
        PlatformUI.getWorkbench().getHelpSystem().setHelp(control, fViewHelpID);
    }

    private void setNewSize()
    {
        fProxy.resizeView(fComposite);
    }

    @Override
    public void setFocus()
    {
        fDummyComposite.setFocus();
        fProxy.setFocus();
    }
   
    protected class SelectionProvider implements ISelectionProvider
    {
        private final ListenerList selectionChangedListeners = new ListenerList();

        
        /**
         * Notifies any selection changed listeners that the viewer's selection has changed.
         * Only listeners registered at the time this method is called are notified.
         *
         * @param event a selection changed event
         *
         * @see ISelectionChangedListener#selectionChanged
         */
        protected void fireSelectionChanged(final SelectionChangedEvent event) {
            Object[] listeners = selectionChangedListeners.getListeners();
            for (int i = 0; i < listeners.length; ++i) {
                final ISelectionChangedListener l = (ISelectionChangedListener) listeners[i];
                SafeRunnable.run(new SafeRunnable() {
                    public void run() {
                        l.selectionChanged(event);
                    }
                });
            }
        }
      
        @Override
        public void addSelectionChangedListener(
                ISelectionChangedListener listener)
        {
            selectionChangedListeners.add(listener);
        }
    
        @Override
        public ISelection getSelection()
        {
            return fCurrentSelection;
        }
        
        @Override
        public void removeSelectionChangedListener(
                ISelectionChangedListener listener)
        {
            selectionChangedListeners.remove(listener);
        }
        
        @Override
        public void setSelection(
                ISelection selection)
        {
            CoreUtil.logError("Not implemented (yet?!)"); //$NON-NLS-1$
        }
    }

    public void activateParentView(
            String viewId)
    {
        IWorkbenchPage page = getSite().getPage();
        IViewReference[] viewReferences = page.getViewReferences();
        boolean requestFocus = false;
        for (IViewReference viewReference : viewReferences) {
            if (viewReference.getId().equals(viewId)) {
                IWorkbenchPart part = viewReference.getPart(false);
         
                if (part == null) {
                    CoreUtil.logWarning("Could not make CurlView active"); //$NON-NLS-1$
                }
                else {
                    if (page.getActivePart() != part) {
                        requestFocus = true;
                        page.activate(part);
                    }
                }
                break;
            }
        }
        if (requestFocus) {
            setFocus();
        }
    }
}
