/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.editors;

import org.eclipse.jface.text.Region;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyleRange;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.printing.Printer;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

import com.curl.eclipse.remote.SourceViewProxy;

/**
 * Implements the text edit panel and widget for the Curl Editor.
 */
public class CurlTextEditPanel extends StyledText
{
    private final CurlSourceViewer fSourceViewer;
    private SourceViewProxy getProxy()
    {
        return fSourceViewer.getProxy();
    }

    public CurlTextEditPanel(CurlSourceViewer curlSourceViewer, Composite parent, int style) {
        super(parent, style);
        fSourceViewer = curlSourceViewer;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.swt.custom.StyledText#getTabs()
     * Curl side will convert each TAB to 1 "space" automatically (see DocumentAgent.scurl)
     */
    @Override
    public int getTabs()
    {
        return 1;
    }
    
    @Override
    public void addListener(
            int eventType,
            Listener listener)
    {
        if (eventType == SWT.Paint) {
            super.addListener(eventType, new Listener() {
                public void handleEvent(Event event) {
                    GC gc = event.gc;
                    Color background = getBackground(); 
                    gc.setBackground(background);
                    Display display = getDisplay();
                    setCursor(display.getSystemCursor(SWT.CURSOR_WAIT));
                    getCaret().setVisible(false);
                }
            });
        } else {
            super.addListener(eventType, listener);
        }
    }
    
    @Override
    public boolean setFocus()
    {
//      kept for history documentation reasons:
//         Set the visiblity to true for the duration of focus processing.
//         Otherwise, our set focus scheme would not work!
//        setVisible(true);
        super.setFocus();
        getProxy().setFocus();
//        setVisible(false);
        return true;
    }

    @Override
    public boolean isVisible()
    {
        // horrible hack, see org.eclipse.swt.widgets.Control.forceFocus(),
        // somehow, on windows at least, this component is not visible
        // probably because it is stacked below the curl editor,
        // anyway we need this to allow forceFocus to do it's job.
        // without this Shell won't be able to track the active control
        // and thus won't fire activation/deactivation when needed.
        return true;
    }
    
    void proxySelectionChange()
    {
        Point selection = getSelection();
        int caretOffset = getCaretOffset();
        int length = selection.y - selection.x;
        Region curlRegion = fSourceViewer.getEditor().getDocumentProxy().eclipseRegionToCurlRegion(selection.x, length);
        
        int curlSelectionAnchor;
        
        // length of selection relative to anchor (maybe 0, negative or positive)
        int curlLength;
        
        if (caretOffset == selection.y) {
            // left to right selection
            curlLength = curlRegion.getLength();
            curlSelectionAnchor = curlRegion.getOffset();
        } else {
            // right to left selection
            curlLength = curlRegion.getLength() * -1;
            curlSelectionAnchor = curlRegion.getOffset() + curlRegion.getLength();
        }
        fSourceViewer.getProxy().setSelectionRange(curlSelectionAnchor, curlLength);
    }
    
    @Override
    public Runnable print(
            Printer printer)
    {
        // TODO: return fSourceViewer.getEditor().getSourceViewProxy().???;
        return new Runnable() {
            public void run()
            {
            }
        };
    }

    // The following methods are overriden individually until we figure out how to
    // get rid of them collectively (by not extending the class, or by other means).
    @Override
    public void setStyleRange(StyleRange range)
    {
    }
   
    //    Curl editor uses some insertable key for commands, e.g. "Shift+[". 
    //    During the incremental search when you press these key sequences, 
    //    the incremental search ends and the insertable key is inserted. 
    //    This means that you cannot search for these characters, like in the 
    //    above case for '{'. To fix this problem we send the KeyPress for 
    //    keys like '{' as NotifyEvent during the incremental search so that
    //    they are never seen by the Key Binding code to convert that KeyPress
    //    to a corresponding Curl Command.    
    //    
    //    Unfortunately the IncrementalFindTarget is not a public class. Also 
    //    there are no notifiers that you can listen to, to detect when the 
    //    editor is in incremental search mode. We know that the incremental 
    //    search starts, it adds handlers to listen to the mouse events and 
    //    when it ends it removes it. We use this knowledge to detect the start 
    //    and end of incremental search.
    //     
    private boolean inIncrementalSearchMode = false;
    
    // HACK: See its usage for details in StyledText. We are using it in
    // few cases during the incremental search, so that the Key Binding
    // code does not see it (as it will convert it to a command and cancel
    // the incremental search.
    static private final int VerifyKey = 3005;
    private MouseListener incrementalSearchListener;
 
    @Override
    public void addMouseListener(
            MouseListener listener)
    {
        super.addMouseListener(listener);
        if (isIncrementalFindTarget(listener)) {
            incrementalSearchListener = listener;
            inIncrementalSearchMode = true;
        }
    }
 
    @Override
    public void removeMouseListener(
            MouseListener listener)
    {
        if (isIncrementalFindTarget(listener)) {
            incrementalSearchListener = null;
            inIncrementalSearchMode = false;
        }
        super.removeMouseListener(listener);
    }
   
    private boolean isIncrementalFindTarget(
            MouseListener listener)
    {
        String cname = listener.getClass().getName();
        return cname.equals("org.eclipse.ui.texteditor.IncrementalFindTarget"); //$NON-NLS-1$
    }
    
    @Override
    public void notifyListeners(
            int eventType,
            Event event)
    {
     
        final char character = event.character;
        boolean sendAsVerifyEvent =       
            inIncrementalSearchMode &&
            eventType == SWT.KeyDown &&
            event.stateMask == SWT.SHIFT &&
            (character == '{' || character == '}');
        
        if (sendAsVerifyEvent) {
            super.notifyListeners(VerifyKey, event);
        } else {
            super.notifyListeners(eventType, event);
        }
        
        //  Send auto complete message if needed.
        if (eventType == SWT.KeyDown) {
            // Filter out keys that are not normal keys. How?
            // FIXME: filter out other keys like: arrow keys, etc...
            if ((event.stateMask == 0 || event.stateMask == SWT.SHIFT)) {
                fSourceViewer.getProxy().sendAutoCompleteMessage(character);
            }
        }
    }
    
    private void cancelIncrementalSearch()
    {
        if (incrementalSearchListener != null) {
            Event event = new Event();
            event.button = 1;
            event.type = SWT.MouseDown;
            event.widget = this;
            MouseEvent e = new MouseEvent(event);
            incrementalSearchListener.mouseDown(e);
        }
    }
    
    public void handleCurlPointerPress()
    {
        cancelIncrementalSearch();
    }
    
    public void handleCurlFocusOut()
    {
//        // Note that we do some focus hacks because of 
//        // which the code below has some undesired side 
//        // effects. For example on Windows it was seen that
//        // two Composites will go in an infinite loop, 
//        // getting FocusIn and FocusOut (Not easy to 
//        // reproduce). I am leaving this
//        // code here as this is the right thing to do
//        // in a perfect world.
//        Event event = new Event();
//        event.type = SWT.FocusOut;
//        event.widget = this;
//        notifyListeners(event.type, event);
        cancelIncrementalSearch();
    }
}
