/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.editors;

import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IFindReplaceTarget;
import org.eclipse.jface.text.ITextViewerExtension5;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.text.source.IVerticalRuler;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.text.undo.DocumentUndoEvent;
import org.eclipse.text.undo.DocumentUndoManagerRegistry;
import org.eclipse.text.undo.IDocumentUndoListener;
import org.eclipse.text.undo.IDocumentUndoManager;

import com.curl.eclipse.remote.DocumentProxy;
import com.curl.eclipse.remote.SourceViewProxy;
import com.curl.eclipse.util.CoreUtil;

/**
 * The source viewer for the Curl editor. It interacts with the
 * CurlTextEditPanel.
 * 
 */
public class CurlSourceViewer extends SourceViewer
{
    private CurlEditor fEditor;
    private boolean fSelectionListenerEnabled = true;
    private IDocumentUndoListener fDocumentUndoListener;

    protected CurlSourceViewer(
            Composite parent,
            IVerticalRuler ruler,
            int styles,
            IPreferenceStore store)
    {
        super(parent, ruler, styles);
    }
    
    @Override
    protected void fireInputDocumentChanged(IDocument oldInput, final IDocument newInput)
    {
        super.fireInputDocumentChanged(oldInput, newInput);
        
        if (oldInput != null) {
            IDocumentUndoManager undoManager= DocumentUndoManagerRegistry.getDocumentUndoManager(oldInput);
            if (fDocumentUndoListener == null) {
                CoreUtil.logError("unexpected null fDocumentUndoListener"); //$NON-NLS-1$
            } else {
                undoManager.removeDocumentUndoListener(fDocumentUndoListener);
                fDocumentUndoListener = null;
            }
            DocumentUndoManagerRegistry.disconnect(oldInput);
        }

        if (newInput == null) {
            return;
        }

        DocumentUndoManagerRegistry.connect(newInput);

        IDocumentUndoManager undoManager= DocumentUndoManagerRegistry.getDocumentUndoManager(newInput);
        final StyledText widget= getTextWidget();

        // The default TextViewerUndoManager won't restore the selection on undo because it checks
        // widget.isFocusControl(), since we needed to "cheat" on this state (see related hacks regarding focus
        // management with Curl views), this widget.isFocusControl() always fails with a Curl Editor.
        // The document undo listener below partly duplicates the existing logic in TextViewerUndoManager
        // except it doesn't check for widget.isFocusControl().
        // see eclipse 3.3 TextViewerUndoManager#selectAndReveal
        fDocumentUndoListener = new IDocumentUndoListener(){

            public void documentUndoNotification(
                    DocumentUndoEvent event)
            {
                int eventType= event.getEventType();
                // see TextViewerUndoManager#DocumentUndoListener  
                if (((eventType & DocumentUndoEvent.UNDONE) != 0) || ((eventType & DocumentUndoEvent.REDONE) != 0))  {
                    if (widget != null && !widget.isDisposed() && // (widget.isFocusControl()))// || fTextViewer.getTextWidget() == control))
                            event.getDocument() == newInput)
                        selectAndReveal(event.getOffset(), event.getText() == null ? 0 : event.getText().length());
                }
            }
        };
        undoManager.addDocumentUndoListener(fDocumentUndoListener);

    }
    
    private void selectAndReveal(int offset, int length) 
    {
        if (this instanceof ITextViewerExtension5) {
            ITextViewerExtension5 extension= (ITextViewerExtension5) this;
            extension.exposeModelRange(new Region(offset, length));
        } else if (!overlapsWithVisibleRegion(offset, length))
            resetVisibleRegion();

        setSelectedRange(offset, length);
        revealRange(offset, length);
    }

    
    @Override
    protected StyledText createTextWidget(
            Composite parent,
            int styles)
    {
        
        return new CurlTextEditPanel(this, parent, styles);
    }

    /**
     * Respond to mediator command that selection has changed.
     *  
     * @param selectionChangedOffset
     * @param selectionChangedLength
     */
    public void selectionChangedResponse(
            int selectionChangedOffset,
            int selectionChangedLength)
    {
        try {
            fSelectionListenerEnabled = false;
            Point convertedSelection = 
                convertCurlToEclipseSelection(selectionChangedOffset, selectionChangedLength);
            final StyledText widget = getTextWidget();         
            widget.setSelectionRange(convertedSelection.x, convertedSelection.y);
            fireSelectionChanged(convertedSelection.x, convertedSelection.y);            
        } catch (IllegalArgumentException e) {
            CoreUtil.logError("Failure converting curl offset/length", e); //$NON-NLS-1$
            return;
        } catch (BadLocationException e) {
            CoreUtil.logError("Could not mediate curl selection", e); //$NON-NLS-1$
            return;
        } finally {
            fSelectionListenerEnabled = true;
        }
    }
    
    /**
     * When this is called, we know that the caret position
     * may have change and/or the selection has changed.
     */
    void handleCursorPositionChanged()
    {
        if (fSelectionListenerEnabled) {
            CurlTextEditPanel textWidget = getCurlTextEditPanel();
            textWidget.proxySelectionChange();
        }
        // else we are currently processing a selection
        // change event that was initiated by curl side
        // so no need to tell curl about new caret position.
    }


    private CurlTextEditPanel getCurlTextEditPanel()
    {
        return (CurlTextEditPanel) getTextWidget();
    }

    /**
     * Since in the Curl world the line terminations are normalized to a
     * single character, and in the eclipse world, line terminations are
     * preserved while loaded in the memory buffer, the curl offset and
     * length corresponding to a selected region must be adjusted to take
     * into account possible CRLF sequence(s) before the selected region
     * and inside the selected region. Note that the length could be
     * negative and it means the the point before the anchor.
     * 
     * @param selectionChangedOffset
     * @param selectionChangedLength
     * @param curlDocument
     * @return
     * @throws BadLocationException
     */
    private Point convertCurlToEclipseSelection(
            int selectionChangedOffset,
            int selectionChangedLength)
        throws BadLocationException
    {
        DocumentProxy curlDoc = getEditor().getDocumentProxy();
        int startOffset = selectionChangedOffset;
        int length = selectionChangedLength;
        final boolean inverted = (length < 0);
        
        if (inverted) {
            length = length * -1;
            startOffset = startOffset - length; 
        }
        
        int eclipseOffset = curlDoc.curlOffsetToEclipseOffset(startOffset);
        int eclipseLength = 
            curlDoc.curlSelectionLengthToEclipseSelectionLength(length, eclipseOffset);
        
        if (inverted) {            
            return new Point(eclipseOffset + eclipseLength, -eclipseLength);
        } else {
            return new Point(eclipseOffset, eclipseLength);
        }
    }

    public void setEditor(
            CurlEditor editor)
    {
        fEditor = editor;
    }

    public CurlEditor getEditor()
    {
        return fEditor;
    }

    public SourceViewProxy getProxy()
    {
        return fEditor.getSourceViewProxy();
    }

    @Override
    public void doOperation(
            int operation)
    {
        if (getProxy() != null && getProxy().isSupportedOperation(operation)) {
            getProxy().doOperation(operation);
        } else {
            super.doOperation(operation);
        }
    }

    @Override
    public boolean canDoOperation(
            int operation)
    {
        if (getProxy() != null && getProxy().isSupportedOperation(operation)) {
            return getProxy().canDoOperation(operation);
        } else {
            return super.canDoOperation(operation);
        }
    }

    public String getLabel(
            int operation)
    {
        if (getProxy() != null && getProxy().isSupportedOperation(operation)) {
            return getProxy().getLabel(operation);
        } else {
            return null;
        }
    }

    /**
     * Adheres to the contract of {@link IFindReplaceTarget#canPerformFind()}.
     *
     * @return <code>true</code> if find can be performed, <code>false</code> otherwise
     */
    // This is overridden to avoid calling getLength() needlessly on
    // our document, to avoid expensive calls to the Mediator.  In
    // this case we allow find operations even if there are no
    // characters in our document.
    @Override
    protected boolean canPerformFind()
    {
        return getControl() != null && getVisibleDocument() != null;
    }
    
    @Override
    public void revealRange(
            int start,
            int end)
    {
        // We don't implement this method.
    }
    
    public void setSelectionByRowLine(
            int lineNumber,
            int columnNumber,
            boolean selectToEOL)
    {
        getProxy().setSelectionByRowLine(lineNumber, columnNumber, selectToEOL);
    }

    public void enableSelectionListener(
            boolean b)
    {
        fSelectionListenerEnabled = b;
    }

}
