/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.editors;

import java.net.URI;

import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.GroupMarker;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IContributionManagerOverrides;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.IShellProvider;
import org.eclipse.search.ui.IContextMenuConstants;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.actions.ActionFactory;
import org.eclipse.ui.actions.OpenWithMenu;
import org.eclipse.ui.dialogs.PropertyDialogAction;
import org.eclipse.ui.navigator.ICommonMenuConstants;
import org.eclipse.ui.texteditor.IWorkbenchActionDefinitionIds;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.actions.OpenAction;
import com.curl.eclipse.remote.IActionRegisterListener;
import com.curl.eclipse.remote.ManifestExplorerViewProxy;
import com.curl.eclipse.remote.ProxyView;
import com.curl.eclipse.remote.WorkbenchWindowProxy;
import com.curl.eclipse.remote.WorkbenchWindowProxy.OPERATIONS;
import com.curl.eclipse.util.CurlUIIDs;
import com.curl.eclipse.util.Messages;

/**
 * TODO: remove this view when integrating curl manifest data model
 * in Common Navigator Framework.
 */
public class CurlManifestExplorerView extends CurlView
{
    private Composite fViewControl;
    private OpenWithMenu fOpenWithMenu;
    private OpenAction fOpenAction;
    private IContributionItem fContributionItem;
    private IActionRegisterListener fActionRegisteredListener;
    private ActionContributionItem fDeleteFromManifestContribution;

    public CurlManifestExplorerView() 
    {
        super(CurlUIIDs.ID_MANIFEST_EXPLORER_CONTEXT_ID);
    }
    
    @Override
    public void createPartControl(
            Composite parent)
    {
        super.createPartControl(parent);
        fViewControl = parent;
        createContextMenu();
    }
    
    private void createContextMenu()
    {
        // Create menu manager.
        final MenuManager menuMgr = new MenuManager();
        /*
         * Some actions are shared, but we need to provide some specific
         * visual attributes.
         * example: "Remove Selected Item" and "Remove Selected item from Manifest"
         * Note that there must/should be an easier way to implement this.
         */
        final IContributionManagerOverrides previousOverrides = menuMgr.getOverrides();
        menuMgr.setOverrides(new IContributionManagerOverrides(){

            @Override
            public Integer getAccelerator(
                    IContributionItem item)
            {
                return previousOverrides.getAccelerator(item);
            }

            @Override
            public String getAcceleratorText(
                    IContributionItem item)
            {
                return previousOverrides.getAcceleratorText(item);
            }

            @Override
            public Boolean getEnabled(
                    IContributionItem item)
            {
                return previousOverrides.getEnabled(item);
            }

            @Override
            public String getText(
                    IContributionItem item)
            {
                if (item == fDeleteFromManifestContribution) {
                    String label = Platform.getResourceString(
                        CurlPlugin.getDefault().getBundle(),
                        "%ProjectManifestRemoveSelectedItemFromProjectAction.label_for_popup"); //$NON-NLS-1$
                    return label; 
                }
                return previousOverrides.getText(item);
            }});
        // see Javaplugin.createStandardGroups
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_NEW));
        menuMgr.add(new GroupMarker(IContextMenuConstants.GROUP_GOTO));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_OPEN));
        menuMgr.add(new GroupMarker(IContextMenuConstants.GROUP_SHOW));
        menuMgr.add(new Separator(ICommonMenuConstants.GROUP_EDIT));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_REORGANIZE));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_GENERATE));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_SEARCH));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_BUILD));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_ADDITIONS));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_VIEWER_SETUP));
        menuMgr.add(new Separator(IContextMenuConstants.GROUP_PROPERTIES));
        menuMgr.add(new GroupMarker(IWorkbenchActionConstants.MB_ADDITIONS));
        getSite().registerContextMenu(menuMgr, fSelectionProvider);

        menuMgr.addMenuListener(new IMenuListener() {
            public void menuAboutToShow(IMenuManager mgr) {
              updateActions(mgr, fSelectionProvider.getSelection());
            }
        });
        Menu menu = menuMgr.createContextMenu(fViewControl);
        fViewControl.setMenu(menu);

        fOpenAction = new OpenAction(getSite());
        fOpenAction.setId("com.curl.eclipse.action.open"); //$NON-NLS-1$
        ISelection selection= fSelectionProvider.getSelection();
        fOpenAction.update(selection);
        fSelectionProvider.addSelectionChangedListener(fOpenAction);
        menuMgr.appendToGroup(ICommonMenuConstants.GROUP_OPEN, fOpenAction);
        fContributionItem = menuMgr.find(fOpenAction.getId());

                
        IMenuManager openWithSubmenu= new MenuManager(EditorMessages.OpenWithMenu_label); 
        menuMgr.appendToGroup(ICommonMenuConstants.GROUP_OPEN, openWithSubmenu);
        fOpenWithMenu = new OpenWithMenu(getSite().getPage(), new IAdaptable(){
            @SuppressWarnings("unchecked") 
            @Override
            public Object getAdapter(
                    Class adapter)
            {
                if (! adapter.equals(IResource.class)) {
                    return null;
                }
                ISelection selection = fSelectionProvider.getSelection();
                if (selection instanceof IStructuredSelection) {
                    IStructuredSelection s = (IStructuredSelection)selection;
                    Object firstElement = s.getFirstElement();
                    if (firstElement instanceof IResource) {
                        return firstElement;
                    } else if (firstElement instanceof IAdaptable) {
                        IAdaptable adaptable = (IAdaptable)firstElement;
                        return adaptable.getAdapter(IResource.class);
                    }
                    return null;
                }
                return null;
            }});

        openWithSubmenu.add(fOpenWithMenu);
        
        PropertyDialogAction propertiesAction = new PropertyDialogAction(new IShellProvider() {
            public Shell getShell() {
                return getViewSite().getShell();
            }
        }, fSelectionProvider);
        propertiesAction.setActionDefinitionId(IWorkbenchActionDefinitionIds.PROPERTIES); 

        getViewSite().getActionBars().setGlobalActionHandler(
                ActionFactory.PROPERTIES.getId(), propertiesAction);
        
        menuMgr.appendToGroup(ICommonMenuConstants.GROUP_PROPERTIES,
                propertiesAction);

        /*
         * actions "Edit Component Settings" and "Remove Selected" have to be shared with 
         * the related WorkBenchWindowProxy because it knows about their enablement.
         */
        WorkbenchWindowProxy workbenchWindowProxy = CurlPlugin.getDefault().getWorkbenchOperations().getWorkbenchWindowProxyFor(getSite().getWorkbenchWindow());
        fActionRegisteredListener = new IActionRegisterListener(){

            public void actionRegistered(IAction action, OPERATIONS operation) {
                if (operation == OPERATIONS.project_editcomponentsettings) {
                    menuMgr.appendToGroup(IContextMenuConstants.GROUP_BUILD,
                            action);
                }
                else if (operation == OPERATIONS.project_manifest_removeselected) {
                    fDeleteFromManifestContribution = new ActionContributionItem(action);
                    menuMgr.appendToGroup(IContextMenuConstants.GROUP_BUILD,
                            fDeleteFromManifestContribution);
                }
            }
        };
        workbenchWindowProxy.addActionRegisterListener(fActionRegisteredListener);
        
    }

    private void updateActions(IMenuManager menuManager, ISelection selection) 
    {
        boolean isFolder = false;
        String filename = ""; //$NON-NLS-1$
        if (selection instanceof IStructuredSelection) {
            IStructuredSelection structuredSelection = (IStructuredSelection)selection;
            Object firstElement = structuredSelection.getFirstElement();
            if (firstElement instanceof IResource) {
                IResource resource = (IResource)firstElement;
                URI locationURI = resource.getLocationURI();
                String pathStr = locationURI.getPath();
                int lastIndexOf = pathStr.lastIndexOf('/');
                filename =  lastIndexOf >= 0 ? pathStr.substring(lastIndexOf + 1) : pathStr;
                
                isFolder = resource instanceof IFolder;
            } else if (firstElement instanceof IFileStore) {
                IFileStore fileStore = (IFileStore)firstElement;
                IFileInfo fetchInfo = fileStore.fetchInfo();
                filename = fetchInfo.getName();
                isFolder = fetchInfo.isDirectory();
            }
        }
        if (isFolder) {
            fContributionItem.setVisible(false);
        } else {
            fContributionItem.setVisible(true);
            fOpenAction.setText(
                    Messages.format(
                            EditorMessages.OpenActionInManifestExplorer_label, filename));
        }
    }

    @Override
    public void dispose() {
        fSelectionProvider.removeSelectionChangedListener(fOpenAction);
        if (fActionRegisteredListener != null) {
            WorkbenchWindowProxy workbenchWindowProxy = CurlPlugin.getDefault().getWorkbenchOperations().getWorkbenchWindowProxyFor(getSite().getWorkbenchWindow());
            workbenchWindowProxy.removeActionRegisterListener(fActionRegisteredListener);
            fActionRegisteredListener = null;
        }
        super.dispose();
    }

    @Override
    protected ProxyView createProxy()
    {
        return new ManifestExplorerViewProxy(this);
    }

    public Menu getContextMenu()
    {
        return fViewControl.getMenu();
    }    
}