/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.editors;

import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.ui.actions.IRunToLineTarget;
import org.eclipse.debug.ui.actions.IToggleBreakpointsTarget;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.reconciler.IReconciler;
import org.eclipse.jface.text.source.IAnnotationModel;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.IVerticalRuler;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.search.ui.IQueryListener;
import org.eclipse.search.ui.ISearchQuery;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.search.ui.text.FileTextSearchScope;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.editors.text.TextEditor;
import org.eclipse.ui.editors.text.TextSourceViewerConfiguration;
import org.eclipse.ui.navigator.ICommonMenuConstants;
import org.eclipse.ui.texteditor.IStatusField;
import org.eclipse.ui.texteditor.IStatusFieldExtension;
import org.eclipse.ui.texteditor.ITextEditor;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;
import org.eclipse.ui.texteditor.IUpdate;
import org.eclipse.ui.texteditor.TextEditorAction;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.contenttype.CurlContentDescriber;
import com.curl.eclipse.contenttype.CurlContentDescriber.ValidationResult;
import com.curl.eclipse.core.CurlElement;
import com.curl.eclipse.core.SearchForEnum;
import com.curl.eclipse.files.CurlFileDocumentProvider;
import com.curl.eclipse.remote.DocumentProxy;
import com.curl.eclipse.remote.SourceViewProxy;
import com.curl.eclipse.search.CurlSearchPage;
import com.curl.eclipse.search.CurlSearchResult;
import com.curl.eclipse.search.CurlSearchScopeFactory;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;
import com.curl.eclipse.util.ICurlEditorActionDefinitionIds;
import com.curl.eclipse.util.Messages;

/**
 * Curl specific text editor for Curl source files. It has a CurlSourceViewer
 * and a CurlContentOutlinePage.
 */
public class CurlEditor extends TextEditor
{
    private Composite fEditorPaneContent;
    private SourceViewProxy fSourceViewProxy;
    private CurlContentOutlinePage fOutlinePage;
    private CurlSourceViewer fCurlSourceViewer;
    private DocumentProxy fCurlDocumentProxy;
    /** The status line manager for output */
    private IStatusLineManager fStatusLine;
    /**
     * The find status field.
     */
    private IStatusField fStatusField;
    /**
     * Tells whether the status field implements
     * <code>IStatusFieldExtension</code>.
     * @see IStatusFieldExtension
     */
    private boolean fIsStatusFieldExtension;
    
    /** This is to parse the label associated with the "goto symbol" curl command */
    private static final Pattern EDITOR_GOTO_SYMBOL_PATTERN = Pattern.compile(".*'(.*)'.*"); //$NON-NLS-1$

    /**
     * A generic action class for Curl editor actions.
     */
    public class CurlEditorAction extends Action implements IUpdate
    {
        protected final int fOperation;

        public CurlEditorAction(int operation) {
            fOperation = operation;
            update();
        }

        @Override
        public void run()
        {
            fSourceViewProxy.doOperation(fOperation);
        }

        /*
         * @see TextEditorAction#update
         */
        @Override
        public void update()
        {
            final boolean enabled = fSourceViewProxy.canDoOperation(fOperation);
            PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
                public void run()
                {
                    // This operation must be performed in the UI thread.
                    setEnabled(enabled);
                }
            });
        }
    }
    
    /**
     *
     */
    public static class SearchDefinitionsInWSAction extends TextEditorAction
    {

        protected ISearchQuery fQuery;
        private IQueryListener fQueryListener;

        protected SearchDefinitionsInWSAction(
                ResourceBundle bundle,
                String prefix,
                ITextEditor editor)
        {
            super(bundle, prefix, editor);
        }

        @Override
        public void run()
        {
            String symbol = getSymbolUnderCursor(getTextEditor());
            if (symbol != null) {
                fQuery = CurlSearchPage.createDefinitionSearchQuery(
                        CurlSearchScopeFactory.getInstance().createWorkspaceScope(), 
                        true, 
                        SearchForEnum.all,
                        symbol);
                fQueryListener = new IQueryListener(){

                    public void queryAdded(
                            ISearchQuery query)
                    {
                    }

                    public void queryFinished(
                            ISearchQuery query)
                    {
                        if (query == fQuery) {
                            NewSearchUI.removeQueryListener(this);
                            SearchDefinitionsInWSAction.this.queryFinished();
                        }
                    }

                    public void queryRemoved(
                            ISearchQuery query)
                    {
                    }

                    public void queryStarting(
                            ISearchQuery query)
                    {
                    }};
                NewSearchUI.addQueryListener(fQueryListener);
                NewSearchUI.runQueryInBackground(fQuery);
            }
        }

        protected void queryFinished()
        {
            // nothing to do, subclasses may override
        }
    }

    /**
     * Action for "Open Definition".
     * It reuses the infrastructure to perform definitions search,
     * and navigates automatically to first match if any.
     */
    public static class GotoDefinitionsInWSAction extends SearchDefinitionsInWSAction
    {

        protected GotoDefinitionsInWSAction(
                ResourceBundle bundle,
                String prefix,
                ITextEditor editor)
        {
            super(bundle, prefix, editor);
        }

        @Override
        protected void queryFinished()
        {
            final Display display = getTextEditor().getSite().getShell().getDisplay();
            display.asyncExec(new Runnable() {
                @Override
                public void run()
                {
                    CurlSearchResult curlSearchResult = (CurlSearchResult)fQuery.getSearchResult();
                    Object[] elements = curlSearchResult.getElements();
                    if (elements.length > 0 && elements[0] instanceof CurlElement) {
                        CoreUtil.openEditorOn((CurlElement)elements[0], elements.length  == 1);
                    } else {
                        // not a single definition found,
                        // since the empty search result eclipse part grabs the focus, take it back
                        GotoDefinitionsInWSAction.this.getTextEditor().setFocus();
                        // beep just like JDT does in this case....
                        display.beep();
                    }
                }
            });
        }
    }
   
    /**
     * 
     */
    public static class LanguageDocumentationAction extends TextEditorAction
    {
        protected LanguageDocumentationAction(
                ResourceBundle bundle,
                String prefix,
                ITextEditor editor)
        {
            super(bundle, prefix, editor);
        }

        @Override
        public void run()
        {
            ISelection selection = getTextEditor().getSelectionProvider().getSelection();
            String symbol = null;
            if (selection instanceof ITextSelection) {
                ITextSelection textSelection = (ITextSelection)selection;
                if (! textSelection.isEmpty()) {
                    symbol = textSelection.getText();
                }
            }
            if (symbol == null || symbol.length() == 0) {
                symbol = getSymbolUnderCursor(getTextEditor());
            }
            if (symbol == null) {
                symbol = ""; //$NON-NLS-1$
            }
            CurlPlugin.getDefault().getWorkbenchProxy().showDocumentationLanguage(symbol);
        }
    }

    /**
    *
    */
   public static class ShowInClassBrowserAction extends TextEditorAction
   {
       protected ShowInClassBrowserAction(
               ResourceBundle bundle,
               String prefix,
               ITextEditor editor)
       {
           super(bundle, prefix, editor);
       }

       @Override
       public void run()
       {
           String symbol = getSymbolUnderCursor(getTextEditor());
           if (symbol == null) {
               symbol = ""; //$NON-NLS-1$
           }
           CurlPlugin.getDefault().getWorkbenchProxy().showInClassBrowser(symbol);
       }
   }
   
    /**
     * 
     */
    public static class SearchReferencesInWSAction extends TextEditorAction
    {

        protected SearchReferencesInWSAction(
                ResourceBundle bundle,
                String prefix,
                ITextEditor editor)
        {
            super(bundle, prefix, editor);
        }

        @Override
        public void run()
        {
            String symbol = getSymbolUnderCursor(getTextEditor());
            if (symbol != null) {
                FileTextSearchScope scope = CurlSearchScopeFactory.getInstance().createWorkspaceScope();
                ISearchQuery query;
                try {
                    query = CurlSearchPage.createPlainTextSearchQuery(symbol, true, scope);
                    NewSearchUI.runQueryInBackground(query);
                } catch (CoreException e) {
                    CoreUtil.logError("Searching plain text reference to:" + symbol, e); //$NON-NLS-1$
                }
            }
        }
    }
   
    static private String getSymbolUnderCursor(
            ITextEditor textEditor)
    {
        if (textEditor instanceof CurlEditor) {
            CurlEditor curlEditor = (CurlEditor)textEditor;
            String label = curlEditor.getCurlEditorSourceViewer().getLabel(SourceViewProxy.EDITOR_GOTO_SYMBOL);
            if (label != null) {
                Matcher matcher = EDITOR_GOTO_SYMBOL_PATTERN.matcher(label);
                if (matcher.matches()) {
                    return matcher.group(1);
                }
            }
        }
        return null;
    }
    
    /**
     * Constructor 
     */
    public CurlEditor()
    {
        super();
        setDocumentProvider(CurlPlugin.getCurlFileDocumentProvider());
        /*
         * Francois's note: I observed what might be a bug in Eclipse regarding the spell-checking
         * feature. When 2 editors are opened on the same file (one if curl editor, 2nd is text editor),
         * when saving the CPU went to the roof and the UI thread was dead-locked on an explicit lock set
         * by the MonoReconciler thread.
         * So far Curl editor doesn't display the annotation gutters anyway
         * so no need to let the reconciler do it's job for now.
         */
        setSourceViewerConfiguration(new TextSourceViewerConfiguration(getPreferenceStore()) {
            @Override
            public IReconciler getReconciler(
                    ISourceViewer sourceViewer)
            {
                return null;
            }
            @Override
            public String[] getConfiguredContentTypes(ISourceViewer sourceViewer) {
                // no magic auto-indent or things like this for curl
                // given the resynchronization mechanism we don't want anything
                // to be inserted "magically"
                return new String[] {};
            }
        });
    }

    /*
     * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#initializeKeyBindingScopes()
     */
    @Override
    protected void initializeKeyBindingScopes() {
        setKeyBindingScopes(new String[] { "com.curl.eclipse.curlEditorScope" });  //$NON-NLS-1$
    }

    /**
     * Performs any extra disposal actions required by the Curl editor.
     */
    @Override
    public void dispose()
    {
        if (fOutlinePage != null) {
            fOutlinePage.setInput(null);
            fOutlinePage = null;
        }
        if (fSourceViewProxy != null) {
            fSourceViewProxy.destroy();
            fSourceViewProxy = null;
        }
        super.dispose();
    }

    /**
     * Performs any extra revert behavior required by the Curl editor.
     */
    @Override
    public void doRevertToSaved()
    {
        super.doRevertToSaved();
        if (fOutlinePage != null)
            fOutlinePage.update();
        // No mediator work left to do. The call to super interacted with the
        // CurlDocumentProvider
        // to notify the mediator of this action.
    }

    /**
     * Performs any extra save behavior required by the Curl editor.
     */
    @Override
    public void doSave(
            IProgressMonitor monitor)
    {
        try {
            if (valideEncodingAndHandleError()) {
                super.doSave(monitor);
            }
            else
            {
                if (monitor != null)
                    monitor.setCanceled(true);
            }
            
        } finally {
            if (fOutlinePage != null)
                fOutlinePage.update();
            // No mediator work left to do. The call to super interacted with the
            // CurlDocumentProvider
            // to notify the mediator of this action.
            giveBackFocusToCurlEditorIfNeeded();
        }
    }

    private boolean valideEncodingAndHandleError()
    {
        ValidationResult validationResult = valideSpecifiedEncoding();
        if (! validationResult.fSuccess) {
            String msg = Messages.format(EditorMessages.wrongEncodingMessage, validationResult.fCurlCharSetName);
            CoreUtil.displayError(EditorMessages.saveProblems, msg);
            return false;
        }
        return true;
    }

    private  ValidationResult valideSpecifiedEncoding()
    {
        try {
            // extract top fragment of document, the max size the CurlContentDescriber would get
            IDocument document = getDocumentProvider().getDocument(getEditorInput());
            int length = document.getLength();
            length = length < 512 ? length : 512;
            String topFragment = document.get(0, length);
            
            // let CurlContentDescriber parse the character set
            // Note that a call to:
            //      documentProvider.getEncoding(getEditorInput());
            // would have been cleaner if only it wasn't buggy (?!)
            // since it doesn't query for the charset option in CurlContentDescriber
            return CurlContentDescriber.validCurlCharset(topFragment);
            
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * performs any extra save as behavior required by the Curl editor.
     */
    @Override
    public void doSaveAs()
    {
        try {
            if (valideEncodingAndHandleError()) {
                super.doSaveAs();
            }
        } finally {
            if (fOutlinePage != null)
                fOutlinePage.update();
            // No mediator work left to do. The call to super interacted with the
            // CurlDocumentProvider
            // to notify the mediator of this action.
            giveBackFocusToCurlEditorIfNeeded();
        }
    }


    /**
     * If save command came from save key shortcut (e.g. CTRL+S),
     * somehow curl editor loses focus.
     */
    private void giveBackFocusToCurlEditorIfNeeded()
    {
        IWorkbenchWindow window = getSite().getWorkbenchWindow(); 
        IWorkbenchPage activePage = window.getActivePage();
        IWorkbenchPart activePart = activePage.getActivePart();
        if (activePart == this) {
            // Note that without asyncExec focus isn't transfered to Curl...
            getDisplay().asyncExec(new Runnable() {
                public void run()
                {
                    setFocus();
                }
            });
        }
    }

    /**
     * Sets the input of the outline page after AbstractTextEditor has set
     * input.
     *
     * @param input
     *            the editor input
     * @throws CoreException
     *             in case the input can not be set
     */
    @Override
    public void doSetInput(
            IEditorInput input) throws CoreException
    {
        super.doSetInput(input);
        fCurlDocumentProxy = getCurlDocumentProvider().establishDocumentProxy(input);
        if (fSourceViewProxy != null) {
            // There is already a proxy, so we must be reloading.  This can happen when search
            // preference has been set to use the same editor and just reload files in it.
            fSourceViewProxy.reloadBufferContent(fCurlDocumentProxy);
        }
        if (fOutlinePage != null) {
            fOutlinePage.setInput(input);
        }
    }

    private CurlFileDocumentProvider getCurlDocumentProvider()
    {
        return (CurlFileDocumentProvider) super.getDocumentProvider();
    }

    /**
     * Gets the content outline page
     *
     * @param required
     *            the required type
     * @return an adapter for the required type or <code>null</code>
     */
    @Override
    @SuppressWarnings("unchecked") 
    public Object getAdapter(
            Class required)
    {
        if (IContentOutlinePage.class.equals(required)) {
            if (fOutlinePage == null) {
                fOutlinePage = new CurlContentOutlinePage(getDocumentProvider(), this);
                if (getEditorInput() != null) {
                    fOutlinePage.setInput(getEditorInput());
                }
            }
            return fOutlinePage;
        } else if (required == IRunToLineTarget.class) {
            return new RunToLineAdapter(fSourceViewProxy);
        } else if (required == IToggleBreakpointsTarget.class) {
            return new ToggleBreakpointAdapter(fSourceViewProxy);
        }
        return super.getAdapter(required);
    }

    /**
     * Tell the outline view that its editor has been selected
     *
     * @see IWorkbenchPart#setFocus()
     */
    @Override
    public void setFocus()
    {
        super.setFocus();
        if (fOutlinePage != null)
            fOutlinePage.showOutlineView();
    }

    /**
     * Tell the outline view to hide itself
     */
    public void editorHidden()
    {
        if (fOutlinePage != null)
            fOutlinePage.hideOutlineView();
    }

    /**
     * Tell the outline view to show itself
     */
    public void editorBroughtToTop()
    {
        if (fOutlinePage != null)
            fOutlinePage.showOutlineView();
    }

    public CurlSourceViewer getCurlEditorSourceViewer()
    {
        return fCurlSourceViewer;
    }

    /**
     * Get some permanent display that we can use throughout the lifetime of the
     * Eclipse instance
     */
    public Display getDisplay()
    {
        return getSite().getShell().getDisplay();
    }

    /**
     * Get Editor's proxy
     */
    SourceViewProxy getSourceViewProxy()
    {
        return fSourceViewProxy;
    }

    void setAssociatedOutlineToNull()
    {
        fOutlinePage = null;
    }

    /**
     * Create Curl editor actions
     */
    @Override
    protected void createActions()
    {
        super.createActions();
        
        //
        // Remove meaningless actions for curl (set by above super.createActions()
        //
        setAction(ITextEditorActionConstants.SHIFT_LEFT, null);
        setAction(ITextEditorActionConstants.SHIFT_RIGHT, null);
        setAction(ITextEditorActionConstants.QUICK_ASSIST, null);
        setAction(ITextEditorActionConstants.SHOW_INFORMATION, null);

        //
        // Create actions specific to the Curl editor.
        //
        IAction action;
        action = new CurlEditorAction(SourceViewProxy.FIX_INDENTATION_OPERATION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.FIX_INDENTATION);
        setAction(ICurlEditorActionDefinitionIds.FixIndentationActionId, action);

        action = new CurlEditorAction(SourceViewProxy.COMMENT_OPERATION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.COMMENT);
        setAction(ICurlEditorActionDefinitionIds.CommentActionId, action);

        action = new CurlEditorAction(SourceViewProxy.UNCOMMENT_OPERATION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.UNCOMMENT);
        setAction(ICurlEditorActionDefinitionIds.UncommentActionId, action);

        action = new CurlEditorAction(SourceViewProxy.BALANCE_OPERATION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.BALANCE);
        setAction(ICurlEditorActionDefinitionIds.BalanceActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SHOW_COMPLETIONS_OPERATION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SHOW_COMPLETIONS);
        setAction(ICurlEditorActionDefinitionIds.ShowCompletionsActionId, action);

        action = new CurlEditorAction(SourceViewProxy.TOGGLE_BREAKPOINT_OPERATION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.TOGGLE_BREAKPOINT);
        setAction(ICurlEditorActionDefinitionIds.ToggleBreakpointActionId, action);

        action = new CurlEditorAction(SourceViewProxy.COPY_TO_WATCH_OPERATION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.COPY_TO_WATCH);
        setAction(ICurlEditorActionDefinitionIds.CopyToWatchActionId, action);

        action = new LanguageDocumentationAction(EditorMessages.getBundleForConstructedKeys(), "LanguageDocumentation.", this); //$NON-NLS-1$
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.LANGUAGE_DOCUMENTATION);
        setAction(ICurlEditorActionDefinitionIds.LanguageDocumentationActionId, action);

        action = new ShowInClassBrowserAction(EditorMessages.getBundleForConstructedKeys(), "ShowInClassBrowser.", this); //$NON-NLS-1$
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SHOW_IN_CLASS_BROWSER);
        setAction(ICurlEditorActionDefinitionIds.ShowInClassBrowserActionId, action);
        
        action = new SearchDefinitionsInWSAction(EditorMessages.getBundleForConstructedKeys(), "SearchDefinitionsInWS.", this); //$NON-NLS-1$
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SEARCH_DEFINITIONS_IN_WS);
        setAction(ICurlEditorActionDefinitionIds.SearchDefinitionInWSActionId, action);
        
        action = new GotoDefinitionsInWSAction(EditorMessages.getBundleForConstructedKeys(), "GotoDefinitionsInWS.", this); //$NON-NLS-1$
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.GOTO_DEFINITIONS_IN_WS);
        setAction(ICurlEditorActionDefinitionIds.GotoDefinitionInWSActionId, action);
        
        action = new SearchReferencesInWSAction(EditorMessages.getBundleForConstructedKeys(), "SearchReferencesInWS.", this); //$NON-NLS-1$
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SEARCH_REFERENCES_IN_WS);
        setAction(ICurlEditorActionDefinitionIds.SearchReferenceInWSActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.PAGE_LEFT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.PAGE_LEFT);
        setAction(ICurlEditorActionDefinitionIds.PageLeftActionId, action);

        action = new CurlEditorAction(SourceViewProxy.PAGE_RIGHT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.PAGE_RIGHT);
        setAction(ICurlEditorActionDefinitionIds.PageRightActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SELECT_PAGE_LEFT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_PAGE_LEFT);
        setAction(ICurlEditorActionDefinitionIds.SelectPageLeftActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SELECT_PAGE_RIGHT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_PAGE_RIGHT);
        setAction(ICurlEditorActionDefinitionIds.SelectPageRightActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.SCROLL_PAGE_LEFT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SCROLL_PAGE_LEFT);
        setAction(ICurlEditorActionDefinitionIds.ScrollPageLeftActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SCROLL_PAGE_RIGHT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SCROLL_PAGE_RIGHT);
        setAction(ICurlEditorActionDefinitionIds.ScrollPageRightActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.SCROLL_COLUMN_LEFT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SCROLL_COLUMN_LEFT);
        setAction(ICurlEditorActionDefinitionIds.ScrollColumnLeftActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SCROLL_COLUMN_RIGHT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SCROLL_COLUMN_RIGHT);
        setAction(ICurlEditorActionDefinitionIds.ScrollColumnRightActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.INDENTED_NEW_LINE);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.INDENTED_NEW_LINE);
        setAction(ICurlEditorActionDefinitionIds.IndentedNewLineActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.UNIT_UP);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.UNIT_UP);
        setAction(ICurlEditorActionDefinitionIds.UnitUpActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.UNIT_DOWN);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.UNIT_DOWN);
        setAction(ICurlEditorActionDefinitionIds.UnitDownActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.SELECT_UNIT_UP);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_UNIT_UP);
        setAction(ICurlEditorActionDefinitionIds.SelectUnitUpActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.SELECT_UNIT_DOWN);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_UNIT_DOWN);
        setAction(ICurlEditorActionDefinitionIds.SelectUnitDownActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.SELECT_UNIT_LEFT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_UNIT_LEFT);
        setAction(ICurlEditorActionDefinitionIds.SelectUnitLeftActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.SELECT_UNIT_RIGHT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_UNIT_RIGHT);
        setAction(ICurlEditorActionDefinitionIds.SelectUnitRightActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.EXPRESSION_UP);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.EXPRESSION_UP);
        setAction(ICurlEditorActionDefinitionIds.ExpressionUpActionId, action);

        action = new CurlEditorAction(SourceViewProxy.EXPRESSION_DOWN);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.EXPRESSION_DOWN);
        setAction(ICurlEditorActionDefinitionIds.ExpressionDownActionId, action);

        action = new CurlEditorAction(SourceViewProxy.EXPRESSION_LEFT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.EXPRESSION_LEFT);
        setAction(ICurlEditorActionDefinitionIds.ExpressionLeftActionId, action);

        action = new CurlEditorAction(SourceViewProxy.EXPRESSION_RIGHT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.EXPRESSION_RIGHT);
        setAction(ICurlEditorActionDefinitionIds.ExpressionRightActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SELECT_EXPRESSION_UP);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_EXPRESSION_UP);
        setAction(ICurlEditorActionDefinitionIds.SelectExpressionUpActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SELECT_EXPRESSION_DOWN);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_EXPRESSION_DOWN);
        setAction(ICurlEditorActionDefinitionIds.SelectExpressionDownActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SELECT_EXPRESSION_LEFT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_EXPRESSION_LEFT);
        setAction(ICurlEditorActionDefinitionIds.SelectExpressionLeftActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SELECT_EXPRESSION_RIGHT);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_EXPRESSION_RIGHT);
        setAction(ICurlEditorActionDefinitionIds.SelectExpressionRightActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SELECT_BLOCK);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SELECT_BLOCK);
        setAction(ICurlEditorActionDefinitionIds.SelectBlockActionId, action);

        action = new CurlEditorAction(SourceViewProxy.INSERT_START_BRACE);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.INSERT_START_BRACE);
        setAction(ICurlEditorActionDefinitionIds.InsertStartBraceActionId, action);

        action = new CurlEditorAction(SourceViewProxy.INSERT_END_BRACE);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.INSERT_END_BRACE);
        setAction(ICurlEditorActionDefinitionIds.InsertEndBraceActionId, action);

        action = new CurlEditorAction(SourceViewProxy.CENTER_CURSOR);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.CENTER_CURSOR);
        setAction(ICurlEditorActionDefinitionIds.CenterCursorActionId, action);

        action = new CurlEditorAction(SourceViewProxy.SWITCH_POINT_AND_ANCHOR);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.SWITCH_POINT_AND_ANCHOR);
        setAction(ICurlEditorActionDefinitionIds.SwitchPointAndAnchorActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.INSERT_TAB);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.INSERT_TAB);
        setAction(ICurlEditorActionDefinitionIds.InsertTabActionId, action);
        
        action = new CurlEditorAction(SourceViewProxy.CANCEL_SELECTION);
        action.setActionDefinitionId(ICurlEditorActionDefinitionIds.CANCEL_SELECTION);
        setAction(ICurlEditorActionDefinitionIds.CancelSelectionActionId, action);
    }

    /* (non-Javadoc)
     * @see org.eclipse.ui.editors.text.TextEditor#editorContextMenuAboutToShow(org.eclipse.jface.action.IMenuManager)
     */
    @Override
    protected void editorContextMenuAboutToShow(IMenuManager menu) {
        super.editorContextMenuAboutToShow(menu);
        addGroup(menu, ITextEditorActionConstants.GROUP_FIND, ICommonMenuConstants.GROUP_SEARCH);              
        addAction(menu, ICommonMenuConstants.GROUP_OPEN, ICurlEditorActionDefinitionIds.ShowInClassBrowserActionId);
        addAction(menu, ICommonMenuConstants.GROUP_OPEN, ICurlEditorActionDefinitionIds.LanguageDocumentationActionId);
        addAction(menu, ICommonMenuConstants.GROUP_OPEN, ICurlEditorActionDefinitionIds.GotoDefinitionInWSActionId);
        addAction(menu, ICommonMenuConstants.GROUP_SEARCH, ICurlEditorActionDefinitionIds.SearchDefinitionInWSActionId);
        addAction(menu, ICommonMenuConstants.GROUP_SEARCH, ICurlEditorActionDefinitionIds.SearchReferenceInWSActionId);
    }
    
    /**
     * @see org.eclipse.ui.texteditor.ExtendedTextEditor#createSourceViewer(org.eclipse.swt.widgets.Composite,
     *      org.eclipse.jface.text.source.IVerticalRuler, int)
     */
    @Override
    protected ISourceViewer createSourceViewer(
            Composite parent,
            IVerticalRuler ruler,
            int styles)
    {
        // fEditorPaneComposite = parent.getParent().getParent().getParent();
        fEditorPaneContent = parent;
        establishViewProxy();
        PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, CurlUIIDs.ID_EDITOR_CONTEXT_ID);
        fAnnotationAccess = createAnnotationAccess();
        fOverviewRuler = createOverviewRuler(getSharedColors());
        fCurlSourceViewer = new CurlSourceViewer(parent, ruler, styles, getPreferenceStore());
        fCurlSourceViewer.setEditor(this);
        parent.addControlListener(new ControlAdapter() {
            @Override
            public void controlResized(
                    ControlEvent e)
            {
                setNewSize();
            }

            @Override
            public void controlMoved(
                    ControlEvent e)
            {
                setNewSize();
            }
        });
        return fCurlSourceViewer;
    }

    /**
     * Establish the proxy to the mediator
     */
    private void establishViewProxy()
    {
        
        if (fSourceViewProxy == null) {
            IEditorInput editorInput = getEditorInput();
            
            Composite editorPaneContent = fEditorPaneContent;
            IProgressMonitor progressMonitor = getEditorSite().getActionBars().getStatusLineManager().getProgressMonitor();

            fSourceViewProxy =
                CurlPlugin.getDefault().getWorkbenchProxy().createSourceView(this, fCurlDocumentProxy, editorPaneContent, progressMonitor);
            
            IAnnotationModel annotationModel = getDocumentProvider().getAnnotationModel(editorInput);
            // if underline file isn't managed by workspace then there is no annotation model
            if (annotationModel != null) {
                annotationModel.addAnnotationModelListener(fSourceViewProxy);
            }
        }
    }

    /**
     * Set the new size using the proxy
     */
    private void setNewSize()
    {
        fSourceViewProxy.resizeView(fEditorPaneContent);
    }

    // This is inspired by the code in IncrementalFindTarget.class.
    public void setStatusText(String text) 
    {
        if (fStatusLine == null) {
            fStatusLine = getStatusLineManager();
        }

        if (fStatusField == null) {
            fStatusField = getStatusField(ITextEditorActionConstants.STATUS_CATEGORY_FIND_FIELD);            
            fIsStatusFieldExtension= fStatusField instanceof IStatusFieldExtension;
        }
        
        if (fStatusField != null) {
            if (fIsStatusFieldExtension) {
                IStatusFieldExtension sfe = (IStatusFieldExtension) fStatusField;
                sfe.setErrorText(null);
                fStatusField.setText(text);
                boolean visible = text.length() > 0; // !text.isEmpty();
                sfe.setVisible(visible);
                if (fStatusLine != null) {
                    fStatusLine.update(true);
                }
            } else {
                if (fStatusLine != null){
                    fStatusLine.setErrorMessage(null);
                }
                fStatusField.setText(text);
            }
        } else {
            if (fStatusLine != null) {
                fStatusLine.setErrorMessage(null);
                fStatusLine.setMessage(text);
            }
        }
    }
    
    public void updateAction(
            String action)
    {
        IAction theAction = getAction(action);
        if (theAction instanceof IUpdate) {
            ((IUpdate)theAction).update();
        }
    }

    public void selectAndRevealByRowCol(
            int row,
            int col,
            int length)
    {
        // first line in IDocument is 0
        int lineOffset;
        IDocument document = getDocumentProvider().getDocument(getEditorInput());
        try {
            lineOffset = document.getLineOffset(row);
            if (length == 0) {
                length = document.getLineLength(row) - col - 1;
            }
            selectAndReveal(lineOffset + col, length);
        } catch (BadLocationException e) {
            CoreUtil.logError("Invalid line number to select:" + row); //$NON-NLS-1$
       }
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#collectContextMenuPreferencePages()
     */
    @Override
    protected String[] collectContextMenuPreferencePages() {
        // since the text editor is managed by the native curl editor
        // there is nothing to collect from super.collectContextMenuPreferencePages();
        String[] result= new String[1];
        result[0]= "com.curl.eclipse.preferences.CurlEditorPreferencesPage"; //$NON-NLS-1$
        return result;
    }

    public void selectAndReveal(
            CurlElement curlElem)
    {
        selectAndRevealByRowCol(curlElem.getRow(), curlElem.getCol(), curlElem.getSymbolName().length());
    }

    public void activateParentView()
    {
        IWorkbenchSite site = getSite();
        IWorkbenchPage page =  site.getPage();
        
        if (page.getActivePart() != this)
        {
            page.activate(this);
            setFocus();
        }
    }

    public DocumentProxy getDocumentProxy()
    {
        return fCurlDocumentProxy;
    }

    public Menu getContextMenu()
    {
        CurlTextEditPanel textWidget = getCurlTextEditPanel();
        return textWidget.getMenu();
    }

    private CurlTextEditPanel getCurlTextEditPanel()
    {
        CurlTextEditPanel textWidget = null;
  
        ISourceViewer sourceViewer = getSourceViewer();
        if (sourceViewer != null) {
            textWidget = (CurlTextEditPanel) sourceViewer.getTextWidget();
        }
        return textWidget;
    }
    
    @Override
    protected void handleCursorPositionChanged()
    {
        super.handleCursorPositionChanged();
        fCurlSourceViewer.handleCursorPositionChanged();
    }
    
    public void handleCurlFocusOut()
    {
        CurlTextEditPanel panel = getCurlTextEditPanel();
        if (panel != null) {
            panel.handleCurlFocusOut();
        }
    }
    
    public void handleCurlPointerPress()
    {
        CurlTextEditPanel panel = getCurlTextEditPanel();
        if (panel != null) {
            panel.handleCurlPointerPress();
        }
    }
}
