/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.editors;

import java.net.URI;
import java.net.URISyntaxException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.SafeRunner;
import org.eclipse.jface.util.SafeRunnable;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.Page;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.remote.EditorOutlineViewProxy;
import com.curl.eclipse.remote.ICurlViewSelectionListener;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;

/**
 * A content outline page represents the content of the connected editor.
 */
public class CurlContentOutlinePage extends Page
    implements ICurlViewSelectionListener, IContentOutlinePage,
        ISelectionChangedListener
{
    private final ListenerList selectionChangedListeners = new ListenerList();

    protected IEditorInput fInput;
    protected IDocumentProvider fDocumentProvider;
    protected CurlEditor fCurlEditor;
    private Composite fOutlineComposite;
    private EditorOutlineViewProxy fOutlineProxy;
    private ControlListener fControlListener;
    private StructuredSelection fCurrentSelection;
    private Composite fChildOfOutlineComposite;


    /**
     * Creates a content outline page using the given provider and the given
     * editor.
     *
     * @param provider
     *            the document provider
     * @param editor
     *            the editor
     */
    public CurlContentOutlinePage(IDocumentProvider provider, CurlEditor curlEditor) {
        super();
        fDocumentProvider = provider;
        fCurlEditor = curlEditor;
    }

    /*
     * (non-Javadoc) Method declared on ContentOutlinePage
     */
    @Override
    public void createControl(
            Composite parent)
    {
        fOutlineComposite = parent;
        establishViewProxy();
        fControlListener = new ControlAdapter() {
            @Override
            public void controlResized(
                    ControlEvent e)
            {
                setNewSize();
            }

            @Override
            public void controlMoved(
                    ControlEvent e)
            {
                setNewSize();
            }
        };
        fOutlineComposite.addControlListener(fControlListener);
        // getSite().getPage().addPartListener(this);
        // PartListenerForCurlEditor.installPartListener(getSite().getPage());
    }

    /**
     * Sets the input of the outline page
     *
     * @param input
     *            the input of this outline page
     */
    public void setInput(
            IEditorInput input)
    {
        fInput = input;
        if (input == null) {
            fOutlineProxy.destroy();
            return;
        }
        update();
    }

    @Override
    public void dispose()
    {
        super.dispose();
        fCurlEditor.setAssociatedOutlineToNull();
        fOutlineProxy.destroy();
    }

    /**
     * Updates the outline page.
     */
    public void update()
    {
    }

    public void activateParentView()
    {
        IWorkbenchPage page = getSite().getPage();
        IViewReference[] viewReferences = page.getViewReferences();
        boolean requestFocus = false;
        for (IViewReference viewReference : viewReferences) {
            if (viewReference.getId().equals(IPageLayout.ID_OUTLINE)) {
                IWorkbenchPart part = viewReference.getPart(false);
                if (part == null) {
                    CoreUtil.logWarning("Could not make outline view active"); //$NON-NLS-1$
                }
                else {
                    if (page.getActivePart() != part)
                    {
                        requestFocus = true;
                        page.activate(part);
                    }
                }
                break;
            }
        }
        if (requestFocus) {
            setFocus();
        }
    }
    
    /**
     * Sets focus to a part in the page.
     */
    @Override
    public void setFocus()
    {
        getControl().setFocus();
        showOutlineView();
        fOutlineProxy.setFocus();
    }

    /**
     * Get some permanent display that we can use throughout the lifetime of the
     * Eclipse instance
     */
    public Display getDisplay()
    {
        return getSite().getShell().getDisplay();
    }

    private void establishViewProxy()
    {
        if (fOutlineProxy == null) {
            // FIXME: probably need refactoring, getControl() callers will dispose directly !!
            // so we need an extra wrapper (fChildOfOutlineComposite) so that
            // we can listen for an opportunity to reparent
            fChildOfOutlineComposite = new Composite(fOutlineComposite, SWT.NONE)
            {
                @Override
                public boolean isVisible()
                {
                    // horrible hack, see org.eclipse.swt.widgets.Control.forceFocus(),
                    // somehow, on windows at least, this component is not visible
                    // probably because it is stacked below the curl component,
                    // anyway we need this to allow forceFocus to do it's job.
                    // without this Shell won't be able to track the active control
                    // and thus won't fire activation/deactivation when needed.
                    return true;
                }
            };

            PlatformUI.getWorkbench().getHelpSystem().setHelp(fChildOfOutlineComposite, CurlUIIDs.ID_OUTLINE_CONTEXT_ID);
            fChildOfOutlineComposite.addDisposeListener(new DisposeListener(){

                public void widgetDisposed(
                        DisposeEvent e)
                {
                    Control control = getControl();
                    if (control != null) {
                        Shell safeShell = CurlPlugin.getDefault().getSafeCompositeManager().getSafeShell();
                        control.setParent(safeShell);
                    }
                }});
            
            fOutlineProxy = new EditorOutlineViewProxy(this);
            fOutlineProxy.createOutline(
                    fCurlEditor.getSourceViewProxy(),
                    fChildOfOutlineComposite,
                    getSite().getActionBars().getStatusLineManager().getProgressMonitor());
        }

    }
    
    @Override
    public void selectionChanged(
            String urlString)
    {
        try {
            // Note the similarity with the implementation of class CurlView
            // For now, listener of this outline should only care with the file resource
            // which, of course, is always the same in an outline view (since the associated editor
            // only edits one file).
            // If this becomes a problem, we could enhance to provide selection that are more
            // in line with what's expected from an outline view (Class, methods, fields, ..)
            IFile[] mappedFiles = ResourcesPlugin.getWorkspace().getRoot()
                .findFilesForLocationURI(new URI(urlString));
            fCurrentSelection = new StructuredSelection(mappedFiles);
            CurlPlugin.getDefault().getWorkbench().getDisplay().asyncExec(
                    new Runnable(){
                        @Override
                        public void run()
                        {
                            fireSelectionChanged(fCurrentSelection);
                        }
                    });
        } catch (URISyntaxException e) {
            CoreUtil.logError("CurlContentOutlinePage current selection invalid url", e); //$NON-NLS-1$
        }                        
    }

    @Override
    public ISelection getSelection()
    {
        return fCurrentSelection;
    }
    
    private void setNewSize()
    {
        if (getControl() != null && !getControl().isDisposed()) {
            fOutlineProxy.resizeOutline(fOutlineComposite);
        }
    }

    void showOutlineView()
    {
        if (getControl() != null && !getControl().isDisposed()) {
            fOutlineProxy.showOutlineView(fOutlineComposite);
            fOutlineProxy.resizeOutline(fOutlineComposite);
        }
    }

    void hideOutlineView()
    {
        if (getControl() != null && !getControl().isDisposed()) {
            fOutlineProxy.hideOutlineView(fOutlineComposite);
        }
    }
    
    @Override
    public Control getControl()
    {
        // Don't give the SafeHolder directly
        // caller will dispose of this control whenever
        // it wants to.
        return fChildOfOutlineComposite;
    }

    /* (non-Javadoc)
     * Method declared on ISelectionProvider.
     */
    public void addSelectionChangedListener(ISelectionChangedListener listener) {
        selectionChangedListeners.add(listener);
    }

    /* (non-Javadoc)
     * Method declared on ISelectionProvider.
     */
    public void removeSelectionChangedListener(
            ISelectionChangedListener listener) {
        selectionChangedListeners.remove(listener);
    }

    public void setSelection(
            ISelection selection)
    {
        // not needed for now
        CoreUtil.logInfo("CurlContentOutlinePage.setSelection not implemented"); //$NON-NLS-1$
    }

    /* (non-Javadoc)
     * Method declared on ISelectionChangeListener.
     * Gives notification that the tree selection has changed.
     */
    public void selectionChanged(SelectionChangedEvent event) {
        fireSelectionChanged(event.getSelection());
    }

    /**
     * Fires a selection changed event.
     *
     * @param selection the new selection
     */
    protected void fireSelectionChanged(ISelection selection) {
        // create an event
        final SelectionChangedEvent event = new SelectionChangedEvent(this,
                selection);

        // fire the event
        Object[] listeners = selectionChangedListeners.getListeners();
        for (int i = 0; i < listeners.length; ++i) {
            final ISelectionChangedListener l = (ISelectionChangedListener) listeners[i];
            SafeRunner.run(new SafeRunnable() {
                public void run() {
                    l.selectionChanged(event);
                }
            });
        }
    }
}
