/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;

import org.eclipse.core.filesystem.IFileInfo;
import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.provider.FileInfo;
import org.eclipse.core.filesystem.provider.FileStore;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.remote.IRemoteSourceQueryMonitor;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;
import com.curl.eclipse.util.ICurlStatusConstants;

public class RemoteSourceFileStore extends FileStore
{

    final URI fURI;
    private final FileInfo fInfo = new FileInfo();
    private volatile String fContent;
    private String fTooltip;
    private String fName;

    public RemoteSourceFileStore(
            URI aURI)
    {
        fURI = aURI;
        fInfo.setExists(true);
        fInfo.setName(fURI.getPath());
        CurlPlugin.getDefault().getWorkbenchProxy().queryRemoteSource(
                fURI,
                new IRemoteSourceQueryMonitor() {

                    public void contentReceived(
                            String content)
                    {
                        fContent = content;
                    }
                });
    }

    @Override
    public String[] childNames(
            int options,
            IProgressMonitor monitor) throws CoreException
    {
        return null;
    }

    @Override
    public IFileInfo fetchInfo(
            int options,
            IProgressMonitor monitor) throws CoreException
    {
        return fInfo;
    }

    @Override
    public IFileStore getChild(
            String name)
    {
        return null;
    }

    @Override
    public String getName()
    {
        if (fName == null) {
            String path = fURI.getPath();
            int index = path.lastIndexOf('/') + 1;
            if (index < path.length()) {
                fName = path.substring(index);
            }
            else {
                fName = fURI.getPath();
            }
        }
        return fName;
    }
    
    @Override
    public String toString()
    {
        if (fTooltip == null) {
            // use nested url (if any) as tooltip
            fTooltip = Util.normalizeURL(fURI.toString());
        }
        return fTooltip;
    }

    @Override
    public IFileStore getParent()
    {
        return null;
    }

    @Override
    public InputStream openInputStream(
            int options,
            IProgressMonitor monitor) throws CoreException
    {
        int retry = 50; // * 200 ms => 10s
        while (fContent == null && retry > 0) {
            if (monitor != null && monitor.isCanceled())
                return null;
            try {
                Thread.sleep(200);
                retry--;
            } catch (InterruptedException e) {
            }
        }
        if (fContent == null) {
            fContent = ""; //$NON-NLS-1$
            CoreUtil.logWarning("Timeout waiting for debugger to fetch source, url=" + fURI); //$NON-NLS-1$
        }
        try {
            byte bytes[] = fContent.getBytes("UTF-8"); //$NON-NLS-1$
            return new ByteArrayInputStream(bytes);
        } catch (UnsupportedEncodingException e) {
            throw new CoreException(new Status(
                    IStatus.ERROR,
                    CurlUIIDs.ID_PLUGIN,
                    ICurlStatusConstants.INTERNAL_ERROR,
                    DebugMessages.RemoteSourceFileStore_3,
                    e));
        }
    }

    @Override
    public URI toURI()
    {
        return fURI;
    }

}
