/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.io.DataInputStream;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointListener;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.IBreakpointManagerListener;
import org.eclipse.debug.core.model.IBreakpoint;

import com.curl.eclipse.remote.DataIO;
import com.curl.eclipse.remote.WorkbenchProxy;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;

/**
 * Manage (add, delete, etc.) breakpoints for the Curl debug model
 */
public class ManageBreakpoints implements IBreakpointListener, IBreakpointManagerListener
{
    final private WorkbenchProxy fWorkbenchProxy;
    
    private IBreakpointManager getBreakpointManager()
    {
        return DebugPlugin.getDefault().getBreakpointManager();
    }

    public ManageBreakpoints(WorkbenchProxy workbenchProxy)
    {
        fWorkbenchProxy = workbenchProxy;
        getBreakpointManager().addBreakpointManagerListener(this);
        getBreakpointManager().addBreakpointListener(this);
        breakpointManagerEnablementChanged(getBreakpointManager().isEnabled());
    }
    
    /**
     * Register an exception breakpoint with the Eclipse breakpoint manager.
     * The breakpoint manager will then notify the listeners, and through that
     * mechanism we tell the mediator to add the breakpoint (we are a listener).
     */
    public void registerExceptionBreakpoint(
            final CurlExceptionBreakpointDescriptor bpeDesc)
    {
        IWorkspaceRunnable registerBreakpointRunnable = new IWorkspaceRunnable() {
            public void run(
                    IProgressMonitor monitor) throws CoreException
            {
                try {
                    getBreakpointManager().addBreakpoint(new CurlExceptionBreakpoint(bpeDesc));
                } catch (CoreException e) {
                    // TODO: Is this an internal error or something we should pop up a message
                    // for the user?
                    CoreUtil.logError("Failed to add exception breakpoint", e); //$NON-NLS-1$
                }
            }
        };
        try {
            ResourcesPlugin.getWorkspace().run(
                    registerBreakpointRunnable,
                    ResourcesPlugin.getWorkspace().getRoot(),
                    IWorkspace.AVOID_UPDATE,
                    null);
        } catch (CoreException e) {
            CoreUtil.logError("Failed to run the register breakpoint workspace runnable", e); //$NON-NLS-1$
        }
    }

    /**
     * Remove all Curl breakpoints and add the ones just received in
     * one job.
     */
    public void lineBreakpointsChangedResponse(
            final DataInputStream fSockIn)
    {
        final int numBPs = DataIO.readInt(fSockIn);
        List<CurlLineBreakpointDescriptor> breakPointList = new ArrayList<CurlLineBreakpointDescriptor>();
        for (int i = 0; i < numBPs; i++) {
            boolean enabled = DataIO.readBoolean(fSockIn);
            String fileName = DataIO.readString(fSockIn);
            int lineNuber = DataIO.readInt(fSockIn);
            String condition = DataIO.readString(fSockIn);

            IResource resource = CoreUtil.getFile(fileName);
            String nonWorkspaceResourceName;
            if (resource == null)
            {
                resource = ResourcesPlugin.getWorkspace().getRoot();
                nonWorkspaceResourceName = fileName;
            }
            else {
                nonWorkspaceResourceName = null;
            }
            breakPointList.add(new CurlLineBreakpointDescriptor(enabled, resource, nonWorkspaceResourceName, lineNuber, condition));
        }
        
        final CurlLineBreakpointDescriptor[] bpDescs = breakPointList.toArray(new CurlLineBreakpointDescriptor[breakPointList.size()]);
        
        IWorkspaceRunnable changeBreakpointsRunnable = new IWorkspaceRunnable() {
            public void run(
                    IProgressMonitor monitor) throws CoreException
            {
                try {
                    // Stop listening to the breakpoint list while we are changing the list.
                    // Otherwise, the mediator would be getting notifications that these breakpoints
                    // have changed, when it is the mediator itself that has ordered these changes.
                    getBreakpointManager().removeBreakpointListener(ManageBreakpoints.this);
                    IBreakpoint[] allCurlBreakpoints = getBreakpointManager().getBreakpoints(CurlUIIDs.ID_CURL_DEBUG_MODEL);
                    for (IBreakpoint bp : allCurlBreakpoints) {
                        if (bp instanceof CurlLineBreakpoint)
                            getBreakpointManager().removeBreakpoint(bp, true);
                    }
                    for (CurlLineBreakpointDescriptor element : bpDescs) {
                        getBreakpointManager().addBreakpoint(new CurlLineBreakpoint(element));
                    }
                } catch (CoreException e) {
                    // TODO: Is this an internal error or something we should pop up a message
                    // for the user?
                    CoreUtil.logError("Failed to add breakpoint", e); //$NON-NLS-1$
                } finally {
                    getBreakpointManager().addBreakpointListener(ManageBreakpoints.this);
                }
            }
        };
        try {
            ResourcesPlugin.getWorkspace().run(
                    changeBreakpointsRunnable,
                    ResourcesPlugin.getWorkspace().getRoot(),
                    IWorkspace.AVOID_UPDATE,
                    null);
        } catch (CoreException e) {
            CoreUtil.logError("Failed to run the change breakpoints workspace runnable", e); //$NON-NLS-1$
        }
    }

    /**
     * Returns whether a breakpoint should be disabled. It's disabled if the
     * breakpoint is a exception breakpoint, and it's registered with the
     * manager and the breakpoint manager is disabled. This is used to implement
     * the Skip Breakpoints feature for exception breakpoints, since the curl
     * "skip breakpoints" support only skips line breakpoints.
     */
    private boolean isBreakpointSkipped(
            IBreakpoint breakpoint)
    {
        try {
            return breakpoint instanceof CurlExceptionBreakpoint && breakpoint.isRegistered()
                    && !getBreakpointManager().isEnabled();
        } catch (CoreException e) {
        }
        return false;
    }

    /**
     * Send to the mediator that a breakpoint has been added to the breakpoints
     * view. We monitor the actions in the breakpoint view by listening to the
     * breakpoint list.
     */
    public void breakpointAdded(
            IBreakpoint breakpoint)
    {
        if (!(breakpoint instanceof CurlBreakpoint))
            return;
        CurlBreakpoint curlBreakpoint = (CurlBreakpoint)breakpoint;
        fWorkbenchProxy.addBreakpoint(curlBreakpoint.makeDescriptor());
    }

    /**
     * Send a message to the mediator that a breakpoint has been changed in the
     * breakpoints view. The only changes that we expect are enable/disable and
     * skip breakpoints. We monitor the actions in the breakpoint view by
     * listening to the breakpoint list.
     */
    public void breakpointChanged(
            IBreakpoint breakpoint,
            IMarkerDelta delta)
    {
        if (!(breakpoint instanceof CurlBreakpoint))
            return;
        CurlBreakpoint curlBreakpoint = (CurlBreakpoint)breakpoint;
        switch (delta.getKind()) {
        case IResourceDelta.ADDED:
        case IResourceDelta.REMOVED:
            CoreUtil.logWarning("breakpointChanged is ignoring ADDED and REMOVED."); //$NON-NLS-1$
            break;
        case IResourceDelta.CHANGED:
            CurlBreakpointDescriptor bpDesc = curlBreakpoint.makeDescriptor();
            if (isBreakpointSkipped(breakpoint))
                bpDesc.fEnabled = false;
            fWorkbenchProxy.updateBreakpointEnableState(bpDesc);
            break;
        }
    }

    /**
     * Send to the mediator that a breakpoint has been removed from the breakpoints view.
     * We monitor the actions in the breakpoint view by listening to the breakpoint list.
     */
    public void breakpointRemoved(
            IBreakpoint breakpoint,
            IMarkerDelta delta)
    {
        if (!(breakpoint instanceof CurlBreakpoint))
            return;
        CurlBreakpoint curlBreakpoint = (CurlBreakpoint)breakpoint;
        fWorkbenchProxy.removeBreakpoint(curlBreakpoint.makeDescriptor());
    }

    @Override
    public void breakpointManagerEnablementChanged(
            boolean enabled)
    {
        fWorkbenchProxy.breakpointManagerEnablementChanged(enabled);
    }

    public void shutdown()
    {
        getBreakpointManager().removeBreakpointListener(this);
    }
}
