/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IDebugElement;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IValue;
import org.eclipse.debug.core.model.IWatchExpressionDelegate;
import org.eclipse.debug.core.model.IWatchExpressionListener;
import org.eclipse.debug.core.model.IWatchExpressionResult;

/**
 * Curl debug model expression evaluator
 */
public class CurlWatchExpressionDelegate implements IWatchExpressionDelegate
{
    private String fExpressionText;
    private IWatchExpressionListener fListener;
    /**
     *  Runnable to evaluate an expression in the context of a stack frame and
     *  call the expression eveluator's listener.
     */
    private final class EvaluationRunnable implements Runnable
    {
        private final CurlStackFrame fStackFrame;

        public EvaluationRunnable(
                CurlStackFrame frame)
        {
            fStackFrame = frame;
        }

        public void run()
        {
            final CurlEvaluationDescriptor evalDesc = 
                fStackFrame.getDebugTarget().evaluateExpression(fExpressionText, fStackFrame);
            fListener.watchEvaluationFinished(new IWatchExpressionResult() {
                public IValue getValue()
                {
                    if (evalDesc.fHasErrors) {
                        return null;
                    }
                    else {
                        return new CurlValue(
                            fStackFrame.getDebugTarget(), 
                            new CurlValueDescriptor(
                                    evalDesc.fVarDesc.fTypeName,
                                    evalDesc.fVarDesc.fValue,
                                    evalDesc.fVarDesc.fVariableID,
                                    evalDesc.fVarDesc.fExpandable),
                            "dummy-path-for-watch-expression"); //$NON-NLS-1$
                    }
                }

                public boolean hasErrors()
                {
                    return evalDesc.fHasErrors;
                }

                public String[] getErrorMessages()
                {
                    return new String[] {evalDesc.fErrorMessage};
                }

                public String getExpressionText()
                {
                    return evalDesc.fExpressionText;
                }

                public DebugException getException()
                {
                    return null;
                }
            });            
        }
    }
    
    public void evaluateExpression(
            String expression,
            IDebugElement context,
            IWatchExpressionListener listener)
    {
        fExpressionText = expression;
        fListener = listener;
        IStackFrame frame = null;
        if (context instanceof IStackFrame) {
            frame = (IStackFrame)context;
        } else if (context instanceof IThread) {
            try {
                frame = ((IThread)context).getTopStackFrame();
            } catch (DebugException e) {
            }
        }
        if (frame == null) {
            fListener.watchEvaluationFinished(null);
        } else {
            // consult the adapter in case of a wrappered debug model
            final CurlStackFrame curlStackFrame = (CurlStackFrame)((IAdaptable)frame).getAdapter(CurlStackFrame.class);
            if (curlStackFrame != null) {
                doEvaluation(curlStackFrame);
            } else {
                fListener.watchEvaluationFinished(null);
            }
        }
    }

    /**
     * Evaluate the expression in the context of the given stack frame.
     * The evaluation is performed asynchronously. A change debug event, with
     * this as the source, is fired when the evaluation is completed.
     */
    private void doEvaluation(
            CurlStackFrame curlStackFrame)
    {
        CurlThread thread = (CurlThread)curlStackFrame.getThread();
        if (preEvaluationCheck(thread)) {
            thread.queueRunnable(new EvaluationRunnable(curlStackFrame));
        } else {
            fListener.watchEvaluationFinished(null);
        }
    }

    private boolean preEvaluationCheck(
            CurlThread curlThread)
    {
        if (curlThread == null || curlThread.isTerminated() || !curlThread.isSuspended()) {
            return false;
        }
        return true;
    }
}
