/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.util.Vector;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.MultiStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;

import com.curl.eclipse.remote.MediatorConnection;
import com.curl.eclipse.util.CoreUtil;

/**
 * Implements an Eclipse thread.  See IThread for details.  It represents a Curl applet
 * as an Eclipse thread in the Eclipse debug model.
 * 
 * This class communicates with the Curl applet through the proxy to keep
 * the state of a running Eclipse thread (stack, breakpoints, etc.) in synch
 * with its corresponding Curl applet.
 */
public class CurlThread extends CurlDebugElement implements IThread
{
    private boolean fAtFinishedStepping;
    private int fCurrentSteppingAction;
    private boolean fCanStepInto;
    private boolean fCanStepOver;
    private boolean fCanStepOut;
    private IBreakpoint[] fBreakpoints;
    private boolean fStepping = false;
    private ThreadJob fAsyncJob;
    private CurlStackFrameDescriptor[] fSFDescsForCurrentBP;
    private CurlStackFrame[] fCurlStackFrames;

    /**
     * Manages and asynchronously executes queue of runnables associated with this Curl applet.
     */
    static class ThreadJob extends Job
    {
        private final Vector<Runnable> fRunnables;
        private final CurlThread fCurlThread;

        public ThreadJob(
                CurlThread thread)
        {
            super(DebugMessages.DebugModelAppletEvalThread);
            fCurlThread = thread;
            fRunnables = new Vector<Runnable>(5);
            setSystem(true);
        }

        public void addRunnable(
                Runnable runnable)
        {
            synchronized (fRunnables) {
                fRunnables.add(runnable);
            }
            schedule();
        }

        public boolean isEmpty()
        {
            return fRunnables.isEmpty();
        }

        @Override
        public IStatus run(
                IProgressMonitor monitor)
        {
            Object[] runnables;
            synchronized (fRunnables) {
                runnables = fRunnables.toArray();
                fRunnables.clear();
            }
            MultiStatus failed = null;
            monitor.beginTask(this.getName(), runnables.length);
            int i = 0;
            while (i < runnables.length && !fCurlThread.isTerminated() && !monitor.isCanceled()) {
                try {
                    ((Runnable)runnables[i]).run();
                } catch (Exception e) {
                    CoreUtil.logError("Evaluation failed", e); //$NON-NLS-1$
                }
                i++;
                monitor.worked(1);
            }
            monitor.done();
            if (failed == null) {
                return Status.OK_STATUS;
            }
            return failed;
        }

        @Override
        public boolean shouldRun()
        {
            return !fCurlThread.isTerminated() && !fRunnables.isEmpty();
        }
    }
        
    public CurlThread(
            CurlDebugTarget target,
            String appletID)
    {
        super(target, appletID);
    }

    synchronized public IStackFrame[] getStackFrames() throws DebugException
    {
        if (isSuspended()) {
            if (fCurlStackFrames != null)
                return fCurlStackFrames;
            fCurlStackFrames = new CurlStackFrame[fSFDescsForCurrentBP.length];
            for (int i = 0; i < fCurlStackFrames.length; i++) {
                fCurlStackFrames[i] = new CurlStackFrame(this, fSFDescsForCurrentBP[i]);
            }
            return fCurlStackFrames;
        } else {
            return new IStackFrame[0];
        }
    }

    public boolean hasStackFrames() throws DebugException
    {
        return isSuspended();
    }

    public int getPriority() throws DebugException
    {
        return 0;
    }

    synchronized public IStackFrame getTopStackFrame() throws DebugException
    {
        IStackFrame[] frames = getStackFrames();
        if (frames.length > 0) {
            return frames[0];
        }
        return null;
    }

    public String getName() throws DebugException
    {
        return fName;
    }

    synchronized public IBreakpoint[] getBreakpoints()
    {
        if (fBreakpoints == null) {
            return new IBreakpoint[0];
        }
        return fBreakpoints;
    }

    /**
     * Sets the breakpoint on which this thread is currently suspended.
     */
    protected void setBreakpoints(
            IBreakpoint[] breakpoints)
    {
        fBreakpoints = breakpoints;
    }

    synchronized protected void initializeBreakpointState()
    {
        fBreakpoints = null;
        fStepping = false;
        fCanStepInto = false;
        fCanStepOver = false;
        fCanStepOut = false;
    }
    
    public boolean canResume()
    {
        return getDebugTarget().getProcess().canResume();
    }

    public boolean canSuspend()
    {
        return getDebugTarget().getProcess().canSuspend();
    }

    public boolean isSuspended()
    {
        return getDebugTarget().getProcess().isSuspended();
    }

    public boolean canStepInto()
    {
        return isSuspended() && fCanStepInto;
    }

    public boolean canStepOver()
    {
        return isSuspended() && fCanStepOver;
    }

    public boolean canStepReturn()
    {
        return isSuspended() && fCanStepOut;
    }

    public boolean canTerminate()
    {
        return getDebugTarget().getProcess().canTerminate(); 
    }

    public boolean isTerminated()
    {
        return getDebugTarget().getProcess().isTerminated();
    }

    public boolean isStepping()
    {
        return fStepping;
    }

    synchronized public void resume() throws DebugException
    {
        if (isSuspended()) {
            fCurrentSteppingAction = DebugEvent.CLIENT_REQUEST;
            getDebugTarget().resume();
        }
    }

    public void terminate() throws DebugException
    {
        getDebugTarget().terminate();
    }

    public void suspend() throws DebugException
    {
        getDebugTarget().suspend();
    }

    synchronized public void stepInto() throws DebugException
    {
        fCurrentSteppingAction = DebugEvent.STEP_INTO;
        getDebugTarget().step(MediatorConnection.commandStepInto);
    }

    synchronized public void stepOver() throws DebugException
    {
        fCurrentSteppingAction = DebugEvent.STEP_OVER;
        getDebugTarget().step(MediatorConnection.commandStepOver);
    }

    synchronized public void stepReturn() throws DebugException
    {
        fCurrentSteppingAction = DebugEvent.STEP_RETURN;
        getDebugTarget().step(MediatorConnection.commandStepOut);
    }
    
    protected void setStepping(
            boolean stepping)
    {
        fStepping = stepping;
    }

    synchronized protected void queueRunnable(
            Runnable evaluation)
    {
        if (fAsyncJob == null) {
            fAsyncJob = new ThreadJob(this);
        }
        fAsyncJob.addRunnable(evaluation);
    }
        
    synchronized protected void setStackFrameDescsForCurrentBreakpoint(
            CurlStackFrameDescriptor[] stackFrameDescs)
    {
        fSFDescsForCurrentBP = stackFrameDescs;
        fCurlStackFrames = null;
    }

    synchronized public void setStepEnablement(
            boolean canStepInto,
            boolean canStepOver,
            boolean canStepOut)
    {
        fCanStepInto = canStepInto;
        fCanStepOver = canStepOver;
        fCanStepOut = canStepOut;
    }

    public CurlVariableDescriptor[] getStackVars(
            int frameNumber)
    {
        return (getDebugTarget()).getStackVars(frameNumber);        
    } 
    
    public int getCurrentSteppingAction()
    {
        return fCurrentSteppingAction;
    }

    public boolean isAtFinishedStepping()
    {
        return fAtFinishedStepping;
    }

    public void setAtFinishedStepping(boolean state)
    {
        fAtFinishedStepping = state;
    }
}
