/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.model.IRegisterGroup;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IThread;
import org.eclipse.debug.core.model.IVariable;

import com.curl.eclipse.remote.MediatorConnection;
import com.curl.eclipse.util.CoreUtil;
import com.ibm.icu.text.MessageFormat;
/**
 * Curl stack frame.
 */
public class CurlStackFrame extends CurlDebugElement implements IStackFrame
{
    private final CurlThread fThread;
    private IVariable[] fVariables;
    private final CurlStackFrameDescriptor fCurlStackFrameDesc;

    public CurlStackFrame(
            CurlThread thread,
            CurlStackFrameDescriptor curlStackFrameDesc)
    {
//        super((CurlDebugTarget)thread.getDebugTarget(), curlStackFrameDesc.getFunctionName());
        super(thread.getDebugTarget(), null);
        fThread = thread;
        fCurlStackFrameDesc = curlStackFrameDesc;
    }

    public IThread getThread()
    {
        return fThread;
    }
    
    
    
    private void ensureVariablesCacheLoaded()
    {
        if (fVariables == null) {
            int frameNumber = fCurlStackFrameDesc.getFrameNumber();
            CurlVariableDescriptor[] varDescs = fThread.getStackVars(frameNumber);
            fVariables = new CurlVariable[varDescs.length];
            String parentPath = getFileName() + '.' + getMethodName();
            for (int i = 0; i < varDescs.length; i++) {
                fVariables[i] = new CurlVariable(getDebugTarget(), varDescs[i], parentPath);
            }
        }
    }

    synchronized public IVariable[] getVariables() throws DebugException
    {
        ensureVariablesCacheLoaded();
        return fVariables;
    }

    synchronized public boolean hasVariables() throws DebugException
    {
        if (!fThread.isSuspended()) {
            return false;
        }
        ensureVariablesCacheLoaded();
        return fVariables.length > 0;
    }

    public int getLineNumber() throws DebugException
    {
        return fCurlStackFrameDesc.getLineNumber();
    }
    
    public String getLineNumberString()
    {
        int lineNumber = fCurlStackFrameDesc.getLineNumber();
        if (lineNumber == -1)
            return ""; //$NON-NLS-1$
        else
            return String.valueOf(lineNumber);
    }

    public int getCharStart() throws DebugException
    {
        return -1;
    }

    public int getCharEnd() throws DebugException
    {
        return -1;
    }

    public String getName() throws DebugException
    {
        return fName;
    }

    public IRegisterGroup[] getRegisterGroups() throws DebugException
    {
        return null;
    }

    public boolean hasRegisterGroups() throws DebugException
    {
        return false;
    }

    public boolean canStepInto()
    {
        return getThread().canStepInto();
    }

    public boolean canStepOver()
    {
        return getThread().canStepOver();
    }

    public boolean canStepReturn()
    {
        return getThread().canStepReturn();
    }

    public boolean isStepping()
    {
        return getThread().isStepping();
    }

    public void stepInto() throws DebugException
    {
        getThread().stepInto();
    }

    public void stepOver() throws DebugException
    {
        getThread().stepOver();
    }

    public void stepReturn() throws DebugException
    {
        getThread().stepReturn();
    }

    public boolean canResume()
    {
        return getThread().canResume();
    }

    public boolean canSuspend()
    {
        return getThread().canSuspend();
    }

    public boolean isSuspended()
    {
        return getThread().isSuspended();
    }

    public void resume() throws DebugException
    {
        getThread().resume();
    }

    public void suspend() throws DebugException
    {
        getThread().suspend();
    }

    public boolean canTerminate()
    {
        return getThread().canTerminate();
    }

    public boolean isTerminated()
    {
        return getThread().isTerminated();
    }

    public void terminate() throws DebugException
    {
        getThread().terminate();
    }

    public String getSourceName()
    {
        return (!fCurlStackFrameDesc.isMissingFrame() && fCurlStackFrameDesc.hasDebugInfo()) ? 
                    fCurlStackFrameDesc.getFileName() :
                    DebugMessages.DebugModelPresentation_2;
    }

    @Override
    public boolean equals(
            Object obj)
    {
        if (obj instanceof CurlStackFrame) {
            CurlStackFrame sf = (CurlStackFrame)obj;
                return 
                    sf.fCurlStackFrameDesc.getFileName().equals(fCurlStackFrameDesc.getFileName()) && 
                    sf.fCurlStackFrameDesc.getLineNumber() == fCurlStackFrameDesc.getLineNumber() &&
                    sf.fCurlStackFrameDesc.getFrameNumber() == fCurlStackFrameDesc.getFrameNumber();
        }
        return false;
    }

    @Override
    public int hashCode()
    {
        return getSourceName().hashCode() + fCurlStackFrameDesc.getFrameNumber();
    }

    protected int getIdentifier()
    {
        return fCurlStackFrameDesc.getFrameNumber();
    }    

    @SuppressWarnings("unchecked")
    @Override
    public Object getAdapter(
            Class adapter)
    {
        if (adapter == CurlStackFrame.class) {
            return this;
        }
        return super.getAdapter(adapter);
    }
        
    public String getFileName()
    {
        return fCurlStackFrameDesc.getFileName();
    }
    
    public String getMethodName()
    {
        return fCurlStackFrameDesc.getFunctionName();
    }
    
    public String getReason()
    {
        if (fCurlStackFrameDesc.getReason() == MediatorConnection.subcommandDebugReasonFinishedStepping)
            return DebugMessages.DebugReason3;
        else if (fCurlStackFrameDesc.getReason() == MediatorConnection.subcommandDebugReasonStoppedAtBreakpoint)
            return DebugMessages.DebugReason1;
        else if (fCurlStackFrameDesc.getReason() == MediatorConnection.subcommandDebugReasonStoppedAtException) {
            String exceptionBreakpointReason = fCurlStackFrameDesc.getExceptionBreakpointReason();
            if (exceptionBreakpointReason.length() == 0) {
                return MessageFormat.format(
                        DebugMessages.DebugReason2,
                        new String[] { fCurlStackFrameDesc.getExceptionBreakpointTypeName() });
            } else {
                return MessageFormat.format(
                        DebugMessages.DebugReason4, 
                        new String[] {fCurlStackFrameDesc.getExceptionBreakpointTypeName(), 
                                      exceptionBreakpointReason });
            }
        } else {
            CoreUtil.logError("Unknown breakpoint reason"); //$NON-NLS-1$
            return ""; //$NON-NLS-1$
        }
    }
    
    public boolean hasDebugInformation()
    {
        return this.fCurlStackFrameDesc.hasDebugInfo();
    }
}