/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;


import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.ILineBreakpoint;

import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;
import com.ibm.icu.text.MessageFormat;

/**
 * Created as user creates breakpoints in the editor.
 */
public class CurlLineBreakpoint extends CurlBreakpoint implements ILineBreakpoint
{
    /**
     * Default constructor is required for the breakpoint manager
     * to re-create persisted breakpoints. After instantiating a breakpoint,
     * the setMarker(...) method is called to restore this breakpoint's attributes.
     */
    public CurlLineBreakpoint()
    {
    }

    public CurlLineBreakpoint(
            final CurlLineBreakpointDescriptor bplDesc)
    {
        try {
            IMarker marker = bplDesc.fResource.createMarker(CurlUIIDs.ID_CURL_LINE_DEBUG_MARKER);
            setMarker(marker);
            if (bplDesc.fNonWorkspaceResourceName != null) {
                marker.setAttribute(CurlUIIDs.ATTR_DEBUG_NON_WORKSPACE_RESOURCE_NAME, bplDesc.fNonWorkspaceResourceName);
                marker.setAttribute(IMarker.TRANSIENT, Boolean.TRUE);
            }
            marker.setAttribute(IBreakpoint.ENABLED, bplDesc.fEnabled);
            marker.setAttribute(IMarker.LINE_NUMBER, bplDesc.fLineNumber);
            marker.setAttribute(CurlUIIDs.ATTR_DEBUG_CONDITION, bplDesc.fCondition);
            marker.setAttribute(IBreakpoint.ID, getModelIdentifier());
            marker.setAttribute(
                    IMarker.MESSAGE,
                    MessageFormat.format(
                            DebugMessages.CurlLineBreakpoint_0,
                            new String[] { bplDesc.fResource.getName(), 
                                           String.valueOf(bplDesc.fLineNumber) }));
        } catch (Exception e) {
            CoreUtil.logError("Curl line breakpoint could not be created.", e); //$NON-NLS-1$
        }
    }
    
    public boolean fileNameEqual(String uri)
    {
        String uriToMatch = Util.normalizeURL(uri);
        return getFileName().equals(uriToMatch);
    }

    public int getCharStart() throws CoreException
    {
        IMarker m = getMarker();
        if (m != null) {
            return m.getAttribute(IMarker.CHAR_START, -1);
        }
        return -1;
    }

    public int getCharEnd() throws CoreException
    {
        IMarker m = getMarker();
        if (m != null) {
            return m.getAttribute(IMarker.CHAR_END, -1);
        }
        return -1;
    }
    
    public boolean match(
            CurlStackFrameDescriptor curlStackFrameDesc)
    {
        try {
            if (getLineNumber() == curlStackFrameDesc.getLineNumber() && 
                fileNameEqual(curlStackFrameDesc.getFileName())) {
                return true;
            }
        } catch (Exception e) {
            CoreUtil.logError("Error matching line number", e); //$NON-NLS-1$
        }
        return false;
    }
    
    @Override
    public CurlBreakpointDescriptor makeDescriptor()
    {
        return new CurlLineBreakpointDescriptor(getMarker());        
    }

    public String getFileName()
    {
        String fileName = getMarker().getAttribute(CurlUIIDs.ATTR_DEBUG_NON_WORKSPACE_RESOURCE_NAME, null);
        if (fileName == null) {
            try {
                fileName = getMarker().getResource().getLocationURI().toString();
            } catch (Exception e) {
                CoreUtil.logError("Bad file in marker", e); //$NON-NLS-1$
                fileName = ""; //$NON-NLS-1$
            }
        }
        return fileName;
    }
}
