/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.ILaunchShortcut;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;

/**
 * Note that plugin config example in java doc of ILaunchShortcut is wrong.
 * better use help on "org.eclipse.debug.ui.launchShortcuts"
 */
public class CurlLaunchShortcut implements ILaunchShortcut
{

    public void launch(
            ISelection selection,
            String mode)
    {
        // TODO: Launching a project would require knowledge of curl manifest project,
        // wait for manifest integration with Common Navigator framework
        if (selection instanceof IStructuredSelection) {
            IStructuredSelection ss = (IStructuredSelection)selection;
            Object object = ss.getFirstElement();
            if (object instanceof IFile) {
                IFile file = (IFile)object;
                launch(file, mode);
            }
        }

    }

    public void launch(
            IEditorPart editor,
            String mode)
    {
        IEditorInput input = editor.getEditorInput();
        IFile file = (IFile) input.getAdapter(IFile.class);
        if (file == null) {
            return;
        }
        launch(file, mode);
    }

    /**
     * Launches a configuration for the given type
     */
    private void launch(IFile curlSourceFile, String mode) {
        ILaunchConfiguration config = findLaunchConfiguration(curlSourceFile, getConfigurationType());
        if (config != null) {
            DebugUITools.launch(config, mode);
        }           
    }
    
    /**
     * Locate a configuration to relaunch for the given type.  If one cannot be found, create one.
     * 
     * @return a re-usable config or <code>null</code> if none
     */
    private ILaunchConfiguration findLaunchConfiguration(IFile curlSourceFile, ILaunchConfigurationType configType) {
        List<ILaunchConfiguration> candidateConfigs = Collections.emptyList();
        try {
            ILaunchConfiguration[] configs = DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations(configType);
            candidateConfigs = new ArrayList<ILaunchConfiguration>(configs.length);
            for (ILaunchConfiguration config : configs) {
                if (config.getAttribute(CurlUIIDs.ATTR_START_FILE_NAME, "").equals(curlSourceFile.getProjectRelativePath().toString())) { //$NON-NLS-1$
                    if (config.getAttribute(CurlUIIDs.ATTR_PROJECT_NAME, "").equals(curlSourceFile.getProject().getName())) { //$NON-NLS-1$
                        candidateConfigs.add(config);
                    }
                }
            }
        } catch (CoreException e) {
            CoreUtil.logError("locating launch configuration for file=" + curlSourceFile, e); //$NON-NLS-1$
        }
        
        // If there are no existing configs associated with the IType, create one.
        // If there is exactly one config associated with the IType, return it.
        // Otherwise, if there is more than one config associated with the IType, prompt the
        // user to choose one.
        int candidateCount = candidateConfigs.size();
        if (candidateCount < 1) {
            return createConfiguration(curlSourceFile);
        } else if (candidateCount == 1) {
            return candidateConfigs.get(0);
        } else {
            // Prompt the user to choose a config.  A null result means the user
            // canceled the dialog, in which case this method returns null,
            // since canceling the dialog should also cancel launching anything.
            ILaunchConfiguration config = chooseConfiguration(candidateConfigs);
            if (config != null) {
                return config;
            }
        }
        
        return null;
    }
    
    /**
     * Show a selection dialog that allows the user to choose one of the specified
     * launch configurations.  Return the chosen config, or <code>null</code> if the
     * user canceled the dialog.
     */
    private ILaunchConfiguration chooseConfiguration(List<ILaunchConfiguration> configList) {
        IDebugModelPresentation labelProvider = DebugUITools.newDebugModelPresentation();
        ElementListSelectionDialog dialog= new ElementListSelectionDialog(getShell(), labelProvider);
        dialog.setElements(configList.toArray());
        dialog.setTitle(DebugMessages.CurlAppletLaunchShortcut_0);  
        dialog.setMessage(DebugMessages.CurlAppletLaunchShortcut_2);
        dialog.setMultipleSelection(false);
        int result = dialog.open();
        labelProvider.dispose();
        if (result == Window.OK) {
            return (ILaunchConfiguration) dialog.getFirstResult();
        }
        return null;        
    }
    
    private Shell getShell()
    {
        return CurlPlugin.getActiveWorkbenchWindow().getShell();
    }

    /* (non-Javadoc)
     * @see org.eclipse.jdt.internal.debug.ui.launcher.JavaLaunchShortcut#createConfiguration(org.eclipse.jdt.core.IType)
     */
    private ILaunchConfiguration createConfiguration(IFile curlFile) {
        ILaunchConfiguration config = null;
        ILaunchConfigurationWorkingCopy wc = null;
        try {
            ILaunchConfigurationType configType = getConfigurationType();
            wc = configType.newInstance(null, getLaunchManager().generateUniqueLaunchConfigurationNameFrom(curlFile.getName()));
            wc.setAttribute(CurlUIIDs.ATTR_START_FILE_NAME, curlFile.getProjectRelativePath().toString());
            wc.setAttribute(CurlUIIDs.ATTR_PROJECT_NAME, curlFile.getProject().getName());
            //CONTEXTLAUNCHING
            wc.setMappedResources(new IResource[] {curlFile});
            config = wc.doSave();
        } catch (CoreException e) {
            CoreUtil.logError("Launching curl file:" + curlFile, e);      //$NON-NLS-1$
        } 
        return config;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.jdt.internal.debug.ui.launcher.JavaLaunchShortcut#getConfigurationType()
     */
    private ILaunchConfigurationType getConfigurationType() {
        return getLaunchManager().getLaunchConfigurationType(CurlUIIDs.ID_CURL_APPLICATION);        
    }
    
    private ILaunchManager getLaunchManager() {
        return DebugPlugin.getDefault().getLaunchManager();
    }
}
