/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.util.Hashtable;

import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IDebugElement;
import org.eclipse.debug.core.model.IDebugTarget;
import org.eclipse.debug.core.model.IMemoryBlock;
import org.eclipse.debug.core.model.IThread;

import com.curl.eclipse.remote.DebugProxyData;
import com.curl.eclipse.remote.MediatorConnection;
import com.curl.eclipse.remote.DebugProxyData.DebugBreakpointHitProxyCommand;
import com.curl.eclipse.util.CurlUIIDs;

/**
 * Represents the debugging session (or context) for a Curl process.  
 * 
 * It has a command handler for handling debugging commands delegated from the Eclipse server, 
 * it has a process manager to manage the Curl process such as suspend, terminate, etc., 
 * and it has a thread manager to manage the Curl process thread such as obtaining stackframes or
 * taking a step, etc.
 * 
 * In essence, this is a "hub" of execution identified by a particular launch.
 */
public class CurlDebugTarget implements IDebugTarget, IDebugElement, IAdaptable
{
    public MediatorCommandHandler fMediatorCommandHandler;
    private final CurlProcess fProcess;
    private final ILaunch fLaunch;
    private final CurlThread fThread;
    private final IThread[] fThreads;
    private final String fName;
    
    /** Records the variable values collected while the thread is suspended 
     * (while user browses variable view) 
     */
    private final Hashtable<String, CurlValue> fVariableValuesRecorder;
    
    /** Stores the variable values recorded between the last 2 thread 
     * suspend states 
     */
    private final Hashtable<String, CurlValue> fVariableValueRecordPlayer;
    
    /**
     * Handle all communication received from the mediator.
     */
    public class MediatorCommandHandler
    {
        static final int fSomeSteppingAction = DebugEvent.STEP_INTO | DebugEvent.STEP_OVER | DebugEvent.STEP_RETURN; 

        /*
         *  Applet started.
         */
        public void started()
        {            
            fThread.initializeBreakpointState();
            fireEvent(new DebugEvent(fProcess, DebugEvent.CREATE));
        }

        /*
         * Applet terminated directly or indirectly (by user or not).
         * Once this method returns, no record of the proxy must exist.  Mediator has
         * already destroyed the agent.
         */
        public void terminated()
        {
            if (fProcess.isTerminated())
                return;
            fProcess.noteTerminatedState();
            fireEvent(new DebugEvent(fProcess, DebugEvent.TERMINATE));
            if (fProcess.getProxy() != null)
                fProcess.getProxy().destroy();
        }

        /*
         * Applet resumed by user picking the "resume" action or picking one of the stepping actions.
         */
        public void resumed()
        {
            final int currentSteppingAction = fThread.getCurrentSteppingAction();
            fThread.initializeBreakpointState();
            if ((currentSteppingAction & fSomeSteppingAction) != 0) 
                fThread.setStepping(true);
            getProcess().noteResumedState();
            fThread.fireResumeEvent(currentSteppingAction);
        }
        
        /*
         * Applet hit a breakpoint.  This could be a line breakpoint, an exception breakpoint, a step
         * finished, or a suspend.  The reason parameter tells us which, except that the suspend case is
         * folded into a "step finished".
         */
        public void breakpointHit(
                DebugBreakpointHitProxyCommand command)
        {
            commitCachedVariableValue();
            fThread.initializeBreakpointState();
            fThread.setStepEnablement(command.canStepInto, command.canStepOver, command.canStepOut);
            fThread.setStackFrameDescsForCurrentBreakpoint(command.stackFrameDescs);
            if (command.reason == MediatorConnection.subcommandDebugReasonFinishedStepping) {
                fThread.setAtFinishedStepping(true);
                suspended(DebugEvent.STEP_END, command.canResume, command.canSuspend, command.canTerminate);
            } else {
                IBreakpoint[] breakpoints = 
                    DebugPlugin.getDefault().getBreakpointManager().getBreakpoints(CurlUIIDs.ID_CURL_DEBUG_MODEL);
                for (IBreakpoint breakpoint : breakpoints) {
                    if (supportsBreakpoint(breakpoint)) {
                        if (breakpoint instanceof CurlLineBreakpoint && command.stackFrameDescs.length > 0) {
                            CurlLineBreakpoint curlLineBreakpoint = (CurlLineBreakpoint)breakpoint;
                            if (curlLineBreakpoint.match(command.stackFrameDescs[0])) {
                                fThread.setBreakpoints(new IBreakpoint[] { breakpoint });
                                break;
                            }
                        }
                        if (breakpoint instanceof CurlExceptionBreakpoint && command.isExceptionBreakpoint) {
                            CurlExceptionBreakpoint curlExceptionBreakpoint = (CurlExceptionBreakpoint)breakpoint;
                            if (curlExceptionBreakpoint.match(command.exceptionBreakpointTypeName)) {
                                fThread.setBreakpoints(new IBreakpoint[] { breakpoint });
                                break;
                            }
                        }
                    }
                }
                suspended(DebugEvent.BREAKPOINT, command.canResume, command.canSuspend, command.canTerminate);
            }
        }        

        private void suspended(
                final int detail,
                boolean canResume,
                boolean canSuspend,
                boolean canTerminate)
        {
            getProcess().noteSuspendedState(canResume, canSuspend, canTerminate);
            fThread.fireSuspendEvent(detail);
        }
    }

    public CurlDebugTarget(
            ILaunch launch,
            String startFileName,
            String appletID,
            String EngineID) throws CoreException
    {
        fMediatorCommandHandler = new MediatorCommandHandler();
        fLaunch = launch;
        // TODO: Perhaps we should do what PDA does:  In PDALaunchDelegate, it
        // creates a java.lang.Process using DebugPlugin.exec.  We could have CurlProcess
        // extend Process.  This would give us the in, out, error streams.
        fProcess = new CurlProcess(this, EngineID);
        launch.addProcess(fProcess);
        fProcess.fireCreationEvent();
        fThread = new CurlThread(this, appletID);
        fThreads = new IThread[] { fThread };
        fName = startFileName;
        
        fVariableValueRecordPlayer = new Hashtable<String, CurlValue>();
        fVariableValuesRecorder = new Hashtable<String, CurlValue>();
    }

    public CurlProcess getProcess()
    {
        return fProcess;
    }

    public IThread[] getThreads() throws DebugException
    {
        return fThreads;
    }

    public boolean hasThreads() throws DebugException
    {
        return (fThreads.length > 0);
    }

    public String getName() throws DebugException
    {
        return fName;
    }

    public IDebugTarget getDebugTarget()
    {
        return this;
    }

    public ILaunch getLaunch()
    {
        return fLaunch;
    }

    public void terminate() throws DebugException
    {
        getProcess().terminate();
    }

    public void resume() throws DebugException
    {
        getProcess().getProxy().resume();
    }

    public void suspend() throws DebugException
    {
        getProcess().getProxy().suspend();
    }

    public void disconnect() throws DebugException
    {
        getProcess().disconnect();
    }

    public boolean canTerminate()
    {
        return getProcess().canTerminate();
    }

    public boolean isTerminated()
    {
        return getProcess().isTerminated();
    }

    public boolean canResume()
    {
        return !isTerminated() && isSuspended();
    }

    public boolean canSuspend()
    {
        return !isTerminated() && !isSuspended();
    }

    public boolean isSuspended()
    {
        return getProcess().isSuspended();
    }

    public boolean canDisconnect()
    {
        return getProcess().canDisconnect();
    }

    public boolean isDisconnected()
    {
        return getProcess().isDisconnected();
    }

    public void breakpointAdded(
            IBreakpoint breakpoint)
    {
        // We have our own listener in ManageBreakpoints
    }

    public void breakpointRemoved(
            IBreakpoint breakpoint,
            IMarkerDelta delta)
    {
        // We have our own listener in ManageBreakpoints
    }

    public void breakpointChanged(
            IBreakpoint breakpoint,
            IMarkerDelta delta)
    {
        // We have our own listener in ManageBreakpoints
    }

    public boolean supportsBreakpoint(
            IBreakpoint breakpoint)
    {
        return breakpoint.getModelIdentifier().equals(CurlUIIDs.ID_CURL_DEBUG_MODEL);
    }

    public boolean supportsStorageRetrieval()
    {
        return false;
    }

    public IMemoryBlock getMemoryBlock(
            long startAddress,
            long length) throws DebugException
    {
        return null;
    }
    
    public String getModelIdentifier()
    {
        return CurlUIIDs.ID_CURL_DEBUG_MODEL;
    }

    @SuppressWarnings("unchecked") 
    public Object getAdapter(
            Class adapter)
    {
        if (adapter == IDebugElement.class) {
            return this;
        }
        return Platform.getAdapterManager().getAdapter(this, adapter);
    }
    
    // User directly terminated the applet from Eclipse, we have told the mediator
    // to kill the applet, now this method is supposed to be invoked to take any
    // actions after the mediator has told us that the applet is indeed terminated.
    // TODO:  Never called.  The engine has some problems with terminating a
    // process directly when applet is not suspended.  But, when it is not, what happens?
    protected void terminated()
    {
        getProcess().noteTerminatedState();
        fireEvent(new DebugEvent(fProcess, DebugEvent.TERMINATE));
    }
    
    protected CurlEvaluationDescriptor evaluateExpression(
            String expressionText,
            CurlStackFrame stackFrame)
    {
            return getProcess().getProxy().evaluateExpression(expressionText, stackFrame.getIdentifier());
    }
    
    protected void setProxy(DebugProxyData proxy)
    {
        fProcess.setProxy(proxy);
    }

    protected void step(
            int commandCode)
    {
        getProcess().getProxy().step(commandCode);
    }

    protected CurlVariableDescriptor[] getStackVars(
            int frameNumber)
    {
        return getProcess().getProxy().getStackVars(frameNumber);
    }

    private void fireEvent(
            final DebugEvent event)
    {
        final DebugPlugin manager= DebugPlugin.getDefault();
        if (manager != null) {
            manager.fireDebugEventSet(new DebugEvent[]{event});
        }
    }

    public CurlValue getPreviouslyRecordedVariableValue(
            String pathVariable)
    {
        return fVariableValueRecordPlayer.get(pathVariable);
    }

    public void recordVariableValue(
            String pathVariable,
            CurlValue curlValue)
    {
        fVariableValuesRecorder.put(pathVariable, curlValue);
    }    

    /**
     * Commit the variable values seen during last record session.
     */
    private void commitCachedVariableValue()
    {
        fVariableValueRecordPlayer.clear();
        fVariableValueRecordPlayer.putAll(fVariableValuesRecorder);
        fVariableValuesRecorder.clear();
        
        
    }

}
