/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.util.HashMap;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.IExpression;
import org.eclipse.debug.core.model.IStackFrame;
import org.eclipse.debug.core.model.IValue;
import org.eclipse.debug.core.model.IWatchExpression;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.IInstructionPointerPresentation;
import org.eclipse.debug.ui.IValueDetailListener;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.viewers.IColorProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;

import com.curl.eclipse.util.CoreUtil;
import com.ibm.icu.text.MessageFormat;

/**
 * Curl's debug model label provider.  See the JDIModelPresentation for how 
 * Java implements this class.
 */
public class CurlDebugModelPresentation extends LabelProvider implements IDebugModelPresentation,
        IColorProvider, IInstructionPointerPresentation
{
    // When DISPLAY_QUALIFIED_NAMES is set to True, provide fully qualified type names.
    public final static String DISPLAY_QUALIFIED_NAMES = "DISPLAY_QUALIFIED_NAMES"; //$NON-NLS-1$
    protected HashMap<String, Object> fAttributes = new HashMap<String, Object>(3);

    public CurlDebugModelPresentation()
    {
        super();
        setAttribute(DISPLAY_VARIABLE_TYPE_NAMES, true);
    }

    @Override
    public void dispose()
    {
        super.dispose();
        fAttributes.clear();
    }

    public void computeDetail(
            IValue value,
            IValueDetailListener listener)
    {
        //TODO:  We can do better than just returning the value:  If it is an
        // array we could return the content of the array, etc.
        listener.detailComputed(value, ((CurlValue)value).getValueString());
    }

    @Override
    public String getText(
            Object item)
    {
        try {
            boolean showQualified = isShowQualifiedNames();
            if (item instanceof CurlVariable) {
                   return getFormattedVariableText((CurlVariable) item);
            } else if (item instanceof IStackFrame) {
                return getStackFrameText((IStackFrame)item);
            } else if (item instanceof IMarker) {
                IBreakpoint breakpoint = DebugPlugin.getDefault().getBreakpointManager().getBreakpoint(
                        (IMarker)item);
                if (breakpoint != null) {
                    return getBreakpointText(breakpoint);
                }
                return null;
            } else if (item instanceof IWatchExpression) {
                return getWatchExpressionText((IWatchExpression)item);
            } else if (item instanceof IExpression) {
                return getExpressionText((IExpression)item);
            } else if (item instanceof IBreakpoint) {
                return getBreakpointText((IBreakpoint)item);
            } else if (item instanceof CurlThread) {
                return getThreadText((CurlThread)item, showQualified);
            } else if (item instanceof CurlProcess) {
                return getSurgeProcessText((CurlProcess)item);
            } else if (item instanceof CurlDebugTarget) {
                return getDebugTargetText((CurlDebugTarget)item);
            } else if (item instanceof CurlValue) {
                return getValueText((CurlValue) item);
             }
        } catch (CoreException e) {
            CoreUtil.logError("Label provider failed"); //$NON-NLS-1$
        }
        return null;
    }

    public String getFormattedVariableText(
            CurlVariable var)
    {
        String varLabel = DebugMessages.DebugModelPresentation_unknown_name;
        try {
            varLabel = var.getName();
        } catch (DebugException exception) {
        }
        CurlValue curlValue = null;
        try {
            curlValue = (CurlValue)var.getValue();
        } catch (DebugException e1) {
        }
        String typeName = DebugMessages.DebugModelPresentation_unknown_type;
        typeName = var.getActualTypeName();
        StringBuffer buff = new StringBuffer();
        buff.append(typeName);
        buff.append(' ');
        buff.append(varLabel);
        String valueString = getFormattedValueText(curlValue);
        if (valueString.length() != 0) {
            buff.append("= "); //$NON-NLS-1$
            buff.append(valueString);
        }
        return buff.toString();
    }

    public String getFormattedValueText(
            CurlValue curlValue)
    {
        String valueString = DebugMessages.DebugModelPresentation_unknown_value;
        if (curlValue != null) {
            try {
                valueString = getValueText(curlValue);
            } catch (DebugException exception) {
            }
        }
        return valueString;
    }

    private String getThreadText(
            CurlThread thread,
            boolean qualified) throws CoreException
    {
        if (thread.isTerminated()) {
                return getFormattedString(DebugMessages.DebugModelAppletTerminated_2, thread.getName());
        }
        if (thread.isStepping()) {
            return getFormattedString(DebugMessages.DebugModelAppletStepping, thread.getName());
        }
        if (!thread.isSuspended()) {
            return getFormattedString(DebugMessages.DebugModelAppletRunning, thread.getName());
        }
        
        // Suspended at: A breakpoint, or an exception, or the stepping
        // finished, or user suspended the applet.
        CurlStackFrame csf = (CurlStackFrame)thread.getTopStackFrame();
        String reasonMessage = ""; //$NON-NLS-1$
        String lineNumber = ""; //$NON-NLS-1$
        String fileName = ""; //$NON-NLS-1$
        // csf may be null if the suspended applet has already died.
        if (csf != null) {
            reasonMessage = csf.getReason();
            lineNumber = csf.getLineNumberString();
            fileName = csf.getFileName();
        }
        String lineNumberLabel;
        try {
            lineNumberLabel = (lineNumber.length() > 0 && Integer.parseInt(lineNumber) > 0) ? DebugMessages.DebugAtLineBreakpoint_1 + lineNumber : ""; //$NON-NLS-1$
        } catch (NumberFormatException e) {
            lineNumberLabel = ""; //$NON-NLS-1$
            CoreUtil.logWarning("Unexpected line number string value=" + lineNumber); //$NON-NLS-1$
        }
        return MessageFormat.format(
                DebugMessages.DebugModelAppletSuspended, 
                new String[] {
                        thread.getName(), 
                        reasonMessage,
                        lineNumberLabel,
                        (fileName.length() > 0) ? DebugMessages.DebugInFileBreakpoint_1 + fileName : ""}); //$NON-NLS-1$
    }

    private String getDebugTargetText(
            CurlDebugTarget debugTarget) throws DebugException
    {
        return getFormattedString(DebugMessages.LaunchAppletDebugTargetName, debugTarget.getName());
    }

    private String getSurgeProcessText(
            CurlProcess curlProcess) throws DebugException
    {
        return getFormattedString(DebugMessages.LaunchSurgeProcessName, curlProcess.getLabel());
    }

    /**
     * Given a JNI-style signature String for a IJavaValue, return true
     * if the signature represents an Object or an array of Objects.
     */
    public static boolean isObjectValue(
            String signature)
    {
        if (signature == null) {
            return false;
        }
        char sigChar = ' ';
        for (int i = 0; i < signature.length(); i++) {
            sigChar = signature.charAt(i);
            if (sigChar == '[') {
                return true;
            }
            break;
        }
        if ((sigChar == 'L') || (sigChar == 'Q')) {
            return true;
        }
        return false;
    }

    @Override
    public Image getImage(
            Object item)
    {
        initImageRegistries();
        try {
            if (item instanceof IMarker) {
                IBreakpoint bp = DebugPlugin.getDefault().getBreakpointManager().getBreakpoint((IMarker)item);
                if (bp instanceof CurlLineBreakpoint) {
                    return getCurlLineBreakpointImage((CurlLineBreakpoint)bp);
                } else if (bp instanceof CurlExceptionBreakpoint) {
                    return getCurlExceptionBreakpointImage((CurlExceptionBreakpoint)bp);
                }
            }
            if (item instanceof CurlLineBreakpoint) {
                return getCurlLineBreakpointImage((CurlLineBreakpoint)item);
            }
            if (item instanceof CurlExceptionBreakpoint) {
                return getCurlExceptionBreakpointImage((CurlExceptionBreakpoint)item);
            }
            if (item instanceof CurlStackFrame || item instanceof CurlThread || item instanceof CurlDebugTarget) {
                return getDebugElementImage(item);
            }
        } catch (CoreException e) {
            // no need to log errors - elements may no longer exist by the time we render them
        }
        return null;
    }

    private synchronized void initImageRegistries()
    {
        /*
        // if not initialized and this is called on the UI thread
        if (!fInitialized && Thread.currentThread().equals(JDIDebugUIPlugin.getStandardDisplay().getThread())) {
            // call get image registries to force them to be created on the UI thread
            getDebugImageRegistry();
            getJavaElementImageRegistry();
            JavaUI.getSharedImages();

        }
        */
    }

    private Image getCurlLineBreakpointImage(
            CurlLineBreakpoint breakpoint) throws CoreException
    {
        return null; // TODO
    }

    private Image getCurlExceptionBreakpointImage(
            CurlExceptionBreakpoint breakpoint) throws CoreException
    {
        return null; // TODO
    }

    private Image getDebugElementImage(
            Object element)
    {
        return null;  // TODO 
    }

    public IEditorInput getEditorInput(
            Object item)
    {
        if (item instanceof IMarker) {
            item = DebugPlugin.getDefault().getBreakpointManager().getBreakpoint((IMarker)item);
        }
        if (item instanceof CurlLineBreakpoint) {
            item = ((CurlLineBreakpoint)item).getMarker().getResource();
        }
        if (item instanceof IFileStore) {
            return CoreUtil.getEditorInput((IFileStore)item);
        }
        if (item instanceof IFile) {
            return new FileEditorInput((IFile)item);
        }
        return null;
    }

    public String getEditorId(
            IEditorInput input,
            Object inputObject)
    {
        try {
            IEditorDescriptor descriptor = IDE.getEditorDescriptor(input.getName());
            return descriptor.getId();
        } catch (PartInitException e) {
            return null;
        }
    }

    public void setAttribute(
            String id,
            Object value)
    {
        if (value == null) {
            return;
        }
        fAttributes.put(id, value);
    }

    private boolean isShowQualifiedNames()
    {
/*
        Boolean showQualified = (Boolean)fAttributes.get(DISPLAY_QUALIFIED_NAMES);
        showQualified = showQualified == null ? Boolean.FALSE : showQualified;
        return showQualified.booleanValue();
*/        
        return true;
    }


    private String getBreakpointText(
            IBreakpoint breakpoint)
    {
        if (breakpoint instanceof CurlLineBreakpoint) {
            return getBreakpointLineText((CurlLineBreakpoint)breakpoint);
        } else if (breakpoint instanceof CurlExceptionBreakpoint) {
            return getBreakpointExceptionText((CurlExceptionBreakpoint)breakpoint);
        }

        CoreUtil.logWarning("A NYI kind of breakpoint in getBreakpointText."); //$NON-NLS-1$ 
        return ""; //$NON-NLS-1$
    }

    private String getBreakpointLineText(
            CurlLineBreakpoint breakpoint)
    {
        StringBuffer label = new StringBuffer();
        String fileName = breakpoint.getFileName();
        label.append(fileName);
        appendLineNumber(breakpoint, label);
        return label.toString();
    }

    private String getBreakpointExceptionText(
            CurlExceptionBreakpoint breakpoint)
    {
        return getFormattedString(DebugMessages.DebugModelPresentation_5, breakpoint.getExceptionTypeName());    
    }

    private StringBuffer appendLineNumber(
            CurlBreakpoint breakpoint,
            StringBuffer label)
    {
        int lineNumber = 0;
        try {
            lineNumber = breakpoint.getLineNumber();
        } catch (CoreException e) {
            // Ok without it.  
        }        
        if (lineNumber > 0) {
            label.append(getFormattedString(DebugMessages.DebugModelPresentation_1, String.valueOf(lineNumber)));            
        }
        return label;
    }

    private String getStackFrameText(
            IStackFrame stackFrame) throws DebugException
    {
        CurlStackFrame frame = (CurlStackFrame)stackFrame.getAdapter(CurlStackFrame.class);
        if (frame != null) {
            StringBuffer label = new StringBuffer();
            String fileName = Util.normalizeURL(frame.getFileName());
            
            if (fileName != null && fileName.length() != 0) {
                // only display last component of fullpath filename
                int lastIndexOf = fileName.lastIndexOf('/');
                String aURLFile = 
                    lastIndexOf >= 0 && lastIndexOf + 1 < fileName.length() ? 
                            fileName.substring(lastIndexOf + 1) :
                            fileName;
                label.append(aURLFile);
                label.append(" - "); //$NON-NLS-1$
            }
            label.append(frame.getMethodName());
            int lineNumber = frame.getLineNumber();
            if (lineNumber > 0) {
                label.append(getFormattedString(DebugMessages.DebugModelPresentation_1, String.valueOf(lineNumber)));            
            }
            if (!frame.hasDebugInformation()) {
                label.append(" - " + DebugMessages.DebugModelPresentation_3); //$NON-NLS-1$
            }            return label.toString();
        }
        return null;
    }

    public static String getFormattedString(
            String key,
            String arg)
    {
        return MessageFormat.format(key, new String[] { arg });
    }

    public Color getForeground(
            Object element)
    {
        // A way to notify the user of something abnormal like a running versus "waiting" applet 
        return null;
    }

    public Color getBackground(
            Object element)
    {
        // A way to notify the user of something abnormal like a running versus "waiting" applet 
        return null;
    }

    private String getWatchExpressionText(
            IWatchExpression expression) throws DebugException
    {
        return getExpressionText(expression) +
                   (expression.isEnabled() ? "" :  DebugMessages.DebugModelPresentation_4); //$NON-NLS-1$ 
    }

    private String getExpressionText(
            IExpression expression) throws DebugException
    {
        StringBuffer buff = new StringBuffer();
        buff.append('"' + expression.getExpressionText() + '"');
        CurlValue value = (CurlValue)expression.getValue();
        if (value != null) {
            String valueString = getValueText(value);
            if (valueString.length() > 0) {
                buff.append("= "); //$NON-NLS-1$
                buff.append(valueString);
            }
        }
        return buff.toString();
    }
    
    private String getValueText(
            CurlValue value) throws DebugException
    {
        StringBuffer buffer = new StringBuffer();
        String refTypeName = value.getReferenceTypeName();
        String valueString = value.getValueString();
        buffer.append(refTypeName);
        buffer.append(' ');
        buffer.append(valueString);
        return buffer.toString();
    }

    public Annotation getInstructionPointerAnnotation(
            IEditorPart editorPart,
            IStackFrame frame)
    {
        return new CurlInstructionPointerAnnotation(frame);
    }

    public String getInstructionPointerAnnotationType(
            IEditorPart editorPart,
            IStackFrame frame)
    {
        return null;
    }

    public Image getInstructionPointerImage(
            IEditorPart editorPart,
            IStackFrame frame)
    {
        return null;
    }

    public String getInstructionPointerText(
            IEditorPart editorPart,
            IStackFrame frame)
    {
        return null;
    }

}
