/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlElementLabelProvider;
import com.curl.eclipse.util.CurlPluginImages;
import com.curl.eclipse.util.CurlProject;
import com.curl.eclipse.util.CurlStartFile;
import com.curl.eclipse.util.CurlUIIDs;
import com.ibm.icu.text.MessageFormat;

/**
 * A launch configuration tab that displays and edits a Curl project
 * and any start files that it may contain.
 */
public class CurlAppletLaunchMainTab extends AbstractLaunchConfigurationTab
{
    private Text fProjText;
    private Button fProjectBrowseButton;
    private Text fStartFileText;
    private Button fSearchButton;
    private static final String EMPTY_STRING = ""; //$NON-NLS-1$

    /**
     * Create the Curl Applet launcher configuration control that consists of:
     *     project text box, and a browse button that shows open Curl projects to choose from,
     *     start file selector, and a search button that shows start files to choose from.
     * The start files are collected from the project's manifest file as interpreted by the
     * mediator.
     */
    public void createControl(
            Composite parent)
    {
        Composite comp = new Composite(parent, SWT.NONE);
        setControl(comp);
        PlatformUI.getWorkbench().getHelpSystem().setHelp(comp, CurlUIIDs.LAUNCH_CONFIGURATION_DIALOG_LAUNCH_APPLET_MAIN_TAB);
        GridLayout topLayout = new GridLayout();
        topLayout.verticalSpacing = 0;
        comp.setLayout(topLayout);
        comp.setFont(parent.getFont());
        createProjectEditor(comp);
        createVerticalSpacer(comp, 1);
        createStartFileEditor(comp);
    }

    @Override
    public Image getImage()
    {
        // TODO:  The image is temporary
        ImageDescriptor imageDescriptor = CoreUtil.createImageDescriptor(
                CurlPlugin.getDefault().getBundle(),
                CurlPluginImages.ICONS_PATH.append("project-curl-file.gif"), //$NON-NLS-1$
                true);
        return imageDescriptor.createImage(true);
    }

    public String getName()
    {
        return DebugMessages.LaunchAppletConfigurationMainTab_7;
    }

    public void initializeFrom(
            ILaunchConfiguration config)
    {
        updateProjectFromConfig(config);
        updateStartFileFromConfig(config);
    }

    private CurlProject getSelectedCurlProject()
    {
        ISelection selection = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getSelectionService().getSelection();
        return CoreUtil.getCurlProject(selection);
    }

    public void setDefaults(
            ILaunchConfigurationWorkingCopy config)
    {
        CurlProject curlProject = getSelectedCurlProject();
        if (curlProject == null)
            config.setAttribute(CurlUIIDs.ATTR_PROJECT_NAME, EMPTY_STRING);
        else
            config.setAttribute(CurlUIIDs.ATTR_PROJECT_NAME, curlProject.getLabel());
        config.setAttribute(CurlUIIDs.ATTR_START_FILE_NAME, EMPTY_STRING);
    }

    @Override
    public boolean isValid(
            ILaunchConfiguration config)
    {
        setErrorMessage(null);
        setMessage(null);
        String name = fProjText.getText().trim();
        if (name.length() > 0) {
            IWorkspace workspace = ResourcesPlugin.getWorkspace();
            IStatus status = workspace.validateName(name, IResource.PROJECT);
            if (status.isOK()) {
                IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(name);
                if (!project.exists()) {
                    setErrorMessage(MessageFormat.format(
                            DebugMessages.LaunchAppletConfigurationMainTab_11,
                            new String[] { name }));
                    return false;
                }
                if (!project.isOpen()) {
                    setErrorMessage(MessageFormat.format(
                            DebugMessages.LaunchAppletConfigurationMainTab_12,
                            new String[] { name }));
                    return false;
                }
                if (!CoreUtil.isCurlProject(project)) {
                    setErrorMessage(MessageFormat.format(
                            DebugMessages.LaunchAppletConfigurationMainTab_14,
                            new String[] { name }));
                    return false;
                }
            } else {
                setErrorMessage(MessageFormat.format(
                        DebugMessages.LaunchAppletConfigurationMainTab_10,
                        new String[] { status.getMessage() }));
                return false;
            }
        }
        else {
            setErrorMessage(DebugMessages.LaunchAppletConfigurationMainTab_18);
            return false;
        }
        name = fStartFileText.getText().trim();
        if (name.length() == 0) {
            setErrorMessage(DebugMessages.LaunchAppletConfigurationMainTab_13);
            return false;
        }
        return isValidStartFile();
    }

    private boolean isValidStartFile()
    {
        IProject project = getWorkspaceCurlProject();
        final URL curlProjectURL = CoreUtil.getCurlProjectURL(project, false);
        String startFile = fStartFileText.getText().trim();
        final String startFileURL;
        URI startFileURI = URI.create(startFile);
        if (startFileURI.isAbsolute()) {
            final boolean isFileInProject[] = new boolean[1];
            startFileURL = startFileURI.toString();
            isFileInProject[0] = CurlPlugin.getDefault().getWorkbenchProxy().isFileInProject(curlProjectURL, startFileURL);
            // When absolute URL start file and not declared in manifest, then the project is no longer
            // used, let's make it clear to the user.
            // Note that eclipse 3.2 doesn't offer a "warning" message api, so we simply replace the default "message"
            if (!isFileInProject[0])
                setMessage(DebugMessages.LauncherDialogStartFileNotDeclared);
        }
        else {
            IResource resource = project.findMember(startFile);
            if (resource == null) {
                setErrorMessage(DebugMessages.LaunchAppletConfigurationInvalidProjectRelativeStartFile);
                return false;
            }
            startFileURL = resource.getLocationURI().toString();
        }
        boolean canRun = CurlPlugin.getDefault().getWorkbenchProxy().canRun(curlProjectURL, startFileURL);
        if (!canRun)
            setErrorMessage(DebugMessages.LaunchAppletConfigurationMainTab_17);
        return canRun;
    }

    public void performApply(
            ILaunchConfigurationWorkingCopy config)
    {
        config.setAttribute(CurlUIIDs.ATTR_PROJECT_NAME, fProjText.getText().trim());
        config.setAttribute(CurlUIIDs.ATTR_START_FILE_NAME, fStartFileText.getText().trim());
        mapResources(config);
    }

    /**
     * Create the project group consisting of the project text box and the browse button
     */
    private void createProjectEditor(
            Composite parent)
    {
        Font font = parent.getFont();
        Group projectGroup = new Group(parent, SWT.NONE);
        projectGroup.setText(DebugMessages.LaunchAppletConfigurationMainTab_0);
        projectGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        projectGroup.setLayout(new GridLayout(2, false));
        projectGroup.setFont(font);
        fProjText = new Text(projectGroup, SWT.SINGLE | SWT.BORDER);
        fProjText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        fProjText.setFont(font);
        fProjText.addModifyListener(new ModifyListener() {
            public void modifyText(
                    ModifyEvent e)
            {
                updateLaunchConfigurationDialog();
            }
        });
        fProjectBrowseButton = createPushButton(projectGroup, DebugMessages.LaunchAppletConfigurationMainTab_1, null);
        fProjectBrowseButton.addSelectionListener(new SelectionListener() {
            public void widgetDefaultSelected(
                    SelectionEvent e)
            {
            }

            public void widgetSelected(
                    SelectionEvent e)
            {
                handleProjectButtonSelected();
            }
        });
    }

    /**
     * Create the start file group consisting of the start file text box and a search button
     */
    private void createStartFileEditor(
            Composite parent)
    {
        Font font = parent.getFont();
        Group mainGroup = new Group(parent, SWT.NONE);
        mainGroup.setText(DebugMessages.LaunchAppletConfigurationMainTab_6);
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        mainGroup.setLayoutData(gd);
        GridLayout layout = new GridLayout();
        layout.numColumns = 2;
        mainGroup.setLayout(layout);
        mainGroup.setFont(font);
        fStartFileText = new Text(mainGroup, SWT.SINGLE | SWT.BORDER);
        gd = new GridData(GridData.FILL_HORIZONTAL);
        fStartFileText.setLayoutData(gd);
        fStartFileText.setFont(font);
        fStartFileText.addModifyListener(new ModifyListener() {
            public void modifyText(
                    ModifyEvent e)
            {
                updateLaunchConfigurationDialog();
            }
        });
        fSearchButton = createPushButton(mainGroup, DebugMessages.LaunchAppletConfigurationMainTab_2, null);
        fSearchButton.addSelectionListener(new SelectionListener() {
            public void widgetDefaultSelected(
                    SelectionEvent e)
            {
            }

            public void widgetSelected(
                    SelectionEvent e)
            {
                handleSearchButtonSelected();
            }
        });
    }

    private void handleProjectButtonSelected()
    {
        CurlProject project = chooseCurlProject();
        if (project == null) {
            return;
        }
        fProjText.setText(project.getLabel());
    }

    private void handleSearchButtonSelected()
    {
        CurlStartFile startFile = chooseCurlStartFile();
        if (startFile == null) {
            return;
        }
        fStartFileText.setText(startFile.getLabel());
    }

    /**
     * chooses a project from a list of open Curl projects
     */
    private CurlProject chooseCurlProject()
    {
        CurlProject[] curlProjects = CoreUtil.getCurlProjects();
        if (curlProjects.length == 0)
            return null;
        ILabelProvider labelProvider = new CurlElementLabelProvider();
        ElementListSelectionDialog dialog = new ElementListSelectionDialog(getShell(), labelProvider);
        dialog.setTitle(DebugMessages.LaunchAppletConfigurationMainTab_3);
        dialog.setMessage(DebugMessages.LaunchAppletConfigurationMainTab_4);
        dialog.setElements(curlProjects);
        CurlProject selectedCurlProject = getSelectedCurlProject();
        if (selectedCurlProject != null) {
            dialog.setInitialSelections(new Object[] { selectedCurlProject });
        }
        if (dialog.open() == Window.OK) {
            return (CurlProject)dialog.getFirstResult();
        }
        return null;
    }

    /**
     * Chooses a start file from list of start files in the selected project
     */
    private CurlStartFile chooseCurlStartFile()
    {
        /*
         * Get the start files from the mediator
         */
        URL manifestURL = getCurlProjectManifestURL();
        if (manifestURL == null) {
            return null;
        }
        IProject project = getWorkspaceCurlProject();
        List<CurlStartFile> startFilesPostProcessed = buildCurlStartFileListFrom(project);
        /*
         * Pick a start file.
         */
        ElementListSelectionDialog dialog = new ElementListSelectionDialog(getShell(), new CurlElementLabelProvider());
        dialog.setTitle(DebugMessages.LaunchAppletConfigurationMainTab_8);
        dialog.setMessage(DebugMessages.LaunchAppletConfigurationMainTab_9);
        dialog.setElements(startFilesPostProcessed.toArray(new CurlStartFile[startFilesPostProcessed.size()]));
        if (dialog.open() == Window.OK) {
            return (CurlStartFile)dialog.getFirstResult();
        }
        return null;
    }

    private IProject getWorkspaceCurlProject()
    {
        final String projectName = fProjText.getText().trim();
        IProject project = CoreUtil.getWorkspaceCurlProject(projectName);
        return project;
    }

    /**
     * When curl start file is in the selected project we display its relative path else we display the
     * absolute path (as a URL)
     */
    private List<CurlStartFile> buildCurlStartFileListFrom(
            IProject project)
    {
        List<URI> startFiles = queryCurlStartFileURI(project);
        Image image = getImage();
        List<CurlStartFile> startFilesPostProcessed = new ArrayList<CurlStartFile>();
        for (URI curlStartFileURI : startFiles) {
            IFile resourceFile = CoreUtil.getFile(curlStartFileURI);
            IPath resourceRelativePath;
            if (resourceFile != null
                    && (resourceRelativePath = resourceFile.getProjectRelativePath()) != null
                    && project.exists(resourceRelativePath)) {
                startFilesPostProcessed.add(new CurlStartFile(resourceRelativePath.toString(), image));
            } else {
                startFilesPostProcessed.add(new CurlStartFile(curlStartFileURI.toString(), image));
            }
        }
        return startFilesPostProcessed;
    }

    private List<URI> queryCurlStartFileURI(
            IProject project)
    {
        @SuppressWarnings("unchecked")  
        final List<URI>[] startFiles = new List[1];
        if (project != null) {
            final URL curlProjectURL = CoreUtil.getCurlProjectURL(project, false);
            if (curlProjectURL != null) {
                startFiles[0] = CurlPlugin.getDefault().getWorkbenchProxy().queryCurlStartFiles(curlProjectURL);
            }
        }
        return startFiles[0];
    }

    private URL getCurlProjectManifestURL()
    {
        String projectName = fProjText.getText().trim();
        if (projectName.length() < 1) {
            return null;
        }
        return CoreUtil.getCurlProjectManifestURL(projectName);
    }

    private void updateProjectFromConfig(
            ILaunchConfiguration config)
    {
        String projectName = EMPTY_STRING;
        try {
            projectName = config.getAttribute(CurlUIIDs.ATTR_PROJECT_NAME, EMPTY_STRING);
        }//end try
        catch (CoreException ce) {
            CoreUtil.logError("Project name attribute could not be read", ce); //$NON-NLS-1$
        }
        fProjText.setText(projectName);
    }

    private void updateStartFileFromConfig(
            ILaunchConfiguration config)
    {
        String startFileName = EMPTY_STRING;
        try {
            startFileName = config.getAttribute(CurlUIIDs.ATTR_START_FILE_NAME, EMPTY_STRING);
        } catch (CoreException ce) {
            CoreUtil.logError("Start file name attribute could not be read.", ce); //$NON-NLS-1$
        }
        fStartFileText.setText(startFileName);
    }

    private IProject getCurlProject()
    {
        String projectName = fProjText.getText().trim();
        if (projectName.length() < 1) {
            return null;
        }
        return CoreUtil.getWorkspaceCurlProject(projectName);
    }

    private void mapResources(
            ILaunchConfigurationWorkingCopy config)
    {
        IProject project = getCurlProject();
        IResource[] resources = null;
        if (project != null) {
            resources = new IResource[] { project };
        }
        config.setMappedResources(resources);
    }
}
