/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.io.DataInputStream;
import java.net.URL;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.IBreakpointManager;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.model.IBreakpoint;
import org.eclipse.debug.core.model.LaunchConfigurationDelegate;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.ui.PlatformUI;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.remote.DataIO;
import com.curl.eclipse.remote.DebugProxyData;
import com.curl.eclipse.remote.ProxyFactory;
import com.curl.eclipse.remote.WorkbenchProxy;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlUIIDs;
import com.ibm.icu.text.MessageFormat;

/**
 * A launch delegate for launching Curl applets.
 */
public class CurlAppletLaunchDelegate extends LaunchConfigurationDelegate
{
    public CurlAppletLaunchDelegate()
    {
        super();
    }

    /**
     * Called from the Eclipse server to handle a "started" message.  It is this
     * message that starts the debugging session.  This message can arrive after the
     * user interactively launched an applet using a start file, or it can arrive from
     * a Curl applet that originated from some other action.  We mimick what the debug
     * plugin does to setup a debug session.
     */
    synchronized public static void appletStartedResponse(
            DataInputStream fSockIn)
    {
        final String appletID = DataIO.readString(fSockIn);
        final String appletFile = DataIO.readString(fSockIn);
        // This method is separated so we can do unit testing without reading from the socket
        CurlAppletLaunchDelegate.debugAppletStarted(appletID, appletFile);
    }

    synchronized public static void debugAppletStarted(
            String appletID,
            String appletFile)
    {
        /*
         * Establish the proxy before launching, otherwise we could get a message
         * from the mediator (such as debug break) that we would not know what to do with.
         * Note that at this time we do not have a debug target for this proxy.  We set the debug
         * target in the launch method.
         */
        establishProxy(appletID);

        /*
         * Launch in debug JIT mode
         */
        try {
            ILaunchConfigurationType configType =
                DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurationType(
                        CurlUIIDs.ID_CURL_APPLICATION);
            // TODO: How are we going to get the launch configuration name?
            final ILaunchConfigurationWorkingCopy workingCopy = configType.newInstance(null, "Curl Configuration"); //$NON-NLS-1$
            workingCopy.setAttribute(CurlUIIDs.ATTR_DEBUG_JIT_MODE, true);
            workingCopy.setAttribute(IDebugUIConstants.ATTR_LAUNCH_IN_BACKGROUND, true);
            workingCopy.setAttribute(CurlUIIDs.ATTR_DEBUG_APPLET_ID, appletID);
            workingCopy.setAttribute(CurlUIIDs.ATTR_DEBUG_ENGINE_ID, appletID.substring(0, appletID.indexOf("-"))); //$NON-NLS-1$
            workingCopy.setAttribute(CurlUIIDs.ATTR_APPLET_FILE_FULL_PATH, new URL(appletFile).getPath());
            PlatformUI.getWorkbench().getDisplay().syncExec(new Runnable() {
                public void run()
                {
                    DebugUITools.launch(workingCopy, "debug"); //$NON-NLS-1$
                }
            });
            // workingCopy.doSave();
        } catch (Exception e) {
            CoreUtil.logError("Could not start debug session for applet", e); //$NON-NLS-1$
        }
    }

    /**
     * This method is called in two ways: When creating a launch from the debug dialog, and when a Debug
     * Just-In-Time (JIT) session is starting. A debug JIT starts when we receive a "started" message from the
     * mediator. It says that a Curl process wants to start a debugging session. When called as result of the
     * debug dialog we send the start file to the mediator to start and remove this launch in anticipation of
     * a "started" message that we will receive once the applet is ready to be debugged. Note that we have no
     * way to associate this launch with the applet that the RTE finally starts. Hence, we have no choice to
     * remove this one and create a brand new one for every subsequent "started" applet. There could be zero
     * or more applets starting as result of launching a single start file. The (known) consequence of
     * removing the launch and then adding the launch is that the debug tab will not show the name of the
     * configuration that was used for the configuration instance. It will show the url of the launched
     * applet!
     */
    synchronized public void launch(
            ILaunchConfiguration configuration,
            String mode,
            ILaunch launch,
            IProgressMonitor monitor) throws CoreException
    {
        if (monitor == null) {
            monitor = new NullProgressMonitor();
        }
        monitor.beginTask(configuration.getName(), 2);
        monitor.subTask(DebugMessages.LaunchVerifyingLaunchAttributes);
        monitor.worked(1);
        if (monitor.isCanceled()) {
            return;
        }
        if (isDebugJITMode(configuration)) {
            CurlDebugTarget curlDebugTarget =
                new CurlDebugTarget(
                    launch,
                    getAppletFileName(configuration),
                    getAppletID(configuration),
                    getEngineID(configuration));
            // The proxy was established when the started message was received, but it
            // was created without a "proxy of".  Now is the time to connect the proxy to its
            // "proxy of".  Once "proxy of" is set, then the Curl debug session has officially started!
            // Now any messages from the mediator (such as a debug break) that may have been
            // waiting to be processed can be processed.
            DebugProxyData proxy = ProxyFactory.getDebugProxyData(getAppletID(configuration));
            curlDebugTarget.setProxy(proxy);
            proxy.setProxyOf(curlDebugTarget);
            launch.addDebugTarget(curlDebugTarget);
            proxy.setStructurallyComplete();
            curlDebugTarget.fMediatorCommandHandler.started();
            CoreUtil.logInfo("Launched debug JIT session for: " + getAppletFileName(configuration)); //$NON-NLS-1$
        } else {
            try {
                DebugPlugin.getDefault().getLaunchManager().removeLaunch(launch);
                URL startFileURL = getStartFileURL(configuration);
                WorkbenchProxy workbenchProxy = CurlPlugin.getDefault().getWorkbenchProxy();
                workbenchProxy.launchAppletForDebugging(startFileURL.toString());
            } catch (InvalidConfigurationException e) {
                CoreUtil.displayError(DebugMessages.LauncherDialogTitle, e.getMessage());
            }
        }
        monitor.done();
    }

    @Override
    synchronized public boolean preLaunchCheck(
            ILaunchConfiguration configuration,
            String mode,
            IProgressMonitor monitor) throws CoreException
    {
        // do generic launch checks
        return super.preLaunchCheck(configuration, mode, monitor);
    }

    @Override
    synchronized protected IBreakpoint[] getBreakpoints(
            ILaunchConfiguration configuration)
    {
        IBreakpointManager breakpointManager = DebugPlugin.getDefault().getBreakpointManager();
        if (!breakpointManager.isEnabled()) {
            // no need to check breakpoints individually.
            return null;
        }
        return breakpointManager.getBreakpoints(CurlUIIDs.ID_PLUGIN);
    }

    @Override
    public boolean buildForLaunch(
            ILaunchConfiguration configuration,
            String mode,
            IProgressMonitor monitor) throws CoreException
    {
        return false;
    }

    /*
     private URL verifyCurlProject(
     ILaunchConfiguration configuration) throws CoreException
     {
     String name = getCurlProjectName(configuration);
     if (name == null || name.length() == 0) {
     CoreUtil.abort(DebugMessages.LaunchProjectNotSpecified, null);
     }
     URL manifestURL = CoreUtil.getCurlProjectFileURL(name, "manifest.mcurl");  //$NON-NLS-1$
     if (manifestURL == null) {
     CoreUtil.abort(DebugMessages.LaunchProjectDoesNotExist, null);
     }
     return manifestURL;
     }
     */
    private URL getStartFileURL(
            ILaunchConfiguration configuration) throws CoreException, InvalidConfigurationException
    {
        String appletFileName = getStartFileName(configuration);
        String projectName = getCurlProjectName(configuration);
        if (projectName == null || projectName.length() == 0 || appletFileName == null
                || appletFileName.length() == 0) {
            CoreUtil.abort(DebugMessages.LaunchAppletFileNotSpecified, null);
        }
        URL appletURL = CoreUtil.getCurlProjectFileURL(projectName, appletFileName);
        if (appletURL == null)
            throw new InvalidConfigurationException(
                        MessageFormat.format(
                            DebugMessages.CouldNotLocateLaunchAppletFile,
                            new String[] { appletFileName, projectName }));
        return appletURL;
    }

    private static class InvalidConfigurationException extends Exception
    {
        /**
         *
         */
        private static final long serialVersionUID = 1L;

        public InvalidConfigurationException(String message)
        {
            super(message);
        }
    }

    private String getCurlProjectName(
            ILaunchConfiguration configuration) throws CoreException
    {
        return configuration.getAttribute(CurlUIIDs.ATTR_PROJECT_NAME, (String)null);
    }

    private String getAppletFileName(
            ILaunchConfiguration configuration) throws CoreException
    {
        return configuration.getAttribute(CurlUIIDs.ATTR_APPLET_FILE_FULL_PATH, (String)null);
    }

    private String getStartFileName(
            ILaunchConfiguration configuration) throws CoreException
    {
        return configuration.getAttribute(CurlUIIDs.ATTR_START_FILE_NAME, (String)null);
    }

    private String getAppletID(
            ILaunchConfiguration configuration) throws CoreException
    {
        return configuration.getAttribute(CurlUIIDs.ATTR_DEBUG_APPLET_ID, ""); //$NON-NLS-1$
    }

    private String getEngineID(
            ILaunchConfiguration configuration) throws CoreException
    {
        return configuration.getAttribute(CurlUIIDs.ATTR_DEBUG_ENGINE_ID, ""); //$NON-NLS-1$
    }

    private boolean isDebugJITMode(
            ILaunchConfiguration configuration)
    {
        try {
            return configuration.getAttribute(CurlUIIDs.ATTR_DEBUG_JIT_MODE, false);
        } catch (Exception e) {
            return false;
        }
    }

    private static DebugProxyData establishProxy(
            String appletID)
    {
        DebugProxyData proxy = new DebugProxyData(appletID);
        proxy.makeDebugSessionProxyData();
        return proxy;
    }
}
