/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.debug;

import java.util.Arrays;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;
import org.eclipse.ui.dialogs.SelectionStatusDialog;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.util.CurlElementLabelProvider;
import com.curl.eclipse.util.CurlUIIDs;
import com.curl.eclipse.util.StatusInfo;

public class AddBreakpointExceptionDialog extends SelectionStatusDialog
{
    private Text fExceptionNameText;
    private Button fExceptionNamesBrowserButton;

    public AddBreakpointExceptionDialog(
            Shell shell)
    {
        super(shell);
        setTitle(DebugMessages.DebugAction1);
        setShellStyle(getShellStyle() | SWT.MAX | SWT.RESIZE);
    }

    @Override
    protected Control createDialogArea(
            Composite parent)
    {
        Composite comp = (Composite)super.createDialogArea(parent);
        PlatformUI.getWorkbench().getHelpSystem().setHelp(
                comp,
                CurlUIIDs.ADD_EXCEPTION_BREAKPOINT_ACTION_DIALOG);
        createMessageArea(comp);
        GridLayout topLayout = new GridLayout();
        topLayout.verticalSpacing = 0;
        comp.setLayout(topLayout);
        comp.setFont(parent.getFont());
        createExceptionGroup(comp);
        return comp;
    }

    private void createExceptionGroup(
            Composite parent)
    {
        Font font = parent.getFont();
        Group projectGroup = new Group(parent, SWT.NONE);
        projectGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        projectGroup.setLayout(new GridLayout(2, false));
        projectGroup.setText(DebugMessages.DebugAction4);
        projectGroup.setFont(font);
        fExceptionNameText = new Text(projectGroup, SWT.SINGLE | SWT.BORDER);
        GridData gd = new GridData(GridData.FILL_HORIZONTAL);
        gd.widthHint = 280;
        fExceptionNameText.setLayoutData(gd);
        fExceptionNameText.setFont(font);
        fExceptionNameText.addModifyListener(new ModifyListener() {
            public void modifyText(
                    ModifyEvent evt)
            {
                mustProvideValidExceptionName();
            }
        });
        mustProvideValidExceptionName();
        fExceptionNamesBrowserButton = createPushButton(
                projectGroup,
                DebugMessages.DebugAction5,
                null);
        fExceptionNamesBrowserButton.addSelectionListener(new SelectionListener() {
            public void widgetDefaultSelected(
                    SelectionEvent e)
            {
            }

            public void widgetSelected(
                    SelectionEvent e)
            {
                handleBrowseButtonSelected();
            }
        });
    }

    private void mustProvideValidExceptionName()
    {
        if (validIdentifier(fExceptionNameText.getText()))
            updateStatus(new StatusInfo(IStatus.OK, "")); //$NON-NLS-1$
        else
            updateStatus(new StatusInfo(IStatus.ERROR, DebugMessages.DebugAction3));
    }

    private boolean validIdentifier(String id)
    {

        if (id.length() == 0 || !Character.isJavaIdentifierStart(id.charAt(0)) || id.charAt(0) == '$') {
            return false;
        }
        for (int i = 1; i < id.length(); i++) {
            if (!Character.isJavaIdentifierPart(id.charAt(i)) || id.charAt(0) == '$') {
                return false;
            }
        }
        return true;
    }

    private void handleBrowseButtonSelected()
    {
        String exceptionName = chooseFromExceptionNames();
        if (exceptionName == null) {
            return;
        }
        fExceptionNameText.setText(exceptionName);
    }

    private String chooseFromExceptionNames()
    {
        String exceptionNames[] = CurlPlugin.getDefault().getWorkbenchProxy().getStandardExceptionBreakpointNames();
        if (exceptionNames.length == 0)
            return null;
        ElementListSelectionDialog dialog =
            new ElementListSelectionDialog(getShell(), new CurlElementLabelProvider());
        dialog.setTitle(DebugMessages.DebugAction6);
        dialog.setMessage(DebugMessages.DebugAction2);
        dialog.setElements(exceptionNames);
        if (dialog.open() == Window.OK) {
            return (String)dialog.getFirstResult();
        }
        return null;
    }

    @Override
    protected void computeResult() {
        setResult(Arrays.asList(fExceptionNameText.getText()));
    }

    private Button createPushButton(
            Composite parent,
            String label,
            Image image)
    {
        Button button = new Button(parent, SWT.PUSH);
        button.setFont(parent.getFont());
        if (image != null) {
            button.setImage(image);
        }
        if (label != null) {
            button.setText(label);
        }
        GridData gd = new GridData();
        button.setLayoutData(gd);
        return button;
    }
}
