/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.contenttype;

import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class CurlToJavaEncodingMap extends HashMap<String, String>
{
    private static final long serialVersionUID = 1L;
    private static final Pattern fPpatternTitle = Pattern.compile("(.*)\\.title"); //$NON-NLS-1$
    private static final Pattern fPpatternAlias = Pattern.compile("(.*)\\.alias(\\d)"); //$NON-NLS-1$
    private static final Pattern fPpatternHTTPEnc = Pattern.compile("(.*)\\.http-enc"); //$NON-NLS-1$
    private final Set<String> fUnsupportedCurlEncoding = new HashSet<String>(); 
    private final Set<String> fAmbiguousCurlEncoding = new HashSet<String>(); 

    private final HashMap<String, String> fMapCurlToHTTPEnc = new HashMap<String, String>();
    
    public static final CurlToJavaEncodingMap MAP = new CurlToJavaEncodingMap();
    
    private CurlToJavaEncodingMap()
    {
        /*
         * init Curl to Java encoding name for this translation map
         * for names that aren't defined in Java
         */
        // approximation! no ISO-8859-10 available
        put("iso-latin-6", "ISO8859_1" ); //$NON-NLS-1$ //$NON-NLS-2$
        // approximation! no ISO-8859-14 available
        put("iso-latin-8", "ISO8859_1" ); //$NON-NLS-1$ //$NON-NLS-2$
        autoInitMap();
    }
    
    /**
     * Load curl encodings from properties file
     * and try to lookup java encodings names using curl names
     * to initialize this translation map.
     */
    private void autoInitMap()
    {
        /*
         * Parse properties and build tree of encoding => aliases
         */
        Map<String, List<String>> treeOfCurlEncodings = readTreeOfCurlEncodings();
        
        /*
         * search for Java encoding for each (curl encoding/aliases)
         */
        for(Map.Entry<String, List<String>> entry : treeOfCurlEncodings.entrySet()) {
            List<String> list = entry.getValue();
            String curlEncodingMainName = entry.getKey();
            list.add(curlEncodingMainName);
            Set<String> javaSet = new HashSet<String>();
            
            // primarily use curl HTTP encoding name to find java peer name
            String httpEncodingName = fMapCurlToHTTPEnc.get(curlEncodingMainName);
            Charset charset = getJavaCharSetName(httpEncodingName);
            if (charset != null) {
                javaSet.add(charset.name());
            } else {
                // Java doesn't know this http encoding, let's try with curl main name and curl aliases
                for(String curlEncoding : list) {
                    charset = getJavaCharSetName(curlEncoding);
                    if (charset != null) {
                        javaSet.add(charset.name());
                    }
                }
            }
            if (javaSet.size() > 1) {
                fAmbiguousCurlEncoding.addAll(list);
            }
            else if (javaSet.size() == 0) {
                for(String s : list) {
                    if (get(s) == null) {
                        fUnsupportedCurlEncoding.add(s);
                    }
                }
            }
            else {
                String javaEncoding = javaSet.toArray(new String[1])[0];
                for(String curlEncoding : list) {
                    put(curlEncoding, javaEncoding);
                }
            }
        }
        
        /*
         * garbage unsupported by RTE
         */
        Set<String> filter = new HashSet<String>();
        Collections.addAll(filter, "utf16-unknown-endian", "ucs2-unknown-endian", "none-specified"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
        fUnsupportedCurlEncoding.removeAll(filter);

    }

    /**
     * @return a map where each key is a curl encoding name and the corresponding
     * value is a list of curl aliases for this encoding.
     */
    private Map<String, List<String>> readTreeOfCurlEncodings()
    {
        Properties properties = new Properties();
        try {
            properties.load(CurlToJavaEncodingMap.class.getResourceAsStream("CurlEncodings.properties")); //$NON-NLS-1$
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        Map<String, List<String>> map = new HashMap<String, List<String>>();
        for(Object key : properties.keySet()) {
            String k = (String) key;
            Matcher matcherTitle = fPpatternTitle.matcher(k);
            if (matcherTitle.matches()) {
                String curlName = matcherTitle.group(1);
                if (map.get(curlName) == null) {
                    map.put(curlName, new ArrayList<String>());
                }
            } else {
                Matcher matcherHTTPEnc = fPpatternHTTPEnc.matcher(k);
                if (matcherHTTPEnc.matches()) {
                    String curlName = matcherHTTPEnc.group(1);
                    fMapCurlToHTTPEnc.put(curlName, properties.getProperty(k));
                } else {
                    Matcher matcherAlias = fPpatternAlias.matcher(k);
                    if (matcherAlias.matches()) {
                        String curlName = matcherAlias.group(1);
                        List<String> listAliases = map.get(curlName);
                        if ( listAliases == null) {
                            listAliases = new ArrayList<String>();
                            map.put(curlName, listAliases);
                        }
                        listAliases.add(properties.getProperty(k));
                    }
                }
            }
        }
        return map;
    }

    // see: http://java.sun.com/javase/6/docs/technotes/guides/intl/encoding.doc.html
    private static Charset getJavaCharSetName(
            String curlName)
    {
        try {
            if (! Charset.isSupported(curlName)) {
                return null;
            }
            Charset charset = Charset.forName(curlName);
            return charset;
        } catch (IllegalCharsetNameException e) {
            return null;
        }
    }
    
    /**
     * @return null if no error occurred, else a detailed message with reasons of warnings
     */
    public String getLoadWarningMessage()
    {
        StringBuilder sb = new StringBuilder();
        if (fUnsupportedCurlEncoding.size() > 0) {
            sb.append("Error, following curl encoding have no corresponding Java encoding:"); //$NON-NLS-1$
            for(String unsupportedCurlEnc : fUnsupportedCurlEncoding) {
                sb.append('\n').append(unsupportedCurlEnc);
            }
        }
        if (fAmbiguousCurlEncoding.size() > 0) {
            sb.append('\n').append("Error, Curl encoding name and aliases matching multiple java encoding:"); //$NON-NLS-1$
            for(String ambiguous : fAmbiguousCurlEncoding) {
                sb.append('\n').append(ambiguous);
            }
        }
        return sb.length() == 0 ? null: sb.toString();
    }
    
    /**
     * 
     * @param args
     */
    public static void main(
            String[] args)
    {
        String msg = MAP.getLoadWarningMessage();
        if (msg == null) {
            System.err.println("Curl to Java character encoding table succesfully loaded"); //$NON-NLS-1$
        } else {
            System.err.println("Error:" + msg); //$NON-NLS-1$
        }
    }
    

}
