/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.contenttype;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;

import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.core.runtime.content.IContentDescription;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.core.runtime.content.ITextContentDescriber;


/**
 * Note that this class (and the package should not depend upon the curl plug-in, see eclipse documentation,
 * see Eclipse-LazyStart in MANIFEST.MF
 * @author fmisiak
 * see org.eclipse.core.internal.content.XMLContentDescriber
 */
public class CurlContentDescriber implements ITextContentDescriber
{
    private static final QualifiedName[] SUPPORTED_OPTIONS = new QualifiedName[] {IContentDescription.CHARSET, IContentDescription.BYTE_ORDER_MARK};
    private static final ValidationResult VALIDATION_RESULT_OK = new ValidationResult();

    public CurlContentDescriber()
    {
    }
    
    public static class ValidationResult
    {
        public final boolean fSuccess;
        public final String fCurlCharSetName;

        private ValidationResult()
        {
            this(true, null);
        }

        private ValidationResult(
                boolean success,
                String curlCharsetName)
        {
            fSuccess = success;
            fCurlCharSetName = curlCharsetName;
        }
        
    }
    public static ValidationResult validCurlCharset(String curlSourceFrag) 
    {
        char[] topBlock = curlSourceFrag.toCharArray();
        String cfa = readCurlFileAttributes(topBlock, topBlock.length);
        if (cfa == null) {
            return VALIDATION_RESULT_OK;
        }
        String curlCharsetName = getCharset(cfa);
        if (CurlToJavaEncodingMap.MAP.get(curlCharsetName) != null) {
            return VALIDATION_RESULT_OK;
        } else {
            return new ValidationResult(false, curlCharsetName);
        }
    }
    
    public QualifiedName[] getSupportedOptions()
    {
        return SUPPORTED_OPTIONS;
    }
    
    public int describe(Reader input, IContentDescription description) throws IOException {
        BufferedReader reader = new BufferedReader(input);
        char[] topBlock = new char[512];
        int length = reader.read(topBlock);
        String cfa = readCurlFileAttributes(topBlock, length);
        return describe(description, cfa);
    }

    public int describe(InputStream input, IContentDescription description) throws IOException {
        byte[] bom = getByteOrderMark(input);
        String declEncoding = "UTF-8"; // "US-ASCII"; //$NON-NLS-1$
        input.reset();
        if (bom != null) {
            if (bom == IContentDescription.BOM_UTF_16BE)
                declEncoding = "UTF-16BE"; //$NON-NLS-1$ 
            else if (bom == IContentDescription.BOM_UTF_16LE)
                declEncoding = "UTF-16LE";  //$NON-NLS-1$
            // skip BOM to make comparison simpler
            input.skip(bom.length);
            // set the BOM in the description if requested
            if (description != null && description.isRequested(IContentDescription.BYTE_ORDER_MARK))
                description.setProperty(IContentDescription.BYTE_ORDER_MARK, bom);
        }
        char[] topBlock = getTopBlock(input, declEncoding);
        String cfa = readCurlFileAttributes(topBlock, topBlock.length);
        return describe(description, cfa);
    }

    private int describe(
            IContentDescription description,
            String cfa)
    {
        if (cfa == null) {
            return INDETERMINATE;
        }
        if (description == null)
            return VALID;
        
        // describe charset if requested
        if (description.isRequested(IContentDescription.CHARSET)) {
            String curlCharset = getCharset(cfa);
            if (curlCharset != null) {
                String javaCharset = curlToJavaEncodingName(curlCharset);
                if (javaCharset != null) {
                    // todo: might want to optmize a la XMLContentDescriber
                    // we'd need to agree and define in the plugin.xml the default
                    // encoding type e.g. ISO-88591-1
                    description.setProperty(IContentDescription.CHARSET, javaCharset);
                    return VALID;
                }
            }
            return INDETERMINATE;
        }
        return VALID;
    }

    private static String curlToJavaEncodingName(
            String curlCharset)
    {
        String javaCharsetName = CurlToJavaEncodingMap.MAP.get(curlCharset);
        if (javaCharsetName == null) {
            // try with Java names,
            // at least eclipse will be able to encode/decode appropriately if name
            // exists.
            // (of course RTE will fails to load and run this curl source code)
            try {
                return Charset.isSupported(curlCharset) ? curlCharset : null;
            } catch (IllegalCharsetNameException e) {
                // may happen when curl source code defines an unknown or buggy encoding
                return null;
            }
        }
        return javaCharsetName;
    }

    private static String getCharset(
            String pragma)
    {
        int indexOfCharacterEncoding = pragma.indexOf("character-encoding"); //$NON-NLS-1$
        if (indexOfCharacterEncoding == -1) {
            return null;
        }
        int indexOfDQuoteStart = pragma.indexOf('"', indexOfCharacterEncoding);
        if (indexOfDQuoteStart == -1) {
            return null;
        }
        int indexOfDQuoteEnd = pragma.indexOf('"', indexOfDQuoteStart + 1);
        if (indexOfDQuoteEnd == -1) {
            return null;
        }
        String encoding = pragma.substring(indexOfDQuoteStart + 1, indexOfDQuoteEnd);
        return encoding;
    }

    /**
     * Extracts something like <code>{curl-file-attributes character-encoding = "windows-latin-1"}</code>
     * from the begining of the passed input stream.
     * @param input
     * @param declEncoding
     * @param cbuf 
     * @param bufferLength 
     * @param offset 
     * @return the curl file attribute declaration source fragment
     * @throws IOException
     */
    private static String readCurlFileAttributes(
            char[] cbuf, int bufferLength)
    {
        
//        int lread = cbuf.length;
        int level = 0;
        StringBuilder block = null;
        boolean skipToEOL = false;
        boolean skipToEOComment = false;
        
        // Note that the following algo doesn't deal with tagged comment (aka nested comments)
        for(int i=0; i < bufferLength; i++) {
            char c = cbuf[i];
            if (skipToEOL) {
                if (c == '\n' || c == '\r') {
                    skipToEOL = false;
                }
                continue;
            }
            else if (skipToEOComment) {
                if (i + 1 == bufferLength) {
                    return null;
                }
                char nxtChar = cbuf[i + 1];
                if (c == '#' && nxtChar == '|') {
                    skipToEOComment = false;
                }
                continue;
            }
            switch(c) {
            case '|':
                if (i + 1 == bufferLength) {
                    return null;
                }
                char nxtChar = cbuf[i + 1];
                switch (nxtChar) {
                case '|':
                        // start of a mono line comment
                        skipToEOL = true; break;
                case '#':
                        // start of a multi line comment
                        skipToEOComment = true; break;
                }
                break;
            case '{':
                if (level == 0) {
                    block = new StringBuilder(128);
                }
                level++;
                if (block != null) {  // in case of syntax error in source
                    block.append(c);
                }
                break;
            case '}':
                if (block != null) {  // in case of syntax error in source
                    block.append(c);
                }
                level--;
                if (level == 0 && block != null) {
                    String candidate = block.toString();
                    if (candidate.indexOf("curl-file-attributes") >= 0) { //$NON-NLS-1$
                        return candidate;
                    }
                    else {
                        // it was a curl block, but doesn't define the attributes
                        block = null;
                    }
                }
                break;
            default:
                if (block != null) {  // in case of syntax error in source
                    block.append(c);
                }
                break;    
            }
        }
        return null;
    }

    private static char[] getTopBlock(
            InputStream input,
            String declEncoding) throws IOException
    {
        // first 512 bytes seems reasonable to find the character encoding
        byte[] buf = new byte[512];
        int lReadBytes = input.read(buf);
        if (lReadBytes <= 0) {
            // files is empty
            return new char[0];
        }
        ByteBuffer bb = ByteBuffer.allocate(512);
        bb.put(buf, 0, lReadBytes);
        
        Charset charset = Charset.forName(declEncoding);
        bb.rewind();
        CharBuffer decoded = charset.decode(bb);
        char[] cbuf = new char[decoded.length()];
        decoded.get(cbuf);
        return cbuf;
    }

    /**
     * @param input
     * @return
     * @throws IOException
     */
    byte[] getByteOrderMark(InputStream input) throws IOException {
        int first = (input.read() & 0xFF);//converts unsigned byte to int
        int second = (input.read() & 0xFF);
        if (first == -1 || second == -1)
            return null;
        //look for the UTF-16 Byte Order Mark (BOM)
        if (first == 0xFE && second == 0xFF)
            return IContentDescription.BOM_UTF_16BE;
        if (first == 0xFF && second == 0xFE)
            return IContentDescription.BOM_UTF_16LE;
        int third = (input.read() & 0xFF);
        if (third == -1)
            return null;
        //look for the UTF-8 BOM
        if (first == 0xEF && second == 0xBB && third == 0xBF)
            return IContentDescription.BOM_UTF_8;
        return null;
    }

 // Unit Tests:
//    public static void main(
//            String[] args) throws IOException
//    {
//        // {curl-file-attributes character-encoding = "windows-latin-1"}
//       String source = 
//            "|| just a line comment\n" +
//            "|| another line comment\n" +
//            "|# a multi line comment - 1 \n" +
//            " multine comment - 2\n" +
//            " multine comment - 3 #|" +
//       		"{curl-file-attributes {some-wacky-expression} || a comment \n" +
//       		"character-encoding = \"windows-latin-1\"}";
//        InputStream input = new ByteArrayInputStream(source.getBytes("UTF-16"));
//        String pragma = readPragma(input, "UTF-16");
//        String encoding = getCharset(pragma); 
//        String javaEncodingName = curlToJavaEncodingName(encoding);
//        System.out.println(javaEncodingName);
//    }
    
    static class ContentDescription implements IContentDescription
    {
        private final String fName;

        public ContentDescription(
            String absolutePath)
        {
            fName = absolutePath;
        }

        public String getCharset()
        {
            // TODO Auto-generated method stub
            return null;
        }

        public IContentType getContentType()
        {
            // TODO Auto-generated method stub
            return null;
        }

        public Object getProperty(
                QualifiedName key)
        {
            // TODO Auto-generated method stub
            return null;
        }

        public boolean isRequested(
                QualifiedName key)
        {
            return true;
        }

        public void setProperty(
                QualifiedName key,
                Object value)
        {
            // TODO Auto-generated method stub
            System.out.println(fName + ", key=" + key + ", value=" + value); //$NON-NLS-1$ //$NON-NLS-2$
        }
    }
//    
//
//    static void process(File[] files) throws IOException
//    {
//        for(File file : files) {
//            if (file.isDirectory()) {
//                process(file.listFiles());
//            }
//            else {
//                if (! file.getName().endsWith("curl")) {
//                    continue;
//                }
//                process(file);
//            }
//        }
//        
//    }
    private static void process(
            File file) throws IOException
    {
        InputStream input = new BufferedInputStream(new FileInputStream(file));
        input.mark(100);
        ContentDescription description = new ContentDescription(file.getAbsolutePath());
        describer.describe(input, description);
        input.close();
    }
//    
    private static final CurlContentDescriber describer = new CurlContentDescriber();
    public static void main(
            String[] args) throws IOException
    {
//        File dir = new File("C:/dev/jackson-ide-eclipse/ide");
//        process(dir.listFiles());
        
        process(new File("C:/dev/jackson-ide-eclipse/curl/tests/debugger/stack-trace-ja.scurl")); //$NON-NLS-1$
    }

}