/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.actions;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.resources.IFile;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IEditorRegistry;
import org.eclipse.ui.IReusableEditor;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.core.CurlElement;
import com.curl.eclipse.editors.CurlEditor;
import com.curl.eclipse.util.CoreUtil;

/**
 * @author fmisiak
 * see model org.eclipse.search.internal.ui.text.EditorOpener
 */
public class EditorOpener
{


    private IEditorReference fReusedEditor;
    
    public IEditorPart open(IFile file, boolean activate, boolean reuseEditor) throws PartInitException {
        IWorkbenchPage wbPage= getActivePage();
        if (reuseEditor)
            return showWithReuse(getEditorID(file), new FileEditorInput(file), wbPage, activate);
        return showWithoutReuse(file, wbPage, activate);
    }


    private IWorkbenchPage getActivePage()
    {
        IWorkbenchWindow window = CurlPlugin.getActiveWorkbenchWindow();
// The following comes from JDT, I have yet to see the need of this for our use cases.
//        if (window == null) {
//            final WindowRef windowRef= new WindowRef();
//            Display.getDefault().syncExec(new Runnable() {
//                public void run() {
//                    setActiveWorkbenchWindow(windowRef);
//                }
//            });
//            return windowRef.window;
//        }
        return window.getActivePage();
    }
    
    private IEditorPart showWithoutReuse(IFile file, IWorkbenchPage wbPage, boolean activate) throws PartInitException {
        return IDE.openEditor(wbPage, file, activate);
    }

    private IEditorPart showWithReuse(String editorID, IEditorInput input, IWorkbenchPage wbPage, boolean activate) throws PartInitException {
        return showInEditor(wbPage, input, editorID, activate);
    }


    private String getEditorID(IFile file) throws PartInitException {
        IEditorDescriptor desc= IDE.getEditorDescriptor(file);
        if (desc == null)
            return CurlPlugin.getDefault().getWorkbench().getEditorRegistry().findEditor(IEditorRegistry.SYSTEM_EXTERNAL_EDITOR_ID).getId();
        return desc.getId();
    }
    
    private String getEditorID(
            IFileStore element)
    {
        try {
            IEditorDescriptor desc= IDE.getEditorDescriptor(element.fetchInfo().getName());
            return desc.getId();
        } catch (PartInitException e) {
            CoreUtil.logError("Could not determine editor id for resource", e); //$NON-NLS-1$
        }
        return null;
    }

    private IEditorPart showInEditor(IWorkbenchPage page, IEditorInput input, String editorId, boolean activate) throws PartInitException {
        IEditorPart editor= page.findEditor(input);
        if (editor != null) {
            page.bringToTop(editor);
            if (activate) {
                page.activate(editor);
            }
            return editor;
        }
        IEditorReference reusedEditorRef= fReusedEditor;
        if (reusedEditorRef !=  null) {
            boolean isOpen= reusedEditorRef.getEditor(false) != null;
            boolean canBeReused= isOpen && !reusedEditorRef.isDirty() && !reusedEditorRef.isPinned();
            if (canBeReused) {
                boolean showsSameInputType= reusedEditorRef.getId().equals(editorId);
                if (!showsSameInputType) {
                    if (isOkToClose(page)) // workaround for https://bugs.eclipse.org/bugs/show_bug.cgi?id=188587
                        page.closeEditors(new IEditorReference[] { reusedEditorRef }, false);
                    fReusedEditor= null;
                } else {
                    editor= reusedEditorRef.getEditor(true);
                    if (editor instanceof IReusableEditor) {
                        ((IReusableEditor) editor).setInput(input);
                        page.bringToTop(editor);
                        if (activate) {
                            page.activate(editor);
                        }
                        return editor;
                    }
                }
            }
        }
        editor= page.openEditor(input, editorId, activate);
        if (editor instanceof IReusableEditor) {
            IEditorReference reference= (IEditorReference) page.getReference(editor);
            fReusedEditor= reference;
        } else {
            fReusedEditor= null;
        }
        return editor;
    }

    private boolean isOkToClose(IWorkbenchPage page) {
        IWorkbenchPartReference searchViewRef= page.getActivePartReference();
        return searchViewRef == null ||
            !NewSearchUI.SEARCH_VIEW_ID.equals(searchViewRef.getId()) ||
            !((IViewReference) searchViewRef).isFastView();
    }

    public void openElement(
            CurlElement curlElement,
            boolean reuseEditor) throws PartInitException
    {
        IEditorPart editor = open(curlElement.getFile(), false, reuseEditor);
        if (editor instanceof CurlEditor) {
            CurlEditor curlEditor = (CurlEditor)editor;
            curlEditor.selectAndReveal(curlElement);
        }
    }


    public void open(
            IFileStore element,
            boolean activateOnOpen,
            boolean activate)
    {
        try {
            IDE.openEditor(getActivePage(), element.toURI(), getEditorID(element), activate);
        } catch (PartInitException e) {
            CoreUtil.logError("Opening resource", e); //$NON-NLS-1$
        }
    }


    

}
