/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
package com.curl.eclipse.actions;


import org.eclipse.core.resources.IProject;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.events.MenuListener;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.IWorkbenchWindowActionDelegate;
import org.eclipse.ui.IWorkbenchWindowPulldownDelegate2;
import org.eclipse.ui.dialogs.PreferencesUtil;
import org.eclipse.ui.texteditor.ResourceAction;

import com.curl.eclipse.CurlPlugin;
import com.curl.eclipse.remote.WorkbenchWindowProxy.ProjectTargets;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.CurlProject;
import com.curl.eclipse.util.CurlUIIDs;
import com.curl.eclipse.util.WorkbenchOperations;

/**
 * Offers submenu items to represent the set of current deployment targets to choose from as well
 * as a link to the curl targets project properties to allow managing the set of targets.
 * The submenus are dynamically (re)created each time the menu is shown. The list of target is obtained
 * from the curl implementation, assuming the current curl project has been set to reflect the selection in eclipse.
 */
public class CurrentTargetAction implements IWorkbenchWindowPulldownDelegate2, IWorkbenchWindowActionDelegate 
{
    private ResourceAction fEditTargetSettingsAction;

    private Menu fMenu;

    private IWorkbenchWindow fWindow;

    private class CurrentTargetSubmenuAction extends Action
    {
        public CurrentTargetSubmenuAction(String id, String text, String description) {
            super(text, Action.AS_CHECK_BOX);
            setId(id);
            setDescription(description);
        }

        @Override
        public void run()
        {
            WorkbenchOperations workbenchOperations = CurlPlugin.getDefault().getWorkbenchProxy().getWorkbenchOperations();
            workbenchOperations.setCurrentTarget(CurlPlugin.getActiveWorkbenchWindow(), getDescription());
        }
    }
    
    public CurrentTargetAction()
    {
        initEditTargetSettingsAction();
    }
    

    public void init(
            IWorkbenchWindow window)
    {
        fWindow = window;
    }

    private void initEditTargetSettingsAction()
    {
        fEditTargetSettingsAction = 
            new ResourceAction(
                    ActionMessages.getBundleForConstructedKeys(),
                    "EditTargetSettingsAction_") { //$NON-NLS-1$
            @Override
            public void run()
            {
                IWorkbenchPart activePart = fWindow.getActivePage().getActivePart();
                IProject project = CoreUtil.getProject(activePart);
                PreferencesUtil.createPropertyDialogOn(
                        fWindow.getShell(),
                        project,
                        CurlUIIDs.ID_CURL_TARGET_PROJECT_PROPERTIES_ID,
                        null,
                        null).open();
            }
        };
        fEditTargetSettingsAction.setHelpContextId("com.curl.eclipse.plugin.CurrentTargetActionContextID"); //$NON-NLS-1$
    }
    
  @Override
  public void selectionChanged(
          IAction action,
          ISelection selection)
  {
      CurlProject curlProject = CoreUtil.getCurlProject(selection);
      action.setEnabled(curlProject != null);
  }

  @Override
  public Menu getMenu(
          Menu parent)
  {
      setMenu(new Menu(parent));
      return fMenu;
  }

  @Override
  public Menu getMenu(
          Control parent)
  {
      setMenu(new Menu(parent));
      return fMenu;
  }

  @Override
  public void dispose()
  {
      setMenu(null);
  }

  @Override
  public void run(
          IAction action)
  {
      // TODO: Is there a way to not implement this method, since it does
      // nothing?
      // I tried. But the menu item is not included unless I implement
      // IWorkbenchWindowActionDelegate
  }

  private void setMenu(
          Menu menu)
  {
      if (fMenu != null)
          fMenu.dispose();
      fMenu = menu;
      // TODO: could optimize since the targets are now cached.
      if (fMenu != null) {
          fMenu.addMenuListener(new MenuListener() {
              @Override
              public void menuHidden(
                      org.eclipse.swt.events.MenuEvent e)
              {
                  // do nothing
              }

              @Override
              public void menuShown(
                      org.eclipse.swt.events.MenuEvent e)
              {
                  createSubmenu();
              }
          });
      }
  }
    
    private void createSubmenu()
    {
        disposeMenuItems();
        WorkbenchOperations workbenchOperations = CurlPlugin.getDefault().getWorkbenchProxy().getWorkbenchOperations();
        ProjectTargets targets = workbenchOperations.getProjectTargets(fWindow);
        String currentTargetName = null;
        String[] names;
        if (targets != null) {
            currentTargetName = targets.getCurrent();
            names = targets.getNames();
        } else {
            names = new String[0];
        }
        Action[] currentTargetSubmenuActions = new Action[names.length];
        for (int i = 0; i < names.length; i++) {
            String targetName = names[i];
            // setup label to define mnemonic
            String label = "&" + (i + 1) + " " + targetName; //$NON-NLS-1$  //$NON-NLS-2$
            CurrentTargetSubmenuAction action = new CurrentTargetSubmenuAction(
                    "com.curl.eclipse.currenttarget.submenu." + String.valueOf(i), //$NON-NLS-1$
                    label,
                    targetName); // action description is used as the "value"
            action.setChecked(currentTargetName != null && targetName.equals(currentTargetName));
            currentTargetSubmenuActions[i] = action;
        }
        fillMenu(currentTargetSubmenuActions);
    }
    
    private void fillMenu(Action[] currentTargetSubmenuActions)
    {
      for (Action element : currentTargetSubmenuActions) {
          ActionContributionItem item = new ActionContributionItem(element);
          item.fill(fMenu, -1);
      }
      Separator separator = new Separator("editTargetSettingsGroup"); //$NON-NLS-1$
      separator.fill(fMenu, -1);
      ActionContributionItem editTargetSettingsContributionItem = new ActionContributionItem(
              fEditTargetSettingsAction);
      editTargetSettingsContributionItem.fill(fMenu, -1);
    }

    private void disposeMenuItems()
    {
      if (fMenu == null || fMenu.isDisposed()) {
          return;
      }
      MenuItem[] items = fMenu.getItems();
      for (MenuItem menuItem : items) {
          if (!menuItem.isDisposed()) {
              menuItem.dispose();
          }
      }
    }

}
