/* Copyright (c) 2008 Sumisho Computer Systems Corp. All rights reserved. This
 * program and the accompanying materials are made available under the terms of the
 * Eclipse Public License v1.0 which accompanies this distribution, and is
 * available at http://www.eclipse.org/legal/epl-v10.html
 * Contributors - Curl, Inc. This plugin includes codes from Eclipse code */
/*******************************************************************************
 * Copyright (c) 2000, 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.curl.eclipse;

import java.io.File;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.URL;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleListener;

import com.curl.eclipse.contenttype.CurlToJavaEncodingMap;
import com.curl.eclipse.debug.ManageBreakpoints;
import com.curl.eclipse.files.CurlFileDocumentProvider;
import com.curl.eclipse.remote.MediatorConnection;
import com.curl.eclipse.remote.WorkbenchProxy;
import com.curl.eclipse.ui.ImageDescriptorRegistry;
import com.curl.eclipse.util.CoreUtil;
import com.curl.eclipse.util.MediatorEventManager;
import com.curl.eclipse.util.WorkbenchOperations;

/**
 * The Curl plugin class.
 *
 * This class is instantiated by the IDE, and used by the plugin to get various
 * information
 */
public class CurlPlugin extends AbstractUIPlugin
{
    private BundleListener fBundleListener;
    private static CurlPlugin fgInstance;
    private MediatorEventManager fMediatorEventManager;
    private CurlFileDocumentProvider fCurlFileDocumentProvider;
    private WorkbenchOperations fWorkbenchOperations;

    private boolean fStoppedOrStopping;
    private SafeCompositeManager fSafeCompositeManger;
    private MediatorConnection fMediatorConnection;
    private ManageBreakpoints fManageBreakpoints;
    private ImageDescriptorRegistry fImageDescriptorRegistry;

    public WorkbenchProxy getWorkbenchProxy()
    {
        return fWorkbenchOperations.getWorkbenchProxy();
    }

    /**
     * Creates a new plug-in instance.
     */
    public CurlPlugin()
    {
        super();
        fgInstance = this;
    }

    @Override
    public void start(
            BundleContext context) throws Exception
    {
        super.start(context);
        
        // force initialization of pool of safe Composite now,
        // the idea is to allocate them relatively early before they are needed.
        getSafeCompositeManager();
        
        // Startup is a good opportunity to log the character encoding support
        // problems if any. Note that this map will  be used before this plugin is even
        // started.
        String loadWarningMessage = CurlToJavaEncodingMap.MAP.getLoadWarningMessage();
        if (loadWarningMessage != null) {
            CoreUtil.logWarning(loadWarningMessage);
        }
        
        fMediatorConnection = new MediatorConnection();        
        fMediatorConnection.versionsMatch();
        fWorkbenchOperations = new WorkbenchOperations();
        fMediatorEventManager = new MediatorEventManager();
        fManageBreakpoints = new ManageBreakpoints(getWorkbenchProxy());
    }

    @Override
    public void stop(
            BundleContext context) throws Exception
    {
        fStoppedOrStopping = true;

        if (fBundleListener != null) {
            context.removeBundleListener(fBundleListener);
        }
        try {
            if (fCurlFileDocumentProvider != null) {
                fCurlFileDocumentProvider.shutdown();
                fCurlFileDocumentProvider = null;
            }
            fMediatorEventManager.dispose();
            fMediatorEventManager = null;
            if (fWorkbenchOperations != null) {
                fWorkbenchOperations.shutdown();
                fWorkbenchOperations = null;
            }
            if (fManageBreakpoints != null) {
                fManageBreakpoints.shutdown();
                fManageBreakpoints = null;
            }
            fMediatorConnection.closeRemoteConnections();
            if (fImageDescriptorRegistry != null) {
                fImageDescriptorRegistry.dispose();
                fImageDescriptorRegistry = null;
            }
        } finally {
            fMediatorConnection = null;
            super.stop(context);
        }
    }

    public boolean isStoppedOrStopping()
    {
        return fStoppedOrStopping;
    }

    public static CurlFileDocumentProvider getCurlFileDocumentProvider()
    {
        if (CurlPlugin.getDefault().fCurlFileDocumentProvider == null) {
            CurlPlugin.getDefault().fCurlFileDocumentProvider = new CurlFileDocumentProvider();
        }
        return CurlPlugin.getDefault().fCurlFileDocumentProvider;
    }

    public static MediatorEventManager getMediatorEventManager()
    {
        return getDefault().fMediatorEventManager;
    }

    public static IWorkbenchPage getActivePage()
    {
        return getDefault().internalGetActivePage();
    }

    public static IWorkbenchWindow getActiveWorkbenchWindow()
    {
        return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
    }

    public static String getPluginId()
    {
        return "com.curl.eclipse.plugin"; //$NON-NLS-1$
    }

    /**
     * Returns the default plug-in instance.
     */
    public static CurlPlugin getDefault()
    {
        return fgInstance;
    }

    /**
     * Returns a free port number on localhost, or -1 if unable to find a free
     * port.
     *
     * @return a free port number on localhost, or -1 if unable to find a free
     *         port
     */
    static public int findFreePort()
    {
        ServerSocket socket = null;
        try {
            socket = new ServerSocket(0);
            return socket.getLocalPort();
        } catch (IOException e) {
        } finally {
            if (socket != null) {
                try {
                    socket.close();
                } catch (IOException e) {
                }
            }
        }
        return -1;
    }

    public File getPluginFolder()
    {
        URL url = getBundle().getEntry("/"); //$NON-NLS-1$
        try {
            url = FileLocator.toFileURL(url);
        } catch (IOException e) {
            CoreUtil.logError("Getting plugin folder failed", e); //$NON-NLS-1$
            return null;
        }
        return new File(url.getPath());
    }

    public File getCurlPropertyFile()
    {
        return new File(getPluginFolder(), "CurlPropertyFile.dat"); //$NON-NLS-1$
    }

    private IWorkbenchPage internalGetActivePage()
    {
        IWorkbenchWindow window = getWorkbench().getActiveWorkbenchWindow();
        if (window == null)
            return null;
        return getWorkbench().getActiveWorkbenchWindow().getActivePage();
    }

    public IDialogSettings getDialogSettingsSection(String name) {
        IDialogSettings dialogSettings= getDialogSettings();
        IDialogSettings section= dialogSettings.getSection(name);
        if (section == null) {
            section= dialogSettings.addNewSection(name);
        }
        return section;
    }

    public SafeCompositeManager getSafeCompositeManager()
    {
        if (fSafeCompositeManger == null) {
            Display current = getWorkbench().getDisplay();
            fSafeCompositeManger = new SafeCompositeManager(current);
        }
        return fSafeCompositeManger;
    }
    
    public MediatorConnection getMediatorConnection()
    {
        if (fMediatorConnection == null) {
            CoreUtil.logError("Connection to Curl runtime must already have been established!"); //$NON-NLS-1$
            return null;
        }
        
        return fMediatorConnection;
    }

    public WorkbenchOperations getWorkbenchOperations()
    {
        return getWorkbenchProxy().getWorkbenchOperations();
    }
    
    public ManageBreakpoints getManageBreakpoints()
    {
        return getDefault().fManageBreakpoints;
    }

    public static ImageDescriptorRegistry getImageDescriptorRegistry()
    {
        CurlPlugin plugin = getDefault();
        if (plugin.fImageDescriptorRegistry == null) {
            plugin.fImageDescriptorRegistry = new ImageDescriptorRegistry();
        }
        return plugin.fImageDescriptorRegistry;
    }
}
